import { RouterState } from 'connected-react-router';
import produce from 'immer';
import * as React from 'react';

import { IRolesResponse } from '../../../../types';
import { RoleItem, RoleProposition } from '../../../models/role';
import { BasicUserInfo } from '../../../models/user';
import { Button, ButtonTypes } from '../../../ui/Button';
import { Confirm } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { TabItem, Tabs } from '../../../ui/Tabs';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import EditRole from '../EditRole';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import * as style from './index.css';
import { VirtualRoles } from './VirtualTr';

type IProps = RouterState

export interface IPropositions {
    [key: string]: RoleProposition[];
}

interface IState {
    isLoading: boolean;
    error: any;
    propositions: IPropositions;
    report: RoleItem[];
    users: BasicUserInfo;
    tabs: TabItem[];
    selectedTab: string;

    _filterText: string;
    filterText: string;
    editRoleIsOpen: boolean;
    editRoleId: string;
    propositionId: string;

    deletedRole: string | null;
    isWorking: boolean;
    deletingError: Error | null;
}

const PROPOSITIONS_TAB = '&propositions';

export class Roles extends React.Component<IProps, IState> {
    state = {
        isLoading: false,
        error: null,
        propositions: {} as IPropositions,
        report: [] as RoleItem[],
        users: {} as BasicUserInfo,
        tabs: [] as TabItem[],
        selectedTab: '',
        _filterText: '',
        filterText: '',
        editRoleIsOpen: false,
        editRoleId: '',
        propositionId: '',
        deletedRole: null,
        isWorking: false,
        deletingError: null,
    };

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    getData(props) {
        this.setState({
            isLoading: true,
            error: null,
        }, () => {
            this.request.exec<IRolesResponse>(REQUESTS.GET_ROLES, { queryParams: { report: 'compact' } })
                .then(response => {
                    this.setState(produce(this.state, draft => {
                        draft.isLoading = false;
                        draft.report = response.report || [];
                        draft.users = response.users || {};

                        draft.propositions = response.propositions
                            && Array.isArray(response.propositions)
                            && response.propositions.reduce((_p, _c) => {
                                if (!_p.hasOwnProperty(_c.role_id)) {
                                    _p[_c.role_id] = [_c];
                                } else {
                                    _p[_c.role_id].push(_c);
                                }

                                return _p;
                            }, {}) || {};

                        let tabs: string[] = [];
                        response.report
                        && Array.isArray(response.report)
                        && response.report.forEach(tag => {
                            const temp = tag.role_groupping_tags && tag.role_groupping_tags.split(',') || [];
                            tabs = [...tabs, ...temp.map(i => i.trim())];
                        });
                        draft.tabs = [...new Set(tabs)]
                            .sort((a: string, b: string) => a.localeCompare(b))
                            .map(i => ({ name: i, link: i }));
                        if (!tabs.includes(this.state.selectedTab) && this.state.selectedTab !== PROPOSITIONS_TAB) {
                            draft.selectedTab = '';
                        }
                    }));
                })
                .catch(error => {
                    this.setState({
                        isLoading: false,
                        error,
                    });
                });
        });
    }

    componentDidMount(): void {
        this.getData(this.props);
        this.getId();
    }

    selectTab(selectedTab) {
        this.setState({
            selectedTab,
        });
    }

    setFilter(value) {
        this.setState({
            _filterText: value,
            filterText: value,
        });
    }

    filterByText(role) {
        const filterTextLowerCase = this.state.filterText.toLowerCase();

        return !this.state.filterText
            ? true
            : role.role_id.toLowerCase().includes(filterTextLowerCase)
            || role.role_description.toLowerCase().includes(filterTextLowerCase)
            || role.slave_role_ids?.some(slave_role_id => slave_role_id?.toLowerCase()?.includes(filterTextLowerCase))
            || role.action_ids?.some(action_id => action_id?.toLowerCase()?.includes(filterTextLowerCase));
    }

    componentDidUpdate(prevProps: Readonly<IProps>, prevState: Readonly<IState>, snapshot?: any): void {
        const location = this.props.location;
        const prevLocation = prevProps.location;

        if (location && prevLocation && location.search !== prevLocation.search) {
            this.getId();
        }
    }

    getId(newRole = false) {
        const params = new URLSearchParams(this.props && this.props.location && this.props.location.search || '');
        const newRoleId = params.get('role_id');
        const propositionId = params.get('proposition_id');
        this.setState({
            editRoleIsOpen: !!newRoleId || newRole,
            editRoleId: newRoleId || '',
            propositionId: propositionId || '',
        });
    }

    closeEditRole() {
        this.setState({
            editRoleIsOpen: false,
        }, () => {
            location.href = '#/settings/roles';
        });
    }

    showConfirm(deletedRole) {
        this.setState({
            deletedRole,
            deletingError: null,
        });
    }

    removeRole() {
        this.setState({
            isWorking: true,
            deletingError: null,
        }, () => {
            this.request.exec(REQUESTS.DELETE_ROLE, {
                queryParams: {
                    roles: this.state.deletedRole,
                },
            })
                .then(() => {
                    this.setState({
                        isWorking: false,
                        deletingError: null,
                        deletedRole: null,
                    }, () => {
                        this.getData(this.props);
                    });
                })
                .catch(deletingError => {
                    this.setState({
                        isWorking: false,
                        deletingError,
                    });
                });
        });
    }

    render() {
        let report: any = this.state.report
            .filter(role => {
                if (!this.state.selectedTab) {
                    return this.filterByText(role);
                }

                return role.role_groupping_tags.includes(this.state.selectedTab)
                        && this.filterByText(role);

            })
            .sort((a, b) => {
                return a.role_group.localeCompare(b.role_group) || a.role_id.localeCompare(b.role_id);
            });

        if (this.state.selectedTab === PROPOSITIONS_TAB) {
            report = Object.keys(this.state.propositions)?.map(key => {
                return {
                    role_id: key,
                };
            });
        }

        return <div className={style.roles}>
            <Input value={this.state._filterText}
                   onChange={this.setFilter.bind(this)}
                   disabled={this.state.isLoading}
                   className={style.filter}
                   placeholder={'Фильтр: название, описание'}/>
            <Button colorType={ButtonTypes.positive} onClick={this.getId.bind(this, true)}>Добавить роль</Button>
            <Tabs currentTab={this.state.selectedTab}
                  selectTab={this.selectTab.bind(this)}
                  tabs={[
                      { name: 'All', link: '' },
                      ...this.state.tabs,
                      { name: 'Предложения', link: PROPOSITIONS_TAB },
                  ]}/>

            {
                this.state.editRoleIsOpen &&
                <EditRole role_id={this.state.editRoleId}
                          proposition_id={this.state.propositionId}
                          update={this.getData.bind(this, this.props)}
                          onCLose={this.closeEditRole.bind(this)}/>
            }

            {this.state.isLoading
                ? <Spin/>
                : this.state.error
                    ? <SimpleError error={this.state.error}/>
                    : <div className={style.roles_content}>
                        <VirtualRoles report={report}
                                      propositions={this.state.propositions}
                                      removeRole={this.showConfirm.bind(this)}/>
                    </div>
            }
            {
                this.state.deletedRole ? <Confirm error={this.state.deletingError}
                                                  onClose={this.showConfirm.bind(this, null)}
                                                  question={`Удалить роль ${this.state.deletedRole}?`}
                                                  isWorking={this.state.isWorking}
                                                  accept={this.removeRole.bind(this)}/> : null
            }
        </div>;
    }
}
