import * as React from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { EMPTY_DATA } from '../../../../constants';
import { UserInfoHandler } from '../../../../models/user';
import { Button, ButtonTypes } from '../../../../ui/Button';
import Checkbox from '../../../../ui/Checkbox';
import { ErrorsModal } from '../../../../ui/ErrorBar';
import { Link } from '../../../../ui/Link';
import { NoInformation } from '../../../../ui/NoInformation';
import TextArea from '../../../../ui/TextArea';
import { Request2 } from '../../../../utils/request';
import { truncateString } from '../../../../utils/utils';
import { AddManageRole } from '../../../Clients/UserRolesView/AddManageRole';
import { buildOptionsForAddUser } from '../../../Clients/UserRolesView/component';
import QueryScheduler, { COUNT_FIRST, IProgressData } from '../../../QueryScheduler/QueryScheduler';
import { REQUESTS, SETTINGS_REQUESTS } from '../../request';
import * as styles from './index.css';

interface IRolesUsersArbitraryListProps {
}

interface IRolesUsersArbitraryListState {
    errors: Error[];
    isLoading: boolean;
    data: string;
    windowSize: { width: number; height: number };
    searchTexts: any;
    users: any;
    usersState: any;
    checkState: boolean;
    rowHeights: number[];
    addManageRoleIsOpen: boolean;
    enableRoles: boolean;
    isWorking: boolean;
    confirmError: Error | null;
}

const HEIGHT_PADDING = 400;
const ITEM_SIZE = 30;

export class RolesUsersArbitraryList extends
    React.Component<IRolesUsersArbitraryListProps, IRolesUsersArbitraryListState> {
    state: IRolesUsersArbitraryListState = {
        errors: [],
        isLoading: false,
        data: '',
        searchTexts: '',
        windowSize: { width: window.innerWidth, height: window.innerHeight },
        users: [],
        usersState: [],
        checkState: false,
        rowHeights: [],
        addManageRoleIsOpen: false,
        enableRoles: false,
        isWorking: false,
        confirmError: null,
    };

    rowIndex = 0;

    onResize = () => {
        this.setState({
            windowSize: { width: window.innerWidth, height: window.innerHeight },
        });
    };

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    onChange(data) {
        this.setState({
            data,
        });
    }

    onCloseModal() {
        this.setState({
            addManageRoleIsOpen: false,
            isWorking: false,
            enableRoles: false,
        });
    }

    getAccount(searchTextEntry: [string, any]) {
        const [searchText, value] = searchTextEntry;

        return this.request.exec(REQUESTS.SEARCH, {
            queryParams: {
                has_all_of: searchText,
                what: 'users',
            },
        })
            .then((response) => {
                let users = response?.objects?.users || [];
                users = users.map((user: any) => {
                    user.rowIndex = this.rowIndex;
                    this.rowIndex++;

                    return user;
                });

                value.users = users;

                return { [searchText]: value };
            });
    }

    search() {
        const LOGIN_EMAIL_REGEX = /(?=.*[a-zA-Z])[\w\d.\-@]+/ig;
        const PHONE_NUMBER_REGEX = /(?!.*[a-zA-Z])\b\d{5,}/ig;

        const loginEmailArray = this.state.data.match(LOGIN_EMAIL_REGEX);
        const phoneArray = this.state.data.match(PHONE_NUMBER_REGEX);

        const phoneNumbers = phoneArray?.reduce((result: { [key: string]: any }, phoneNumber: string) => {
            result[phoneNumber] = { displayText: phoneNumber };

            return result;
        }, {});

        const loginsEmails = loginEmailArray?.reduce((result: { [key: string]: any }, login: string) => {
            result[login] = { displayText: login };

            return result;
        }, {});

        const searchTexts = Object.assign({}, phoneNumbers, loginsEmails);
        const searchTextsLength = Object.keys(searchTexts)?.length;

        if (searchTextsLength) {
            this.setState({
                searchTexts,
                isLoading: true,
            }, () => {
                this.rowIndex = 0;
                const queue = Object.entries(searchTexts)
                    .map((searchTextEntry) => this.getAccount.bind(this, searchTextEntry));

                const qs = new QueryScheduler({
                    queue,
                    limit: COUNT_FIRST,
                    onProgress: this.onGetUsersProgress.bind(this),
                    onSuccess: this.onGetUsersSuccess.bind(this),
                });

                qs.run();
            });
        } else {
            const { errors } = this.state;
            errors.push(new Error('Ваш запрос не содержит корректные логины, номера телефонов или email'));

            this.setState({
                errors,
            });
        }
    }

    onGetUsersProgress() {}

    onGetUsersSuccess(data: IProgressData) {
        const { success, failed } = data;
        const usersState = new Array(this.rowIndex).fill(false) ?? [];

        if (failed.length) {
            this.setState({
                errors: failed.map((error) => error.error),
            });
        }

        this.setState({
            isLoading: false,
            users: success,
            usersState,
        });
    }

    checkboxHandler(value) {
        this.setState({
            checkState: value,
            usersState: this.state.usersState.map(() => value),
        });
    }

    userCheckboxHandler(index, value) {
        const usersState = [...this.state.usersState];
        usersState[index] = value;
        this.setState({ usersState });
    }

    setRowHeights(height, index) {
        const newRowHeights: number[] = this.state.rowHeights;
        newRowHeights[index] = height;
        this.setState({
            rowHeights: newRowHeights,
        });
    }

    addOneRole(stateModal) {
        const { users, usersState } = this.state;

        const rawUsers = users.reduce((acc, userGroup) => {
            const values: any = Object.values(userGroup)[0];
            acc.push(...values?.users);

            return acc;
        }, []);

        const filteredUsers = rawUsers.filter((user: any, index: number) => {
            return usersState[index];
        });

        this.setState({
            isWorking: true,
        }, () => {
            Promise.all(filteredUsers.map((item: any) => {
                return this.request.exec(REQUESTS.ADD_MANAGE_ROLES,
                    buildOptionsForAddUser({
                        user_id: item.id,
                        ...stateModal,
                    }),
                );
            }))
                .then(() => {
                    this.onCloseModal();
                })
                .catch((confirmError) => {
                    this.setState({
                        isWorking: false,
                        confirmError,
                    });
                });
        });
    }

    addRoles() {
        this.setState({
            addManageRoleIsOpen: true,
            confirmError: null,
        });
    }

    closeErrorsModal() {
        this.setState({
            errors: [],
        });
    }

    componentDidMount() {
        location.hash = '#/roles-users/arbitrary-list';
        window.addEventListener('resize', this.onResize);
    }

    componentWillUnmount() {
        window.removeEventListener('resize', this.onResize);
    }

    render() {
        const {
            errors,
            data,
            users,
            checkState,
            usersState,
            rowHeights,
            windowSize,
            addManageRoleIsOpen,
            isWorking,
            confirmError,
            isLoading,
        } = this.state;

        return <div>
            <TextArea placeholder={'Введите список телефонов, email или логинов для поиска'}
                      value={data}
                      onChange={this.onChange.bind(this)}/>
            <div className={styles.buttonContainer}>
                <Button isLoading={isLoading}
                        className={styles.button}
                        disabled={!data?.length}
                        onClick={this.search.bind(this)}>Найти</Button>
                <Button className={styles.button}
                        colorType={ButtonTypes.positive}
                        disabled={!usersState.some((item: boolean) => item)}
                        onClick={this.addRoles.bind(this, false)}>Добавить ещё роль</Button>
            </div>
            <div className={styles.content}>
                {users?.length
                    ? <>
                        <div className={`${styles.header} ${styles.row}`}>
                            <span>#</span>
                            <span>
                                <Checkbox checked={checkState}
                                          onChange={this.checkboxHandler.bind(this)}/>
                            </span>
                            <span>Username</span>
                            <span>Статус</span>
                            <span>ФИО</span>
                        </div>
                        <VirtualList width={'100%'}
                                     itemCount={users.length}
                                     height={windowSize.height - HEIGHT_PADDING}
                                     key={data}
                                     itemSize={(index) => {
                                         return rowHeights[index] ?? ITEM_SIZE;
                                     }}
                                     renderItem={({ index, style }) => {
                                         const usersData = this.state.users;
                                         const successItemObj: any = Object.values(usersData[index])[0];
                                         const { displayText, users } = successItemObj;

                                         return <ArbitraryListItem displayText={displayText}
                                                                   users={users}
                                                                   usersState={usersState}
                                                                   userCheckboxHandler={this.userCheckboxHandler
                                                                       .bind(this)}
                                                                   style={style}
                                                                   setHeight={this.setRowHeights.bind(this)}
                                                                   index={index}
                                                                   key={`list-item-${index}`}/>;
                                     }}/>
                    </>
                    : <NoInformation/>
                }
            </div>
            {addManageRoleIsOpen && (
                <AddManageRole onClose={this.onCloseModal.bind(this)}
                               isWorking={isWorking}
                               action={this.addOneRole.bind(this)}
                               currentRole={{ role_id: '', deadline: '', active: '' }}
                               error={confirmError}/>
            )}
            {errors.length
                ? <ErrorsModal errors={errors}
                               title={'Ошибка поиска пользователей'}
                               onClose={this.closeErrorsModal.bind(this)}/>
                : null
            }
        </div>;
    }
}

interface IArbitraryListItem {
    displayText: string;
    users: any;
    usersState: boolean[];
    userCheckboxHandler: () => void;
    style: any;
    setHeight: (height: number, index: number) => void;
    index: number;
}

const ArbitraryListItem = (props: IArbitraryListItem) => {
    const { displayText, users, usersState, userCheckboxHandler, style, setHeight, index } = props;
    const rows = [
        <div className={styles.textRow} key={displayText}>
            <p>{displayText}</p>
        </div>,
    ];
    const MAX_LENGTH_STRING = 45;

    React.useEffect(() => {
        setHeight(ITEM_SIZE + users.length * ITEM_SIZE, index);
    }, [users]);

    users.forEach((user: any, userIndex: number) => {
        const userPrintName = UserInfoHandler.getPrintNameWithoutUserName.call(user);

        const row = <div className={styles.row} key={userIndex}>
            <span>{user.rowIndex + 1}</span>
            <span>
                <Checkbox checked={usersState[user.rowIndex]}
                          onChange={userCheckboxHandler.bind(null, user.rowIndex)}/>
            </span>
            <span>
                {user.username ? (
                    <Link href={`#/clients/${user.id}/user-access`}>
                        {user.username}
                    </Link>
                ) : EMPTY_DATA}
            </span>
            <span>{user.status || EMPTY_DATA}</span>
            <span title={userPrintName}>{truncateString(userPrintName, MAX_LENGTH_STRING)}</span>
        </div>;

        rows.push(row);
    });

    return <div key={index}
                style={style}>
        {rows}
    </div>;
};
