import * as React from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { Dict } from '../../../../types';
import { EMPTY_DATA } from '../../../constants';
import { UserInfoHandler } from '../../../models/user';
import { Button, ButtonTypes } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { ErrorsModal } from '../../../ui/ErrorBar';
import { Confirm } from '../../../ui/FullModal';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import Select, { IOptionInfo } from '../../../ui/Select';
import { Request2 } from '../../../utils/request';
import { AddManageRole } from '../../Clients/UserRolesView/AddManageRole';
import { buildOptionsForAddUser } from '../../Clients/UserRolesView/component';
import { IBlockRules } from '../../Clients/UserRolesView/types';
import QueryScheduler, { COUNT_FIRST, IProgressData } from '../../QueryScheduler/QueryScheduler';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import * as styles from './index.css';
import { SaveUserButton } from './SaveUsersButton/SaveUserButton';

interface IRolesUsersProps extends IBlockRules {
    location: any;
}

interface IRolesUsersState {
    error: Error | null;
    isLoading: boolean;
    pagination: any;
    users: any;
    usersAll: any[];
    usersState: any;
    checkState: boolean;
    windowSize: Dict<number>;
    addManageRoleIsOpen: boolean;
    withRemove: boolean;
    enableRoles: boolean;
    isWorking: boolean;
    question: string;
    accept: () => void;
    confirmRemoveIsOpen: boolean;
    currentRole: any;
    confirmError: Error | null;
    rolesOptions: IOptionInfo[];
    rolesValue: string;
    selectError: Error | null;
    selectLoading: boolean;
    areAllUsersLoading: boolean;
    addEverybodyErrors: Error[];
}

const USERS_PER_PAGE = 50;
const HEIGHT_PADDING = 287;
const ITEM_SIZE = 30;

export class RolesUsers extends React.Component<IRolesUsersProps, IRolesUsersState> {
    state: IRolesUsersState = {
        error: null,
        isLoading: false,
        pagination: { current_page: 0 },
        usersAll: [],
        users: [],
        usersState: [],
        checkState: false,
        windowSize: { width: window.innerWidth, height: window.innerHeight },
        addManageRoleIsOpen: false,
        withRemove: false,
        enableRoles: false,
        isWorking: false,
        question: '',
        accept: () => undefined,
        confirmRemoveIsOpen: false,
        currentRole: { role_id: '', deadline: '', active: '' },
        confirmError: null,
        rolesOptions: [],
        rolesValue: '',
        selectError: null,
        selectLoading: false,
        areAllUsersLoading: false,
        addEverybodyErrors: [],
    };

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    onResize = () => {
        this.setState({
            windowSize: { width: window.innerWidth, height: window.innerHeight },
        });
    };

    getData(page) {
        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_USERS_BY_ROLE, {
                queryParams: {
                    roles: this.state.rolesValue,
                    page,
                },
            })
                .then((response) => {
                    this.setState({
                        isLoading: false,
                        pagination: response?.pagination,
                        users: response?.report?.sort((a: any, b: any) => a?.username.localeCompare(b?.username)) || [],
                        usersState: response?.report && new Array(response?.report?.length || 0).fill(false) || [],
                    });
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    getRoles() {
        this.setState({
            selectLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_ROLES, { queryParams: { report: 'compact' } })
                .then((response) => {
                    const rolesOptions = response?.report?.map((role) => ({
                        text: role.role_description,
                        value: role.role_id,
                    }));

                    this.setState({
                        rolesOptions,
                        selectLoading: false,
                    });
                })
                .catch((selectError) => {
                    this.setState({
                        selectError,
                        selectLoading: false,
                    });
                });
        });
    }

    checkboxHandler(value) {
        this.setState({
            checkState: value,
            usersState: this.state.usersState.map(() => value),
        });
    }

    userCheckboxHandler(index, value) {
        const usersState = [...this.state.usersState];
        usersState[index] = value;
        this.setState({ usersState });
    }

    activateRole() {
        const { rolesValue } = this.state;
        const filteredUsers = this.state.users
            .filter((user: any, index: number) => {
                return this.state.usersState[index];
            }).map((user: any) => user.id);

        this.setState({
            confirmError: null,
            enableRoles: true,
            question: `Активировать роль ${filteredUsers.length}чел?`,
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    Promise.all(filteredUsers.map((user_id: string) => {
                        return this.request.exec(REQUESTS.ACTIVATE_ROLE_USERS, {
                            queryParams: {
                                role_id: rolesValue,
                                user_id,
                            },
                        });
                    }))
                        .then(() => {
                            this.onCloseModal();
                        })
                        .catch((confirmError) => {
                            this.setState({
                                confirmError,
                                isWorking: false,
                            });
                        });
                });
            },
        });
    }

    deactivateRole() {
        const { rolesValue } = this.state;
        const filteredUsers = this.state.users
            .filter((user: any, index: number) => {
                return this.state.usersState[index];
            }).map((user: any) => user.id);

        this.setState({
            confirmError: null,
            enableRoles: true,
            question: `Деактивировать роль ${filteredUsers.length}чел?`,
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    Promise.all(filteredUsers.map((user_id: string) => {
                        return this.request.exec(REQUESTS.DEACTIVATE_ROLE_USERS, {
                            queryParams: {
                                role_id: rolesValue,
                                user_id,
                            },
                        });
                    }))
                        .then(() => {
                            this.onCloseModal();
                        })
                        .catch((confirmError) => {
                            this.setState({
                                confirmError,
                                isWorking: false,
                            });
                        });
                });
            },
        });
    }

    addRoles(withRemove) {
        this.setState({
            addManageRoleIsOpen: true,
            withRemove,
            confirmError: null,
        });
    }

    removeRoles() {
        this.setState({
            confirmRemoveIsOpen: true,
            confirmError: null,
        });
    }

    onCloseModal() {
        this.setState({
            confirmRemoveIsOpen: false,
            addManageRoleIsOpen: false,
            isWorking: false,
            withRemove: false,
            enableRoles: false,
        });
    }

    accept() {
        const filteredUsers = this.state.users.filter((user: any, index: number) => {
            return this.state.usersState[index];
        });

        this.setState({
            isWorking: true,
        }, () => {
            Promise.all(filteredUsers.map((item: any) => {
                return this.request.exec(REQUESTS.DELETE_MANAGE_ROLES, {
                    queryParams: {
                        user_id: item.id,
                        roles: this.state.rolesValue,
                    },
                });
            }))
                .then(() => {
                    this.onCloseModal();
                    this.getData(0);
                })
                .catch((confirmError) => {
                    this.setState({
                        isWorking: false,
                        confirmError,
                    });
                });

        });
    }

    addOneRole(stateModal) {
        let { users, usersState, usersAll } = this.state;

        if (usersAll.length) {
            users = usersAll;
            usersState = Array(usersAll.length).fill(true);
        }

        const filteredUsers = users.filter((user: any, index: number) => {
            return usersState[index];
        });

        this.setState({
            isWorking: true,
        }, () => {
            Promise.all(filteredUsers.map((item: any) => {
                return this.request.exec(REQUESTS.ADD_MANAGE_ROLES, buildOptionsForAddUser({
                    user_id: item.id,
                    ...stateModal,
                }));
            }))
                .then(() => {
                    if (this.state.withRemove) {
                        Promise.all(filteredUsers.map((item: any) => {
                            return this.request.exec(REQUESTS.DELETE_MANAGE_ROLES, {
                                queryParams: {
                                    user_id: item.id,
                                    roles: this.state.rolesValue,
                                },
                            });
                        }))
                            .then(() => {
                                this.onCloseModal();
                                this.getData(0);
                            });
                    } else {
                        this.onCloseModal();
                    }
                })
                .catch((confirmError) => {
                    this.setState({
                        isWorking: false,
                        confirmError,
                    });
                });
        });
    }

    changePage(page) {
        this.getData(page);
    }

    onSelect(value) {
        if (value) {
            location.href = `#/roles-users/main?roleId=${value}`;
        }
    }

    addEverybodyRoles() {
        const { pagination } = this.state;
        const queue: any = [];

        for (let page = 0; page < pagination.total_pages; page++) {
            queue.push(
                this.request.exec.bind(this.request, REQUESTS.GET_USERS_BY_ROLE, {
                    queryParams: {
                        roles: this.state.rolesValue,
                        page,
                    },
                }),
            );
        }

        const qs = new QueryScheduler({
            queue,
            limit: COUNT_FIRST,
            onProgress: this.onProgress.bind(this),
            onSuccess: this.onSuccess.bind(this),
        });

        qs.run();

        this.setState({
            areAllUsersLoading: true,
        });
    }

    onProgress() {}

    onSuccess(data: IProgressData) {
        const { success, failed } = data;
        const usersAll: any = [];

        this.setState({
            areAllUsersLoading: false,
        });

        success.forEach((response) => {
            usersAll.push(...response?.report);
        });

        if (!failed.length) {
            this.setState({
                usersAll,
                addManageRoleIsOpen: true,
                withRemove: false,
                confirmError: null,
            });
        } else {
            this.setState({
                addEverybodyErrors: failed.map((error) => error.error),
            });
        }
    }

    closeAddEverybodyErrors() {
        this.setState({ addEverybodyErrors: [] });
    }

    componentDidMount() {
        const roleId = (new URLSearchParams(this.props?.location?.search)).get('roleId') ?? '';

        this.setState({
            rolesValue: roleId,
        }, () => {
            this.getData(0);
        });

        this.getRoles();
        window.addEventListener('resize', this.onResize);
    }

    componentWillUnmount() {
        window.removeEventListener('resize', this.onResize);
        this.request.abort();
    }

    componentDidUpdate(prevProps: Readonly<IRolesUsersProps>, prevState: Readonly<IRolesUsersState>) {
        if (this.props?.location.search !== prevProps?.location.search) {
            const roleId = (new URLSearchParams(this.props?.location?.search)).get('roleId') ?? '';

            this.setState({
                rolesValue: roleId,
            }, () => {
                this.getData(0);
            });
        }
    }

    render() {
        const {
            error,
            isLoading,
            pagination,
            checkState,
            users,
            windowSize,
            addManageRoleIsOpen,
            confirmRemoveIsOpen,
            accept,
            confirmError,
            isWorking,
            rolesValue,
            rolesOptions,
            selectError,
            selectLoading,
            enableRoles,
            currentRole,
            usersState,
            question,
            areAllUsersLoading,
            addEverybodyErrors,
        } = this.state;
        const { BlockRules } = this.props;

        return error ? <SimpleError error={error}/> : isLoading ? <Spin/> : (
            <div className={styles.container}>
                <h2 className={styles.title}>Роль:&nbsp;
                    <span>{rolesValue}</span>
                    {pagination?.total_results &&
                    ` (${pagination?.total_results})`}
                </h2>
                <div className={styles.content}>
                    {!users.length ? <NoInformation/> : (
                        <>
                            <div className={`${styles.header} ${styles.row}`}>
                                <span>#</span>
                                <span>
                                    <Checkbox checked={checkState}
                                              onChange={this.checkboxHandler.bind(this)}/>
                                </span>
                                <span>Username</span>
                                <span>Статус</span>
                                <span>ФИО</span>
                            </div>
                            <VirtualList width={'100%'}
                                         height={windowSize.height - HEIGHT_PADDING}
                                         itemCount={users.length}
                                         itemSize={ITEM_SIZE}
                                         renderItem={({ index, style }) => {
                                             const user = users[index];
                                             const { usersState } = this.state;

                                             return (
                                                 <div key={index}
                                                      style={style}
                                                      className={styles.row}>
                                                     <span>
                                                         {USERS_PER_PAGE * (pagination?.current_page ?? 0) + index + 1}
                                                     </span>
                                                     <span>
                                                         <Checkbox checked={usersState[index]}
                                                                   onChange={this.userCheckboxHandler
                                                                       .bind(this, index)}/>
                                                     </span>
                                                     <span>
                                                         {user.username ? (
                                                             <Link href={`#/clients/${user.id}/user-access`}>
                                                                 {user.username}
                                                             </Link>
                                                         ) : EMPTY_DATA}
                                                     </span>
                                                     <span>{user.status || EMPTY_DATA}</span>
                                                     <span>
                                                         {UserInfoHandler.getPrintNameWithoutUserName.call(user)}
                                                     </span>
                                                 </div>
                                             );
                                         }}/>
                            <div className={styles.pagination}>
                                {pagination?.current_page
                                    ? <span>
                                        <Link onClick={this.changePage.bind(this, pagination.current_page - 1)}>
                                            назад
                                        </Link>
                                    </span>
                                    : undefined}
                                <span className={styles.pages}>
                                    {`${pagination?.current_page + 1} / ${pagination?.total_pages}`}
                                </span>
                                {pagination?.current_page >= 0
                                && pagination?.total_pages
                                && pagination.total_pages - 1 !== pagination.current_page
                                    ? <span>
                                        <Link onClick={this.changePage.bind(this, pagination.current_page + 1)}>
                                            вперед
                                        </Link>
                                    </span>
                                    : undefined}
                            </div>
                        </>
                    )}
                </div>
                <div className={styles.sidebar}>
                    {selectError
                        ? <SimpleError error={selectError}/>
                        : selectLoading
                            ? <Spin size={'s'}/>
                            : <Select options={rolesOptions}
                                      initialValues={rolesValue ? [rolesValue] : []}
                                      placeholder={'Выбрать роль'}
                                      onSelect={this.onSelect.bind(this)}/>
                    }
                    <hr/>
                    <Button className={styles.button}
                            colorType={ButtonTypes.negative}
                            onClick={this.removeRoles.bind(this)}
                            disabled={!usersState.some((item: boolean) => item)}>Отозвать текущую роль</Button>
                    {BlockRules?.RoleAdd && (
                        <>
                            <div>
                                <Button className={styles.button}
                                        colorType={ButtonTypes.positive}
                                        disabled={!usersState.some((item: boolean) => item)}
                                        onClick={this.addRoles.bind(this, false)}>Добавить ещё роль</Button>
                                {addEverybodyErrors.length
                                    ? <ErrorsModal errors={addEverybodyErrors}
                                                   onClose={this.closeAddEverybodyErrors.bind(this)}
                                                   title={'Ошибка загрузки списков пользователей'}/>
                                    : null
                                }
                                <Button className={styles.button}
                                        isLoading={areAllUsersLoading}
                                        colorType={ButtonTypes.negative}
                                        disabled={!users.length || isLoading}
                                        onClick={this.addEverybodyRoles.bind(this)}>Добавить ВСЕМ ещё роль</Button>
                                <Button className={styles.button}
                                        disabled={!usersState.some((item: boolean) => item)}
                                        onClick={this.addRoles.bind(this, true)}>
                                    Добавить + Отозвать текущую роль
                                </Button>
                            </div>
                            <hr/>
                        </>
                    )}
                    <Button className={styles.button}
                            disabled={!usersState.some((item: boolean) => item)}
                            onClick={this.activateRole.bind(this)}>Активировать роль</Button>
                    <Button className={styles.button}
                            colorType={ButtonTypes.negative}
                            disabled={!usersState.some((item: boolean) => item)}
                            onClick={this.deactivateRole.bind(this)}>ДЕактивировать роль</Button>
                    <hr/>

                    <SaveUserButton roles={rolesValue}/>

                    {addManageRoleIsOpen && (
                        <AddManageRole onClose={this.onCloseModal.bind(this)}
                                       isWorking={isWorking}
                                       action={this.addOneRole.bind(this)}
                                       currentRole={currentRole}
                                       error={confirmError}/>
                    )}
                    {confirmRemoveIsOpen && (
                        <Confirm accept={this.accept.bind(this)}
                                 onClose={this.onCloseModal.bind(this)}
                                 isWorking={isWorking}
                                 question={`Отозвать роль ${rolesValue}?`}
                                 error={confirmError}/>
                    )}
                    {enableRoles && (
                        <Confirm accept={accept.bind(this)}
                                 onClose={this.onCloseModal.bind(this)}
                                 isWorking={isWorking}
                                 question={question}
                                 error={confirmError}/>
                    )}
                </div>
            </div>
        );
    }
}
