import * as React from 'react';

import { Dict, ENVIRONMENT, LSSettingItems } from '../../../../../types';
import { EMPTY_DATA } from '../../../../constants';
import { CancelButton, SaveButton } from '../../../../ui/Button';
import { Confirm, Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Link } from '../../../../ui/Link';
import * as styleTable from '../../../../ui/Table/index.css';
import LS from '../../../../utils/localStorage/localStorage';
import { FormConstructor } from '../../../FormConstructor';
import { controlType, ISchemaItem } from '../../../FormConstructor/types';

const NEW_EXTRA_ACCOUNT_SCHEMA: Dict<ISchemaItem> = {
    name: { type: controlType.string, display_name: 'Имя' },
    id: { type: controlType.string, display_name: 'id' },
    service: { type: controlType.variants, variants: Object.values(ENVIRONMENT), display_name: 'Окружение' },
};

export interface IExtraAccount {
    readonly id: string;
    name?: string;
    service: string;
}

interface IUserExtraAccountsProps {
    onClose?: (accounts: IExtraAccount[]) => void;
}

interface IUserExtraAccountsState {
    currentService: ENVIRONMENT;
    isExtraAccountsModalOpen: boolean;
    accounts: IExtraAccount[];
    formData: Dict<any>;
    deletingId: string;
    isDeleteConfirmOpen: boolean;
}

export class UserExtraAccounts extends React.Component<IUserExtraAccountsProps, IUserExtraAccountsState> {
    state: IUserExtraAccountsState = {
        currentService: ENVIRONMENT.PRESTABLE,
        isExtraAccountsModalOpen: false,
        accounts: [],
        formData: {},
        deletingId: '',
        isDeleteConfirmOpen: false,
    };
    ls = new LS();

    componentDidMount(): void {
        const accounts = this.getAccountsData();
        const currentService = this.ls.get(LSSettingItems.env);
        this.setState({ accounts, currentService });
    }

    getAccountsData(): IExtraAccount[] {
        return this.ls.get(LSSettingItems.extra_accounts) ?? [];
    }

    openExtraAccountsModal() {
        this.setState({ isExtraAccountsModalOpen: true });
    }

    closeExtraAccountsModal() {
        this.setState({ isExtraAccountsModalOpen: false }, () => {
            if (this.props.onClose) {
                this.props.onClose(this.getAccountsData());
            }
        });
    }

    onFormChange(formData: Dict<any>) {
        this.setState({ formData });
    }

    addExtraAccount() {
        const accounts = this.ls.get(LSSettingItems.extra_accounts);
        const data = this.state.formData;
        accounts.push(data);

        this.ls.set(LSSettingItems.extra_accounts, accounts);
        this.setState({ accounts: this.getAccountsData() });
    }

    removeExtraAccount() {
        const accounts = this.getAccountsData();
        const filteredAccounts = accounts.filter(account => account.id !== this.state.deletingId);

        this.ls.set(LSSettingItems.extra_accounts, filteredAccounts);
        this.setState({ accounts: this.getAccountsData(), deletingId: '', isDeleteConfirmOpen: false });
    }

    openDeleteConfirm(deletingId: string) {
        this.setState({ deletingId, isDeleteConfirmOpen: true });
    }

    closeDeleteConfirm() {
        this.setState({ deletingId: '', isDeleteConfirmOpen: false });
    }

    render() {
        const { currentService, accounts, isExtraAccountsModalOpen, isDeleteConfirmOpen } = this.state;

        const deletingAccount: IExtraAccount = accounts
            .find(account => account.id === this.state.deletingId) as IExtraAccount;

        return <>
            <Link onClick={this.openExtraAccountsModal.bind(this)}>
                Связанные аккаунты
            </Link>
            {isExtraAccountsModalOpen
                ? <Window title={`Связанные аккаунты - ${currentService}`}
                          onClose={this.closeExtraAccountsModal.bind(this)}>
                    {accounts?.length
                        ? <table className={styleTable.table}>
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Имя</th>
                                    <th>Id</th>
                                    <th>Окружение</th>
                                    <th/>
                                </tr>
                            </thead>
                            <tbody>
                                {
                                    accounts.map((account, index) => {
                                        return <tr key={account.id}>
                                            <td>{index + 1}</td>
                                            <td>{account?.name ?? EMPTY_DATA}</td>
                                            <td>
                                                <Link href={`#/clients/${account.id}/info`}>
                                                    {account.id ?? EMPTY_DATA}
                                                </Link>
                                            </td>
                                            <td>
                                                {account?.service ?? EMPTY_DATA}
                                            </td>
                                            <td>
                                                <Link onClick={this.openDeleteConfirm.bind(this, account.id)}>
                                                Удалить
                                                </Link>
                                            </td>
                                        </tr>;
                                    })
                                }
                            </tbody>
                        </table>
                        : <h4>Нет данных о связанных пользователях</h4>}
                    <FormConstructor schema={NEW_EXTRA_ACCOUNT_SCHEMA}
                                     onChange={this.onFormChange.bind(this)}/>
                    <div className={coreStyle.button_container}>
                        <CancelButton onClick={this.closeExtraAccountsModal.bind(this)}/>
                        <SaveButton onClick={this.addExtraAccount.bind(this)}/>
                    </div>
                </Window>
                : null}
            {isDeleteConfirmOpen
                ? <Confirm question={`Удалить запись ${deletingAccount?.name ?? EMPTY_DATA}?`}
                           error={null}
                           onClose={this.closeDeleteConfirm.bind(this)}
                           accept={this.removeExtraAccount.bind(this)}/>
                : null}
        </>;
    }
}
