import * as React from 'react';

import { Dict } from '../../../../../types';
import { Button } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import { isValidJSONString } from '../../../../utils/isValidJSONString';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import { REQUESTS, SETTINGS_REQUESTS } from '../../request';
import { IStatusFilter } from '../component';
import * as styles from './index.css';
import { ADD_STATUS_SCHEMA } from './schema';

interface IAddStatusModalProps {
    onClose: () => void;
    reloadData?: () => void;
    statusToEdit: IStatusFilter;
}

interface IAddStatusModalState {
    [x: number]: any;

    error: Error | null;
    isWorking: boolean;
    body: Dict<any>;
}

export class AddStatusModal extends React.Component<IAddStatusModalProps, IAddStatusModalState> {
    state: IAddStatusModalState = {
        error: null,
        isWorking: false,
        body: {},
    };

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    sendData() {
        const { state_id, state_description, state_priority, conditions, sensor_filter } = this.state.body;
        const { onClose, reloadData } = this.props;

        const sensorFilterConditions = sensor_filter?.conditions?.map((el) => {
            const value_range = isValidJSONString(el.value_range) ? JSON.parse(el.value_range) : null;
            const timestamp_age = isValidJSONString(el.timestamp_age) ? JSON.parse(el.timestamp_age) : null;
            const since_age = isValidJSONString(el.since_age) ? JSON.parse(el.since_age) : null;

            return { sensor: el.sensor, value_range, timestamp_age, since_age };
        });

        this.setState({
            isWorking: true,
        }, () => {
            this.request.exec(REQUESTS.UPSERT_STATUS, {
                body: {
                    objects: [{
                        state_id,
                        state_description,
                        state_priority: +state_priority || 0,
                        state_filter: {
                            conditions,
                        },
                        sensor_filter: {
                            ...sensor_filter,
                            conditions: sensorFilterConditions,
                        },
                    }],
                },
            })
                .then(() => {
                    onClose();
                    if (reloadData) {
                        reloadData();
                    }
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isWorking: false,
                    });
                });
        });
    }

    onChange(value) {
        this.setState({
            body: value,
        });
    }

    render() {
        const { error, isWorking } = this.state;
        const { onClose, statusToEdit } = this.props;
        const arrayType = ADD_STATUS_SCHEMA.sensor_filter.structure.conditions.array_type;
        const sensor_filter = statusToEdit?.sensor_filter
            ? {
                ...statusToEdit.sensor_filter,
                conditions: statusToEdit.sensor_filter?.conditions?.map((el) => {
                    return {
                        sensor: el.sensor,
                        timestamp_age: el.timestamp_age
                            ? JSON.stringify(el.timestamp_age)
                            : arrayType.timestamp_age.default,
                        since_age: el.since_age
                            ? JSON.stringify(el.since_age)
                            : arrayType.since_age.default,
                        value_range: el.value_range
                            ? JSON.stringify(el.value_range)
                            : arrayType.value_range.default,
                    };
                }),
            }
            : {};

        const initialData = {
            state_id: statusToEdit && statusToEdit.state_id || '',
            state_description: statusToEdit && statusToEdit.state_description || '',
            state_priority: statusToEdit
                && statusToEdit.state_priority
                && statusToEdit.state_priority.toString()
                || '',
            conditions: statusToEdit
                && statusToEdit.state_filter
                && statusToEdit.state_filter.conditions
                || '',
            sensor_filter,
        };

        return (
            <Window error={error}
                    onClose={onClose}
                    title={'Добавить/обновить статус'}>
                <FormConstructor schema={ADD_STATUS_SCHEMA}
                                 initialData={initialData}
                                 onChange={this.onChange.bind(this)}/>
                <div className={styles.button_container}>
                    <Button onClick={this.props.onClose.bind(this)}
                            basic>Отмена</Button>
                    <Button isLoading={isWorking}
                            onClick={this.sendData.bind(this, this.state)}>
                        {this.props.statusToEdit && this.props.statusToEdit.state_id ? 'Обновить' : 'Добавить'}
                    </Button>
                </div>
            </Window>
        );
    }
}
