import * as React from 'react';

import { GetFilterTagRulesDesc } from '../../../constants';
import { Button } from '../../../ui/Button';
import { JsonModal } from '../../../ui/FullModal/JsonModal';
import { Link } from '../../../ui/Link';
import * as tblStyles from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import AddStatusModal from './AddStatusModal';
import * as styles from './index.css';
import RemoveStatusModal from './RemoveStatusModal';

interface IStatusEditorProps {

}

export interface IStatusFilter {
    state_priority: number;
    state_id: string;
    state_description: string;
    state_filter: {
        conditions: string;
    };
    sensor_filter: {
        conditions: Record<string, any>[];
    };
}

interface IStatusEditorState {
    [x: number]: any;
    error: Error | null;
    isLoading: boolean;
    objects: IStatusFilter[];
    showAddModal: boolean;
    showJSONModal: boolean;
    showRemoveModal: boolean;
    statusToEdit: IStatusFilter | null;
}

export class StatusEditor extends React.Component<IStatusEditorProps, IStatusEditorState> {
    state: IStatusEditorState = {
        objects: [],
        error: null,
        isLoading: false,
        showAddModal: false,
        showJSONModal: false,
        showRemoveModal: false,
        statusToEdit: null,
    };

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    getData() {
        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_STATE_FILTERS)
                .then((response) => {
                    const { objects } = response;

                    this.setState({
                        objects,
                        isLoading: false,
                    });
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    showModal(key, value, object) {
        this.setState({
            [key]: value,
            statusToEdit: object,
        });
    }

    componentDidMount() {
        this.getData();
    }

    render() {
        const { error, isLoading, objects, statusToEdit, showAddModal, showRemoveModal, showJSONModal } = this.state;

        return error ? <SimpleError error={error} /> : isLoading ? <Spin /> : (
            <>
                {showAddModal && (
                    <AddStatusModal statusToEdit={statusToEdit}
                                    reloadData={this.getData.bind(this)}
                                    onClose={this.showModal.bind(this, 'showAddModal', false, null)}/>
                )}
                {showRemoveModal && (
                    <RemoveStatusModal statusToEdit={statusToEdit}
                                       reloadData={this.getData.bind(this)}
                                       onClose={this.showModal.bind(this, 'showRemoveModal', false, null)}/>
                )}
                {showJSONModal && <JsonModal onClose={this.showModal.bind(this, 'showJSONModal', false, null)}
                                             title={'Sensor filter JSON'}
                                             obj={statusToEdit?.sensor_filter}/>}
                <Button className={styles.button}
                        onClick={this.showModal.bind(this, 'showAddModal', true, null)}>Добавить статус</Button>
                <table className={`${tblStyles.table} ${styles.table}`}>
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>State priority</th>
                            <th>State ID</th>
                            <th>State description</th>
                            <th>Conditions</th>
                            <th>Sensor filter</th>
                            <th/>
                            <th/>
                        </tr>
                    </thead>
                    <tbody>
                        {objects.map((object, index) => {
                            return (
                                <StatusRow key={index}
                                           object={object}
                                           index={index}
                                           showJSONModal={this.showModal.bind(this, 'showJSONModal', true, object)}
                                           edit={this.showModal.bind(this, 'showAddModal', true, object)}
                                           remove={this.showModal.bind(this, 'showRemoveModal', true, object)}/>
                            );
                        })}
                    </tbody>
                </table>
                {GetFilterTagRulesDesc(styles.addendum)}
            </>
        );
    }
}

const StatusRow = (props: {
    object: IStatusFilter; index: number; edit: () => void; remove: () => void; showJSONModal: () => void;
}) => {
    const { object, index, edit, remove, showJSONModal } = props;

    return (
        <tr>
            <td>{index + 1}</td>
            <td>{object.state_priority}</td>
            <td>{object.state_id}</td>
            <td>{object.state_description}</td>
            <td>{object.state_filter && object.state_filter.conditions}</td>
            <td><Link onClick={showJSONModal}>{object.sensor_filter ? 'JSON' : ''}</Link></td>
            <td><Link onClick={edit}>обновить</Link></td>
            <td><Link onClick={remove}>удалить</Link></td>
        </tr>
    );
};
