import * as React from 'react';

import { Button, CancelButton } from '../../../../ui/Button';
import { Collapse2 } from '../../../../ui/Collapse2';
import { Window } from '../../../../ui/FullModal';
import Select from '../../../../ui/Select';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import { SimpleError } from '../../../SimpleError';
import { REQUESTS,SETTINGS_REQUESTS } from '../../request';
import style from '../index.css';

interface IChangeTagTypeModalProps {
    onClose: () => void;
    tagName: string;
    typeItems: { text: string; value: string }[];
    tagDescriptions: Record<string, any>;
    formValues: Record<string, any>;
    tagChangeData: Record<string, any>;
}

interface IChangeTagTypeModalState {
    formValues: Record<string, any>;
    newType: string | null;
    isLoading: boolean;
    deletingError: Error | null;
    creatingError: Error | null;
}

export default class ChangeTagTypeModal extends React.Component <IChangeTagTypeModalProps, IChangeTagTypeModalState> {
    state: IChangeTagTypeModalState = {
        formValues: {},
        newType: null,
        isLoading: false,
        deletingError: null,
        creatingError: null,
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    setType(newType) {
        this.setState({ newType });
    }

    onFormChange(formValues) {
        this.setState({ formValues });
    }

    changeType() {
        const { tagChangeData } = this.props;
        const { formValues, newType } = this.state;

        const tag = {
            ...tagChangeData,
            type: newType,
            meta: { ...formValues },
        };

        this.deletePrevTag(tag);
    }

    deletePrevTag(tag) {
        const { tagName } = this.props;

        this.setState({ isLoading: true }, () => {
            this.request.exec(REQUESTS.DELETE_TAG, {
                queryParams: {
                    tag_name: tagName,
                    real_remove: true,
                },
            })
                .then(() => {
                    this.createTag(tag);
                })
                .catch(error => {
                    this.setState({
                        isLoading: false,
                        deletingError: error,
                    });
                });
        });
    }

    createTag(data) {
        const { onClose } = this.props;

        this.request.exec(REQUESTS.CREATE_TAG, {
            body: data,
        })
            .then(() => {
                this.setState({ isLoading: false }, () => {
                    onClose();
                });
            })
            .catch(creatingError => {
                this.setState({ isLoading: false, creatingError });
            });
    }

    render() {
        const { onClose, tagName, typeItems, tagDescriptions, formValues } = this.props;
        const { newType, creatingError, deletingError, isLoading } = this.state;

        const schema = newType ? tagDescriptions[newType] ?? tagDescriptions.default : null;

        return <Window onClose={onClose?.bind(this)} title={`Изменить тип тега ${tagName}`}>
            <Select options={typeItems} onSelect={this.setType.bind(this)} placeholder={'Новый тип тега'}/>

            {newType && !schema
                ? <h4>Не удалось получить форму</h4>
                : null
            }

            {schema
                ? <FormConstructor schema={schema}
                                   initialData={formValues}
                                   hideChanges={true}
                                   onChange={this.onFormChange.bind(this)}/>
                : null
            }

            {deletingError
                ? <Collapse2 title={'Не удалось удалить тег'}
                             headerClassname={style.change_error}
                             children={<SimpleError error={deletingError}/>}/>
                : null
            }

            {creatingError
                ? <Collapse2 title={'Не удалось создать тег с новым типом'}
                             headerClassname={style.change_error}
                             children={<SimpleError error={creatingError}/>}/>
                : null
            }

            <div className={style.change_buttons}>
                <CancelButton onClick={onClose.bind(this)}/>

                <Button onClick={this.changeType.bind(this)} isLoading={isLoading}>
                    Изменить
                </Button>
            </div>
        </Window>;
    }
}
