import * as React from 'react';
import { RouteComponentProps } from 'react-router';

import { UserInfoHandler } from '../../../../models/user';
import { Button, ButtonTypes, CancelButton, DeleteButton, SaveButton } from '../../../../ui/Button';
import { Confirm, Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Input } from '../../../../ui/Input';
import { Link } from '../../../../ui/Link';
import Select from '../../../../ui/Select';
import TextArea from '../../../../ui/TextArea';
import { isValidJSONString } from '../../../../utils/isValidJSONString';
import { Request2 } from '../../../../utils/request';
import { deepCopy } from '../../../../utils/utils';
import { Copy } from '../../../Copy';
import { FormConstructor } from '../../../FormConstructor';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import { ITag } from '../../../TagModal/component';
import { ProposeCommentModal } from '../../ProposeCommentModal';
import { REQUESTS, SETTINGS_REQUESTS } from '../../request';
import { DEFAULT_LOCATION } from '../constants';
import * as ListStyle from '../index.css';
import ChangeTagTypeModal from './ChangeTagTypeModal';

interface ITagEditModalProps extends RouteComponentProps<{ tag_id: string; proposition_index: string }> {
    constants: Promise<any>;
    createTag: (data: any) => void;
    onClose: (success: boolean) => {};
    deleteTag: (tagName: string) => void;
    error: Error | null;
    tagType?: string;
    tagName?: string;
    displayName?: string;
    tagDescription?: string;
    initialData?: Record<string,any>;
}

interface ITagEditModalState {
    [key: string]: any;

    tagName: string;
    displayName: string;
    tagDescription: string;
    default_priority: string;
    question: string;
    accept: () => void;

    isLoading: boolean;
    isConfirmOpen: boolean;
    loadingError: Error | null;
    error: Error | null;
    proposeError: Error | null;

    tags: ITag[];
    tagType: string[];
    tagTypes: string[];
    iface_tag_descriptions: any;
    selectedTag: any;

    initFormData: Record<string,any> | null;
    meta: Record<string,any>;
    tag: Record<string, any>;
    propositions: Record<string, any>[];
    users: Record<string, any>;
    propositionIndex: number;

    isFormValid: boolean;
    isFormChanged: boolean;
    isChangeOpened: boolean;
    isPropositionConfirmOpen: boolean;
    isWorking: boolean;
}

const TITLE = 'Создание тега';

enum PropositionsActions {
    CONFIRM = 'confirm',
    REJECT = 'reject'
}

const PROPOSITIONS_ACTIONS: Record<string, { text: string; question: string; answer: string; action: string }> = {
    CONFIRM: {
        text: 'Принять',
        question: 'Принять предложение?',
        answer: 'Модификация тега была принята',
        action: PropositionsActions.CONFIRM,
    },
    REJECT: {
        text: 'Отклонить',
        question: 'Отклонить предложение?',
        answer: 'Модификация тега была отклонена',
        action: PropositionsActions.REJECT,
    },
};

export default class TagEditModal extends React.Component <ITagEditModalProps, ITagEditModalState> {
    state: ITagEditModalState = {
        tagName: '',
        displayName: '',
        tagDescription: '',
        tagType: [],

        tags: [],

        default_priority: '',
        question: '',
        accept: () => {},
        isLoading: false,
        loadingError: null,
        error: null,
        proposeError: null,
        isConfirmOpen: false,
        tagTypes: [],
        iface_tag_descriptions: {},
        selectedTag: null,
        initFormData: null,
        meta: {},
        tag: {},
        propositions: [],
        users: [],
        propositionIndex: -1,
        isFormValid: false,
        isFormChanged: false,
        isChangeOpened: false,
        isPropositionConfirmOpen: false,
        isWorking: false,
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    componentDidMount() {
        this.getData();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    componentDidUpdate(prevProps: Readonly<ITagEditModalProps>): void {
        if (this.props.location.pathname !== prevProps.location.pathname) {
            this.getData();
        }
    }

    getData() {
        this.setState({ isLoading: true, loadingError: null }, () => {
            const {
                match, constants, tagType: tagTypeProps, initialData,
                tagName, tagDescription, displayName,
            } = this.props;
            const propositionIndex: any = match?.params?.proposition_index || -1;

            Promise.all([
                constants,
                this.request.exec(REQUESTS.GET_TAGS)])
                .then(([responseConstants, responseTags]) => {

                    const tagTypes = responseConstants?.tag_types ?? [];
                    const iface_tag_descriptions = responseConstants?.iface_tag_descriptions ?? {};

                    const activeTags = responseTags?.records ?? [];
                    const deprecatedTags = responseTags?.deprecated ?? [];

                    const tags = [...activeTags, ...deprecatedTags];
                    const propositions = responseTags?.propositions ?? [];
                    const users = responseTags?.users ?? [];

                    const id = match?.params?.tag_id
                        ? decodeURIComponent(match?.params?.tag_id)
                        : '';
                    const tagsById = tags?.filter(tag => tag?.name === id);
                    let tag: Record<string, any> = tagsById[tagsById.length - 1] || {};
                    const tagPropositions = propositions?.filter((el) => el.name === id);
                    if (propositionIndex !== -1 && propositionIndex < tagPropositions.length) {
                        tag = tagPropositions[propositionIndex];
                    } else {
                        tag.propositions = tagPropositions;
                    }

                    const { meta = '', type = '', default_priority = '0' } = tag;
                    const name = tag?.name ?? tagName ?? '';
                    const comment = tag?.comment ?? tagDescription ?? '';
                    const display_name = tag?.display_name ?? displayName ?? '';

                    const initFormData = meta && isValidJSONString(meta)
                        ? JSON.parse(meta)
                        : typeof(meta) === 'object'
                            ? meta
                            : initialData ?? null;

                    this.setState({
                        isLoading: false,
                        tagTypes,
                        iface_tag_descriptions,
                        tags,
                        propositionIndex,
                        propositions,
                        users,
                        tagType: type
                            ? [type]
                            : tagTypeProps
                                ? [tagTypeProps]
                                : [],
                        tagDescription: comment,
                        tag,
                        tagName: name,
                        displayName: display_name,
                        default_priority: default_priority,
                        initFormData,
                    });
                })
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    onChangeValues(type: string, value: string) {
        this.setState({
            [type]: type === 'tagDescription'
                ? value || ''
                : value,
        });
    }

    onChangeForm(meta: Record<string, any>, isFormValid: boolean, isFormChanged: boolean) {
        this.setState({ meta, isFormValid, isFormChanged });
    }

    onSelect(type: string, value: string) {
        this.setState({ [type]: [value] });
    }

    openConfirm() {
        const { tagName } = this.state;
        this.setState({
            isConfirmOpen: true,
            question: `Удалить тег ${tagName}?`,
            accept: () => this.deleteTag(tagName),
        });
    }

    closeConfirm() {
        this.setState({ isConfirmOpen: false });
    }

    deleteTag(tagName: string) {
        const { onClose } = this.props;
        this.request.exec(REQUESTS.DELETE_TAG, {
            queryParams: { tag_name: tagName },
        })
            .then(() => {
                onClose(true);
                location.hash = DEFAULT_LOCATION;
            })
            .catch((error) => {
                this.setState({ error });
            });
    }

    createTag(state: Record<string, any>) {
        const { onClose } = this.props;

        const data = {
            name: state.tagName,
            display_name: state.displayName,
            comment: state.tagDescription,
            default_priority: state.default_priority,
            meta: state.meta,
            type: state.tagType?.toString(),
            entities: state.selectedTag?.entities,
        };

        this.request.exec(REQUESTS.CREATE_TAG, { body: data })
            .then(() => {
                onClose(true);
            })
            .catch((error) => {
                this.setState({ error });
            });
    }

    openChangeType(isChangeOpened) {
        this.setState({ isChangeOpened });
    }

    handleProposition(type: string) {
        this.setState({
            isConfirmOpen: true,
            question: PROPOSITIONS_ACTIONS[type.toUpperCase()].question,
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    const { tag } = this.state;
                    const proposition_id = tag?.proposition_id;
                    const action = `TAG_${type.toUpperCase()}`;

                    this.request.exec(REQUESTS[action], {
                        body: { proposition_ids: [proposition_id] },
                        queryParams: { comment: tag?.proposition_description || type },
                    })
                        .then(() => {
                            this.setState({
                                isWorking: false,
                                isConfirmOpen: false,
                            }, () => {
                                this.props.onClose(true);
                            });
                        })
                        .catch((error) => {
                            const SELF_CONFIRM_ERROR_KEY = 'self confirm is denied';
                            if (error?.data?.error_details?.special_info?.session_info
                                .hasOwnProperty(SELF_CONFIRM_ERROR_KEY)) {
                                error = new Error('Нельзя подтверждать свои предложения');
                            }

                            this.setState({ error, isWorking: false });
                        });
                });
            },
        });
    }

    setConfirmOpened(isPropositionConfirmOpen: boolean) {
        this.setState({ isPropositionConfirmOpen });
    }

    setComment(comment) {
        this.setState({
            isWorking: true,
            proposeError: null,
        }, () => {
            const { tag, tagName, meta, displayName, tagDescription, default_priority, tagType } = this.state;
            const data = {
                name: tagName,
                display_name: displayName,
                comment: tagDescription,
                default_priority,
                meta,
                type: tagType?.toString(),
                entities: tag?.entities,
                unique_policy: tag?.unique_policy,
                revision: tag?.revision,
            };

            this.request.exec(REQUESTS.TAG_PROPOSE, {
                queryParams: {
                    comment: encodeURIComponent(comment || `proposition to ${tagName}`),
                },
                body: { ...data },
            })
                .then(() => {
                    this.setState({
                        isWorking: false,
                        proposeError: null,
                        isPropositionConfirmOpen: false,
                    }, () => {
                        this.props.onClose(true);
                    });
                })
                .catch((proposeError) => {
                    this.setState({
                        isWorking: false,
                        proposeError,
                    });
                });
        });
    }

    render() {
        const { error, onClose } = this.props;
        const {
            isConfirmOpen, iface_tag_descriptions, tagType, tagTypes, isLoading, proposeError, users,
            initFormData, default_priority, tagName, displayName, tagDescription, isWorking, tag, propositionIndex,
            isFormChanged, isChangeOpened, selectedTag, question, accept, isPropositionConfirmOpen,
        } = this.state;
        const href = `#/settings/tags/${tagName}`;
        const authorInfo = users?.[tag.proposition_author];
        const author = UserInfoHandler.getPrintName.call(authorInfo);
        const schema = iface_tag_descriptions[tagType[0]] || iface_tag_descriptions['default'];
        const selectItems = tagTypes?.map(item => {
            return { text: item, value: item };
        }) || [];

        return <Window onClose={onClose?.bind(this)}
                       title={TITLE}
                       error={this.state.error || error}
                       closeWithConfirm={isFormChanged}>
            {isLoading
                ? <Spin/>
                : <div className={ListStyle.window_content}>
                    {propositionIndex !== -1
                        && <div className={ListStyle.loc_item}>
                            <div>
                                <span>
                                    Автор
                                </span>: <span className={ListStyle.loc_item_value}>
                                    <Link href={`#/clients/${tag.proposition_author}/info`}
                                          target={'_blank'}>
                                        {author}
                                    </Link>
                                </span>
                            </div>
                            <div>
                                <span>
                                    Комментарий
                                </span>: <span className={ListStyle.loc_item_value}>
                                    {tag.proposition_description}
                                </span>
                            </div>
                        </div>}
                    <Select options={selectItems}
                            placeholder={'Тип тега'}
                            onSelect={this.onSelect.bind(this, 'tagType')}
                            initialValues={tagType}/>
                    {
                        iface_tag_descriptions && schema
                            ? <FormConstructor initialData={initFormData}
                                               schema={deepCopy(schema)}
                                               onChange={this.onChangeForm.bind(this)}/>
                            : null
                    }
                    <Input placeholder={'Приоритет по стандарту'}
                           value={default_priority}
                           onChange={this.onChangeValues.bind(this, 'default_priority')}/>
                    <Input placeholder={'Название тега (id)'}
                           value={tagName}
                           onChange={this.onChangeValues.bind(this, 'tagName')}/>
                    <Input placeholder={'Display name'}
                           value={displayName}
                           onChange={this.onChangeValues.bind(this, 'displayName')}/>
                    <TextArea placeholder={'Описание тега'}
                              value={tagDescription}
                              onChange={this.onChangeValues.bind(this, 'tagDescription')}/>
                    {this.state.error || error
                        ? <SimpleError error={this.state.error || error}/>
                        : null}
                    <div>
                        {tag?.propositions?.length
                            ? <>
                                <div>Доступные предложения
                                        ({tag.propositions.length}): {tag.propositions.map((item, index) => {
                                    return <div key={index}>
                                        <Link href={`${href}/${index}`}>
                                            {`Предложение ${index}`}
                                        </Link>&nbsp;—&nbsp;
                                        <Copy text={`${location.origin}${location.pathname}${href}/${index}`}>
                                            Копировать ссылку на предложение
                                        </Copy>
                                    </div>;
                                })}
                                </div>
                            </>
                            : ''}
                        {propositionIndex !== -1
                            && <div>
                                <Link href={href}>
                                    Базовое действие
                                </Link>
                            </div>}
                    </div>
                    <div className={coreStyle.button_container}>
                        <Button onClick={this.setConfirmOpened.bind(this)}
                                disabled={propositionIndex !== -1}>
                            Заявка
                        </Button>
                        <Button colorType={ButtonTypes.positive}
                                disabled={propositionIndex === -1}
                                onClick={this.handleProposition
                                    .bind(this, PROPOSITIONS_ACTIONS.CONFIRM.action)}>
                            {PROPOSITIONS_ACTIONS.CONFIRM.text}
                        </Button>
                        <Button colorType={ButtonTypes.negative}
                                disabled={propositionIndex === -1}
                                onClick={this.handleProposition
                                    .bind(this, PROPOSITIONS_ACTIONS.REJECT.action)}>
                            {PROPOSITIONS_ACTIONS.REJECT.text}
                        </Button>
                    </div>
                    <div className={coreStyle.button_container}>
                        <CancelButton onClick={onClose.bind(this)}/>
                        <Button basic={true}
                                onClick={this.openChangeType.bind(this, true)}
                                disabled={!tagType[0] || propositionIndex !== -1}>
                            Изменить тип
                        </Button>
                        <DeleteButton onClick={this.openConfirm.bind(this)} disabled={propositionIndex !== -1}/>
                        <SaveButton onClick={this.createTag.bind(this, this.state)} disabled={propositionIndex !== -1}/>
                    </div>
                </div>
            }

            {isConfirmOpen
                ? <Confirm question={question}
                           error={error}
                           accept={accept?.bind(this, tagName)}
                           onClose={this.closeConfirm.bind(this)}/>
                : null
            }

            {isPropositionConfirmOpen
                ? <ProposeCommentModal isWorking={isWorking}
                                       error={proposeError}
                                       setComment={this.setComment.bind(this)}
                                       onClose={this.setConfirmOpened.bind(this, false)}/>
                : null}

            {isChangeOpened
                ? <ChangeTagTypeModal onClose={this.openChangeType.bind(this, false)}
                                      tagName={tagName}
                                      typeItems={selectItems}
                                      tagDescriptions={iface_tag_descriptions}
                                      formValues={initFormData ?? {}}
                                      tagChangeData={{
                                          name: tagName,
                                          display_name: displayName,
                                          comment: tagDescription,
                                          default_priority: default_priority,
                                          type: tagType?.toString(),
                                          entities: selectedTag?.entities,
                                      } ?? {}}/>
                : null
            }
        </Window>;
    }
}
