import * as React from 'react';
import { Route, Switch } from 'react-router';

import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { Input } from '../../../ui/Input';
import { IPlaneTreeItem, VirtualTreeList } from '../../../ui/VirtualTreeList';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { ITag } from '../../TagModal/component';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import { DEFAULT_LOCATION, DEPRECATED } from './constants';
import * as componentStyle from './index.css';
import TagEditModal from './TagEditModal';
import { TagListItem } from './TagListItem/component';

interface ISimpleTreeTag {
    name: string;
}

interface ITagsState {
    tags: ITag[];
    propositions: Record<string, any>[];
    tagFilterValue: string;
    isLoading: boolean;
    loadingError: Error | null;
    showPropositions: boolean;
}

const CREATE_NEW_TAG_ID = 'create-new-tag';
const OTHER_TAG_ENTITY = 'other_entity';
const OTHER_TAG_TYPE = 'other_type';
const ID_PATH_SEPARATOR = '.';

export default class Tags extends React.Component<{}, ITagsState> {
    state: ITagsState = {
        tags: [],
        propositions: [],
        tagFilterValue: '',
        isLoading: false,
        loadingError: null,
        showPropositions: false,
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });
    vTree: React.RefObject<any>;

    constructor(props) {
        super(props);
        this.vTree = React.createRef();
    }

    componentDidMount() {
        this.getTags(true);
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getTags(initial = false) {
        this.setState({ isLoading: initial, loadingError: null }, () => {
            this.request.exec(REQUESTS.GET_TAGS)
                .then((response) => {
                    const activeTags = response?.records ?? [];
                    const deprecatedTags = response?.deprecated ?? [];
                    const tags = [...activeTags, ...deprecatedTags];
                    const propositions = response?.propositions ?? [];

                    this.setState({ tags, propositions, isLoading: false }, () => {
                        !initial && this.vTree.current.getPlaneTree(false);
                    });
                })
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    treeBuilder(tags): IPlaneTreeItem<ISimpleTreeTag>[] {
        const sortTreeArray = (treeNode1: IPlaneTreeItem<ISimpleTreeTag>,
            treeNode2: IPlaneTreeItem<ISimpleTreeTag>) => {
            if (treeNode1.meta.id === DEPRECATED) {
                return 1;
            }

            if (treeNode2.meta.id === DEPRECATED) {
                return -1;
            }

            return treeNode1.meta.id.localeCompare(treeNode2.meta.id);
        };

        const tree: IPlaneTreeItem<ISimpleTreeTag>[] = [
            {
                data: { name: DEPRECATED },
                meta: {
                    id: DEPRECATED,
                    children: [],
                    nestingLevel: 0,
                    parentsIds: [],
                    isTag: false,
                    active: false,
                    count: 0,
                },
            },
        ];
        const { showPropositions, propositions } = this.state;

        tags.forEach(tag => {
            const tagEntity = tag?.entities?.[0] ?? OTHER_TAG_ENTITY;
            const tagType = tag?.type ?? OTHER_TAG_TYPE;
            const tagName = tag?.name ?? '';

            if (!showPropositions) {
                tag.propositions = propositions?.filter((el) => el.name === tag.name);
            }

            if (!tree.find(treeItem => treeItem?.meta?.id === tagEntity)) {
                tree.push({
                    data: { name: tagEntity },
                    meta: {
                        id: tagEntity,
                        children: [],
                        nestingLevel: 0,
                        parentsIds: [],
                        isTag: false,
                        active: false,
                        count: 0,
                    },
                });
            }

            const tagEntityItem = tree
                .find(treeItem => treeItem?.meta?.id === tagEntity) as IPlaneTreeItem<ISimpleTreeTag>;
            tagEntityItem.meta.count += 1;
            const tagEntityItemChildren = tagEntityItem?.meta?.children ?? [];

            if (!tagEntityItemChildren
                ?.find(tagEntityItemChild => tagEntityItemChild.meta.id === [tagEntity, tagType]
                    .join(ID_PATH_SEPARATOR))) {
                tagEntityItemChildren.push({
                    data: { name: tagType },
                    meta: {
                        id: [tagEntity, tagType].join(ID_PATH_SEPARATOR),
                        children: [],
                        nestingLevel: 1,
                        parentsIds: [tagEntity],
                        isTag: false,
                        active: false,
                        count: 0,
                    },
                });
            }

            const tagTypeItem = tagEntityItemChildren
                .find(tagEntityItemChild => tagEntityItemChild.meta.id === [tagEntity, tagType]
                    .join(ID_PATH_SEPARATOR)) as IPlaneTreeItem<ISimpleTreeTag>;
            tagTypeItem.meta.count += 1;
            const tagTypeItemChildren = tagTypeItem?.meta?.children ?? [];

            const actionMeta = isValidJSONString(tag.meta)
                ? JSON.parse(tag.meta)
                : typeof(tag.meta) === 'object' ? tag.meta : null;
            const { deprecated } = actionMeta;

            const child = {
                data: tag,
                meta: {
                    id: [tagEntity, tagType, tagName].join(ID_PATH_SEPARATOR),
                    children: null,
                    nestingLevel: 2,
                    parentsIds: [tagEntity, tagType],
                    className: deprecated ? componentStyle.deprecated : '',
                    isTag: true,
                    active: true,
                },
            };
            tagTypeItemChildren.push(child);

            if (deprecated) {
                const tagDeprecatedItem = tree
                    .find(treeItem => treeItem?.meta?.id === DEPRECATED) as IPlaneTreeItem<ISimpleTreeTag>;
                tagDeprecatedItem.meta.count += 1;
                const tagDeprecatedItemChildren = tagDeprecatedItem?.meta?.children ?? [];

                if (!tagDeprecatedItemChildren
                    ?.find(tagDeprecatedItemChild => tagDeprecatedItemChild.meta.id === [DEPRECATED, tagType]
                        .join(ID_PATH_SEPARATOR))) {
                    tagDeprecatedItemChildren.push({
                        data: { name: tagType },
                        meta: {
                            id: [DEPRECATED, tagType].join(ID_PATH_SEPARATOR),
                            children: [],
                            nestingLevel: 1,
                            parentsIds: [DEPRECATED],
                            isTag: false,
                            active: false,
                            count: 0,
                        },
                    });
                }

                const tagDeprecatedTypeItem = tagDeprecatedItemChildren
                    .find(tagDeprecatedItemChild => tagDeprecatedItemChild.meta.id === [DEPRECATED, tagType]
                        .join(ID_PATH_SEPARATOR)) as IPlaneTreeItem<ISimpleTreeTag>;
                tagDeprecatedTypeItem.meta.count += 1;
                const tagDeprecatedTypeItemChildren = tagDeprecatedTypeItem?.meta?.children ?? [];

                tagDeprecatedTypeItemChildren.push({
                    data: tag,
                    meta: {
                        id: [DEPRECATED, tagType, tagName].join(ID_PATH_SEPARATOR),
                        children: null,
                        nestingLevel: 2,
                        parentsIds: [DEPRECATED, tagType],
                        className: componentStyle.deprecated,
                        isTag: true,
                        active: true,
                    },
                });
            }

            [tree, tagEntityItemChildren, tagTypeItemChildren].forEach(treeArr => treeArr.sort(sortTreeArray));
        });

        return tree;
    }

    treeFilter() {
        const { tagFilterValue } = this.state;
        const lowTagFilterValue = tagFilterValue.toLowerCase();

        if (tagFilterValue) {
            return (tag: IPlaneTreeItem<ITag>) => {
                const { name = '', display_name = '', comment = '' } = tag.data;

                return [name, display_name, comment]
                    .some(value => value && value.toLowerCase().includes(lowTagFilterValue));
            };
        }

        return null;

    }

    onFilterChange(tagFilterValue: string) {
        this.setState({ tagFilterValue });
    }

    onCreateTagClick() {
        location.hash = `${DEFAULT_LOCATION}${CREATE_NEW_TAG_ID}`;
    }

    onTagEditModalClose(success: boolean) {
        location.hash = DEFAULT_LOCATION;

        if (success) {
            this.getTags();
        }
    }

    changePropositions(showPropositions: boolean) {
        this.setState({ showPropositions });
    }

    render() {
        const { loadingError, isLoading, tagFilterValue, showPropositions, propositions, tags } = this.state;
        const treeBuilder = this.treeBuilder.bind(this, showPropositions ? propositions : tags);
        const treeFilter = this.treeFilter.bind(this);

        return <>
            {loadingError
                ? <SimpleError error={loadingError} data={{ label: 'Ошибка при загрузке тегов' }}/>
                : null}
            <div className={componentStyle.controls}>
                <Input autoFocus
                       className={componentStyle.filter}
                       placeholder={'Фильтр'}
                       value={tagFilterValue}
                       onChange={this.onFilterChange.bind(this)}/>
                <Button onClick={this.onCreateTagClick.bind(this)}
                        basic
                        className={componentStyle.addNewTagButton}>Добавить новый тег</Button>
                <div className={componentStyle.propositions}>
                    <div className={componentStyle.title}>Предложения:</div>
                    <Checkbox onChange={this.changePropositions.bind(this)}
                              checked={showPropositions}
                              placeholder={'Показать предложения'}/>
                </div>
            </div>
            {isLoading
                ? <Spin/>
                : <VirtualTreeList treeBuilder={treeBuilder}
                                   treeFilter={treeFilter}
                                   ref={this.vTree}
                                   showPropositions={showPropositions}
                                   isFilterValueExist={!!tagFilterValue}
                                   initCollapsedLevels={[0]}
                                   treeListItem={[TagListItem, { showPropositions, propositions }]}/>}
            <Switch>
                <Route path={`${DEFAULT_LOCATION}:tag_id/:proposition_index?`}
                       render={() => <TagEditModal onClose={this.onTagEditModalClose.bind(this)}/>}/>
            </Switch>
        </>;
    }
}
