import React from 'react';
import { connect } from 'react-redux';

import { Dict } from '../../../../types';
import { ONE_SECOND } from '../../../constants';
import { Button, DeleteButton } from '../../../ui/Button';
import FormatDate from '../../../ui/FormatDate';
import { Window } from '../../../ui/FullModal';
import { JsonModal } from '../../../ui/FullModal/JsonModal';
import { Link } from '../../../ui/Link';
import { ConstantsKey, fetchConstants } from '../../../utils/fetchConstants';
import { Request2 } from '../../../utils/request';
import { rubs } from '../../../utils/rubs';
import { IStore } from '../../App/store';
import { WalletActiveIndicator } from '../../B2B/utils/WalletActiveIndicator/component';
import { Copy } from '../../Copy';
import { FormConstructor } from '../../FormConstructor';
import { ISchemaItem } from '../../FormConstructor/types';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import { AccountRemoveModal } from './AccountRemoveModal';
import * as styles from './AddModal.css';
import { IAccount, IAccountHistoryItem } from './types';
import { maskId } from './utils';

interface IAccountDetailsModalState {
    isLoading: boolean;
    isUploading: boolean;
    isHistoryLoading: boolean;
    schema: Dict<Dict<ISchemaItem>>;
    history: IAccountHistoryItem;
    onChangeData: any;
    error: Error | null;
    errorHistory: Error | null;
    accountOperations: any;
    deleteModalOpen: boolean;
}

interface IAccountDetailsModalProps {
    onClose: () => void;
    account: IAccount;
    reloadData?: () => void;
    blockRules?: Dict<boolean>;
}

interface Internal {
    constants: Promise<any>;
}

const mapDispatchToProps = (dispatch: any) => {
    return {
        constants: fetchConstants([ConstantsKey.BILLING], dispatch),
    };
};

class AccountDetailsModalInternal extends
    React.Component<IAccountDetailsModalProps & Internal, IAccountDetailsModalState> {
    state: IAccountDetailsModalState = {
        isLoading: true,
        isHistoryLoading: true,
        isUploading: false,
        schema: {},
        history: {} as IAccountHistoryItem,
        onChangeData: undefined,
        error: null,
        errorHistory: null,
        accountOperations: null,
        deleteModalOpen: false,
    };

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    onChange(onChangeData) {
        this.setState({
            onChangeData,
        });
    }

    getData() {
        this.props.constants
            .then((response) => {
                this.setState({
                    schema: response.billing && response.billing.accounts_data,
                }, () => {
                    this.setState({
                        isLoading: false,
                    });
                });
            })
            .catch((error) => {
                this.setState({
                    error,
                    isLoading: false,
                });
            });
    }

    getHistory() {
        const { account } = this.props;

        if (account.id) {
            this.request.exec(REQUESTS.GET_ACCOUNT_HISTORY, {
                queryParams: {
                    account_id: account.id,
                },
            })
                .then((response) => {
                    response?.links_history?.sort((a, b) => b.history_timestamp - a.history_timestamp);
                    response?.operations_history?.sort((a, b) => b.timestamp - a.timestamp);
                    this.setState({
                        history: response,
                    }, () => {
                        this.setState({
                            isHistoryLoading: false,
                        });
                    });
                })
                .catch((errorHistory) => {
                    this.setState({
                        errorHistory,
                        isHistoryLoading: false,
                    });
                });
        }
    }

    uploadData() {
        const { onClose, reloadData, account } = this.props;

        this.setState({
            isUploading: true,
        }, () => {
            this.request.exec(REQUESTS.UPDATE_ACCOUNT, {
                body: {
                    ...this.state.onChangeData,
                    account_id: account.id,
                },
            })
                .then(() => {
                    reloadData && reloadData();
                    onClose();
                })
                .catch((error) => {
                    this.setState({
                        isUploading: false,
                        error,
                    });
                });
        });
    }

    toggleMetaModal(value) {
        this.setState({
            accountOperations: value,
        });
    }

    componentDidMount() {
        this.getData();
        this.getHistory();
    }

    setDeleteModalOpen(deleteModalOpen) {
        this.setState({ deleteModalOpen });
    }

    render() {
        const {
            isLoading,
            schema,
            isHistoryLoading,
            error,
            errorHistory,
            history,
            isUploading,
            accountOperations,
            deleteModalOpen,
        } = this.state;

        const { onClose, account, blockRules, reloadData } = this.props;
        const showDeleteButton = blockRules?.RemoveB2B;

        return (
            <>
                {accountOperations &&
                    <JsonModal title={'Operation state'}
                               obj={accountOperations}
                               onClose={this.toggleMetaModal.bind(this, null)}/>
                }

                {deleteModalOpen
                    ? <AccountRemoveModal accountId={account.id?.toString()}
                                          onClose={this.setDeleteModalOpen.bind(this, false)}
                                          reloadData={reloadData?.bind(this)}/>
                    : null
                }

                <Window onClose={onClose.bind(this)}
                        title={'Редактировать аккаунт/кошелек'}
                        error={error}>
                    <div className={styles.accountDetailsContainer}>
                        <div className={styles.accountContainerForm}>
                            {isLoading
                                ? <Spin size={'l'} /> : (
                                    <>
                                        <p className={styles.accountId}>
                                            ID: <Copy>{account.id}</Copy>
                                        </p>
                                        <FormConstructor className={styles.form}
                                                         schema={schema[account.data_type]}
                                                         initialData={account?.details}
                                                         onChange={this.onChange.bind(this)}/>
                                        <div className={styles.buttonContainer}>
                                            <Button basic
                                                    onClick={onClose.bind(this)}>Отмена</Button>

                                            {showDeleteButton
                                                ? <DeleteButton onClick={this.setDeleteModalOpen.bind(this, true)}/>
                                                : null
                                            }

                                            <Button disabled={isLoading}
                                                    isLoading={isUploading}
                                                    onClick={this.uploadData.bind(this)}>Отправить</Button>
                                        </div>
                                    </>
                                )
                            }
                        </div>
                        {account.id ? (
                            <div className={styles.accountDetailsHistory}>
                                <h3>История</h3>
                                {isHistoryLoading ? <Spin/> :
                                    errorHistory ? <SimpleError error={errorHistory} data={{ label: 'История' }}/> : (
                                        <>
                                            <h4>Links history</h4>
                                            {history?.links_history && (
                                                <table>
                                                    <thead>
                                                        <tr>
                                                            <th>#</th>
                                                            <th>history_action</th>
                                                            <th>timestamp</th>
                                                            <th>users</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        {history.links_history
                                                            ?.map((el, index) => {
                                                                return (
                                                                    <tr key={index}>
                                                                        <td>{index + 1}</td>
                                                                        <td>{el.history_action}</td>
                                                                        <td>
                                                                            <FormatDate withSecond
                                                                                        value={el.history_timestamp
                                                                                            * ONE_SECOND}/>
                                                                        </td>
                                                                        <td>
                                                                            <Link href={`#/clients/${el.user_id}/billing`}
                                                                                  title={el.user_id}>
                                                                                {maskId(el.user_id)}
                                                                            </Link>
                                                                            <Link href={`#/clients/${el.history_user_id}/billing`}
                                                                                  title={el.history_user_id}>
                                                                            /{maskId(el.history_user_id)}
                                                                            </Link>
                                                                        </td>
                                                                    </tr>
                                                                );
                                                            })
                                                        }
                                                    </tbody>
                                                </table>
                                            )
                                            }
                                            <h4>operations_history</h4>
                                            {history?.operations_history && (
                                                <table>
                                                    <thead>
                                                        <tr>
                                                            <th>#</th>
                                                            <th>history_action</th>
                                                            <th>timestamp</th>
                                                            <th>type_id</th>
                                                            <th>active</th>
                                                            <th>spent</th>
                                                            <th>user</th>
                                                            <th/>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        {history.operations_history
                                                            ?.map((el, index) => {
                                                                return (
                                                                    <tr key={index}>
                                                                        <td>{index + 1}</td>
                                                                        <td>{el.action}</td>
                                                                        <td>
                                                                            <FormatDate withSecond
                                                                                        value={
                                                                                            el.timestamp * ONE_SECOND
                                                                                        }/>
                                                                        </td>
                                                                        <td>{el.type_id}</td>
                                                                        <td>
                                                                            {/* eslint-disable-next-line max-len */}
                                                                            <WalletActiveIndicator className={styles.activeIndicator}
                                                                                                   isActive={
                                                                                                       el?.active
                                                                                                   }/>
                                                                        </td>
                                                                        <td>{rubs(el.spent)}</td>
                                                                        <td>
                                                                            <Link href={`/#/clients/${el.user_id}/billing`}
                                                                                  title={el.user_id}>
                                                                                {maskId(el.user_id)}
                                                                            </Link>
                                                                        </td>
                                                                        <td>
                                                                            <Link onClick={this.toggleMetaModal
                                                                                .bind(this, el)}>
                                                                                json
                                                                            </Link>
                                                                        </td>
                                                                    </tr>
                                                                );
                                                            })
                                                        }
                                                    </tbody>
                                                </table>
                                            )
                                            }
                                        </>
                                    )
                                }
                            </div>
                        ) : undefined}
                    </div>
                </Window>
            </>
        );
    }
}

export const AccountDetailsModal = connect((store: IStore) => {
    return {
        blockRules: store.AdminUser?.blockRules,
    };
}, mapDispatchToProps)(AccountDetailsModalInternal);
