import React from 'react';
import { connect } from 'react-redux';

import { Dict } from '../../../../types';
import { Button, CancelButton } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import { ConstantsKey, fetchConstants } from '../../../utils/fetchConstants';
import { Request2 } from '../../../utils/request';
import { IWalletLimit } from '../../B2B/SearchWallets/types';
import { FormConstructor } from '../../FormConstructor';
import { ISchemaItem } from '../../FormConstructor/types';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import * as styles from './AddModal.css';

interface IWalletAddModalState {
    isLoading: boolean;
    isUploading: boolean;
    schema: Dict<ISchemaItem>;
    wallet: IWalletLimit | null;
    onChangeData: any;
    error: Error | null;
    isFormValid: boolean;
    isFormChanged: boolean;
}

interface IWalletAddModalProps {
    onClose: () => void;
    walletName?: string;
    parentId?: string;
    title: string;
    reloadData?: () => void;
    onCloseAfterSave?: () => void;
}

interface Internal {
    constants: Promise<any>;
}

const mapDispatchToProps = (dispatch: any) => {
    return {
        constants: fetchConstants([ConstantsKey.BILLING], dispatch),
    };
};

class WalletAddModalInternal extends React.Component<IWalletAddModalProps & Internal, IWalletAddModalState> {
    state: IWalletAddModalState = {
        isLoading: true,
        isUploading: false,
        schema: {},
        wallet: null,
        onChangeData: undefined,
        error: null,
        isFormValid: false,
        isFormChanged: false,
    };

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    onChange(onChangeData, isFormValid: boolean, isFormChanged: boolean) {
        this.setState({
            onChangeData, isFormValid, isFormChanged,
        });
    }

    getData() {
        this.props.constants
            .then((response) => {
                this.setState({
                    schema: response.billing && response.billing.accounts,
                }, () => {
                    const { walletName, parentId } = this.props;
                    if (walletName) {
                        this.getAccount(walletName);
                    } else {
                        this.setState({
                            isLoading: false,
                        });

                        if (parentId) {
                            this.setParentId(parentId);
                        }
                    }
                });
            })
            .catch((error) => {
                this.setState({
                    isLoading: false,
                    error,
                });
            });
    }

    setParentId(parentId) {
        const wallet = {
            meta: {
                parent_id: parentId,
            },
        } as IWalletLimit;

        this.setState({
            wallet,
        });
    }

    getAccount(walletName) {
        this.request.exec(REQUESTS.GET_WALLETS_DESCRIPTIONS, {
            queryParams: {
                account_name: walletName,
            },
        })
            .then((response) => {
                this.setState({
                    isLoading: false,
                    wallet: response.accounts[0],
                });
            })
            .catch(() => {
                this.setState({ isLoading: false });
            });
    }

    uploadData() {
        const { onClose, reloadData, onCloseAfterSave } = this.props;

        this.setState({
            isUploading: true,
        }, () => {
            this.request.exec(REQUESTS.UPDATE_WALLET, {
                body: {
                    ...this.state.onChangeData,
                },
            })
                .then(() => {
                    reloadData && reloadData();
                    onCloseAfterSave?.();
                    onClose();
                })
                .catch((error) => {
                    this.setState({
                        isUploading: false,
                        error,
                    });
                });
        });
    }

    componentDidMount() {
        this.getData();
    }

    render() {
        const { isLoading, schema, wallet, error, isFormChanged, isFormValid } = this.state;
        const { title, onClose } = this.props;

        return (
            <Window onClose={onClose.bind(this)}
                    title={title}
                    closeWithConfirm={isFormChanged}>
                <div>
                    {isLoading
                        ? <Spin size={'l'}/> : error
                            ? <SimpleError error={error}/>
                            : (
                                <>
                                    <FormConstructor schema={schema}
                                                     initialData={wallet}
                                                     onChange={this.onChange.bind(this)}/>
                                    <div className={styles.buttonContainer}>
                                        <CancelButton onClick={onClose.bind(this)}/>
                                        <Button disabled={!isFormValid}
                                                isLoading={isLoading}
                                                onClick={() => this.uploadData()}>
                                            Отправить
                                        </Button>
                                    </div>
                                </>
                            )
                    }
                </div>
            </Window>
        );
    }
}

export const WalletAddModal = connect(() => {
    return {};
}, mapDispatchToProps)(WalletAddModalInternal);
