import React from 'react';
import ReactTooltip from 'react-tooltip';

import { EMPTY_DATA } from '../../../constants';
import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { ErrorsModal } from '../../../ui/ErrorBar';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import * as tblStyle from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import { rubs } from '../../../utils/rubs';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import { AccountAddModal } from './AccountAddModal';
import { AccountDetailsModal } from './AccountDetailsModal';
import { AccountRemoveModal } from './AccountRemoveModal';
import * as styles from './index.css';
import { IAccount, IAccountDetails, IAccountUser, IWalletType } from './types';
import { WalletAddModal } from './WalletAddModal';

interface IState {
    loading: boolean;
    numDoc: number;
    error: Error | null;
    rootAccounts: IWalletType[];
    childAccounts: {[account_id: string]: IWalletType[]};
    walletQuantity: number;
    accountDetails: {[account_name: string]: {has_more: boolean; accounts: IAccountDetails[]}};
    showAddModal: boolean;
}

const DEFAULT_NUMDOC = 15;

export class Wallets extends React.Component<any, IState> {
    state: IState = {
        numDoc: DEFAULT_NUMDOC,
        loading: true,
        error: null,
        rootAccounts: [],
        childAccounts: {},
        walletQuantity: 0,
        accountDetails: {},
        showAddModal: false,
    };

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });
    wallets: IWalletType[] = [];

    fetchAccounts() {
        this.request.exec(REQUESTS.GET_WALLETS_DESCRIPTIONS)
            .then((response) => {
                const accounts: IWalletType[] = response.accounts;
                this.wallets = accounts;

                const rootAccounts: IWalletType[] = [];

                accounts.forEach((wallet) => {
                    if (!(wallet && wallet.meta && wallet.meta.parent_id)) {
                        rootAccounts.push(wallet);
                    }
                });

                const walletQuantity = accounts.length;

                this.setState({
                    rootAccounts,
                    walletQuantity,
                    loading: false,
                });
            })
            .catch((error) => {
                this.setState({
                    error,
                    loading: false,
                });
            });
    }

    loadAccountDetails(walletName, from) {
        ReactTooltip.rebuild();

        const { numDoc } = this.state;

        this.request.exec(REQUESTS.GET_WALLET, {
            queryParams: {
                account_name: encodeURIComponent(walletName),
                numdoc: numDoc && !isNaN(numDoc) ? numDoc : DEFAULT_NUMDOC,
                from: from || 0,
            },
        })
            .then((response: {has_more: boolean; accounts: IAccountDetails[]}) => {
                const { accountDetails, childAccounts } = this.state;

                if (from) {
                    accountDetails[walletName] = {
                        accounts: [
                            ...accountDetails[walletName].accounts,
                            ...response.accounts,
                        ],
                        has_more: response.has_more,
                    };
                } else {
                    accountDetails[walletName] = response;
                }

                accountDetails[walletName].accounts.forEach((account) => {
                    const accountId = account && account.data && account.data.id;

                    if (!childAccounts[accountId]) {
                        childAccounts[accountId] = this.wallets.filter((wallet) => {
                            return wallet && wallet.meta && wallet.meta.parent_id === accountId;
                        });
                    }
                });

                this.setState({
                    accountDetails,
                    childAccounts,
                });
            })
            .catch((error) => {
                this.setState({
                    error,
                });

                throw error;
            });
    }

    closeModal() {
        this.setState({
            showAddModal: false,
        });
    }

    numdocChange(numDoc) {
        this.setState({
            numDoc,
        });
    }

    componentDidMount() {
        this.fetchAccounts();
    }

    render() {
        const {
            loading,
            error,
            rootAccounts,
            walletQuantity,
            childAccounts,
            accountDetails,
            showAddModal,
            numDoc,
        } = this.state;

        return (
            <>
                {showAddModal && (
                    <WalletAddModal onClose={this.closeModal.bind(this)}
                                    title={'Добавить кошелек'}
                                    reloadData={this.fetchAccounts.bind(this)}/>
                )}
                {error ? <SimpleError error={error} /> : loading ? <Spin /> : (
                    <>
                        <div className={styles.headerContainer}>
                            <Button onClick={() => {
                                this.setState({
                                    showAddModal: true,
                                });
                            }}>Добавить кошелек</Button>
                            <div className={styles.numDoc}>
                                <span className={styles.numDocText}>Кол-во экземпляров кошельков на странице:</span>
                                <Input className={`${styles.numDoc} ${styles.numDocInput}`}
                                       value={numDoc}
                                       placeholder={''}
                                       onChange={(value) => this.numdocChange(value)}/>
                            </div>
                        </div>
                        <h5>Всего кошельков : {walletQuantity}</h5>
                        <div className={styles.tableContainer}>
                            <table className={`${tblStyle.table} ${tblStyle.clearBackground} ${tblStyle.clearFirstOpacity} ${styles.walletTable}`}>
                                <thead>
                                    <tr>
                                        <th />
                                        <th>Name</th>
                                        <th>HR name</th>
                                        <th>Type</th>
                                        <th>Hard limits</th>
                                        <th>Soft limits</th>
                                        <th>Refresh</th>
                                        <th />
                                    </tr>
                                </thead>
                                <tbody>
                                    {
                                        rootAccounts.map((account, index) => {
                                            return (
                                                <TrParentAccount key={'rootAccount' + index}
                                                                 account={account}
                                                                 index={index}
                                                                 data={{
                                                                     childAccounts: childAccounts,
                                                                     accounts: accountDetails,
                                                                     onClick: this.loadAccountDetails.bind(this),
                                                                 }}
                                                                 isRoot/>);
                                        })
                                    }
                                </tbody>
                            </table>
                        </div>
                    </>
                )}
            </>
        );
    }
}

interface ITrParentAccountProps {
    account: IWalletType;
    index: number;
    data: {
        childAccounts: {[account_id: string]: IWalletType[] };
        accounts: {[account_name: string]: {has_more: boolean; accounts: IAccountDetails[]}};
        onClick: (account, from) => void;
    };
    className?: string;
    isRoot?: boolean;
}

const TrParentAccount = (props: ITrParentAccountProps) => {
    const { account, className, isRoot } = props;
    const { accounts, onClick } = props.data;
    const [isExpanded, setIsExpanded] = React.useState(false);
    const [loading, setLoading] = React.useState(true);
    const [isAccountAddModalShowing, setAccountAddModalShow] = React.useState(false);
    const [isWalletModalShowing, setWalletModalShow] = React.useState(false);
    const [walletName, setWalletName] = React.useState('');
    const [newAccount, setNewAccount] = React.useState<IAccount | null>(null);
    const [errors, setErrors] = React.useState<Error[]>([]);

    const accountsLoaded = accounts && accounts[account.name] && accounts[account.name].accounts;
    const accountsNumber = accountsLoaded && accounts[account.name].accounts.length;

    if (loading && accountsLoaded) {
        setLoading(false);
    }

    const request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    const getNewAccount = async (newAccountId) => {
        try {
            const response = await request.exec(REQUESTS.GET_WALLET, {
                queryParams: {
                    account_name: encodeURIComponent(account.name),
                    numdoc: 999999,
                    from: 0,
                },
            });
            const responseAccount = response?.accounts
                ?.find((account) => account?.data?.id === newAccountId)?.data ?? null;

            if (responseAccount) {
                setNewAccount(responseAccount);
            } else {
                setErrors([new Error('Аккаунт не найден :(')]);
            }
        }
        catch (error) {
            setErrors([error]);
        }
    };

    return (
        <>
            {errors.length
                ? <ErrorsModal title={'Ошибка открытия окна редактирования'}
                               errors={errors}
                               onClose={() => {
                                   setErrors([]);
                                   onClick(account.name, 0);
                               }}/>
                : null
            }
            <ReactTooltip id={account.name}
                          place={'right'}
                          type={'info'}/>
            {isWalletModalShowing && (
                <WalletAddModal onClose={() => {
                    setWalletModalShow(false);
                }}
                                walletName={walletName}
                                title={walletName ? 'Редактировать кошелек' : 'Добавить кошелек'}/>
            )}
            {isAccountAddModalShowing && (
                <AccountAddModal accountName={account.name}
                                 onClose={setAccountAddModalShow.bind(null, false)}
                                 getNewAccount={getNewAccount.bind(null)}/>
            )}
            {newAccount && (
                <AccountDetailsModal reloadData={onClick.bind(null, account.name, 0)}
                                     onClose={() => {
                                         setNewAccount(null);
                                     }}
                                     account={newAccount}/>
            )}
            <tr className={className && isExpanded ? `${className} ${styles.selectedRow}`:
                isExpanded ? styles.selectedRow :
                    className ? `${className}` :
                        isRoot ? styles.rootRow : undefined}
                onClick={() => {
                    setIsExpanded(!isExpanded);

                    if (!isExpanded) {
                        onClick(account.name, 0);
                    }
                }}>
                <td>
                    <span className={`${styles.expandToggle} ${isExpanded && styles.expandedToggle}`}>
                        {loading && isExpanded ? <Spin size={'s'}/> : isExpanded ? 'Свернуть' : 'Развернуть'}
                    </span>
                </td>
                <td>
                    <Link onClick={() => {
                        setWalletModalShow(true);
                        setWalletName(account.name);
                    }}>
                        {account.name}
                    </Link>
                </td>
                <td>{account.meta && account.meta.hr_name || EMPTY_DATA}</td>
                <td>{account.type}</td>
                <td>{rubs(account.hard_limit)}</td>
                <td>{rubs(account.soft_limit)}</td>
                <td>{(account.meta && account.meta.refresh_interval || EMPTY_DATA) + ' ' + (account.meta && account.meta.refresh_policy || EMPTY_DATA)}</td>
            </tr>
            {isExpanded && accountsLoaded && (
                <>
                    <tr className={`${isRoot && styles.childRow} ${!isRoot && styles.grandchildRow}`}>
                        <th />
                        <th>Account ID</th>
                        <th>Hard limit/balance</th>
                        <th colSpan={4}>Users</th>
                    </tr>
                    <tr className={`${isRoot && styles.childRow} ${!isRoot && styles.grandchildRow}`}>
                        <td colSpan={7} className={styles.addAccountButton}>
                            <Link onClick={() => {
                                setAccountAddModalShow(true);
                            }}>
                                Добавить экземпляр
                            </Link>
                        </td>
                    </tr>
                    {accounts[account.name].accounts.map((account, index) => {
                        return (
                            <TrAccount key={'account'+ index}
                                       data={props.data}
                                       account={account}
                                       className={!isRoot && styles.grandchildRow}/>
                        );
                    })}
                    {accounts[account.name].has_more && (
                        <tr>
                            <td colSpan={9}
                                className={`${!isRoot && styles.grandchildRow} ${styles.paginationToggle} ${isRoot && styles.childRow}`}>
                                <Link onClick={() => {
                                    onClick(account.name, accountsNumber);
                                }}>
                                    Ещё (показано {accountsNumber})
                                </Link>
                            </td>
                        </tr>
                    )}
                </>
            )}
        </>
    );
};

interface ITrAccountProps {
    data: {
        childAccounts: {[account_id: string]: IWalletType[]};
        accounts: {
            [account_name: string]: {
                has_more: boolean;
                accounts: IAccountDetails[];
            };
        };
        onClick: (account, from) => void;};
    account: IAccountDetails;
    className?: string;
}

const TrAccount = (props: ITrAccountProps) => {
    const { className } = props;
    const { childAccounts, onClick } = props.data;
    const account = props.account.data;
    const users: IAccountUser[] = props.account.users;

    const children = childAccounts[account.id];
    const accountName = account.comment ? `${account.comment} (${account.id})` : `${account.id}`;

    const [isExpanded, setIsExpanded] = React.useState(false);
    const [isRemoveModalShowing, setRemoveModalShow] = React.useState(false);
    const [isAccountModalShowing, setAccountModalShow] = React.useState(false);
    const [isActive, setActive] = React.useState(account.is_active);

    const request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    const onChange = () => {
        request.exec(REQUESTS.ACTIVATE_ACCOUNT, {
            body: {
                account_id: account.id,
                active_flag: !account.is_active,
            },
        })
            .then(() => {
                setActive(!isActive);
            })
            .catch((error) => {
                throw error;
            });
    };

    return (
        <>
            {isRemoveModalShowing && (
                <AccountRemoveModal accountId={`${account.id}`}
                                    onClose={() => setRemoveModalShow(false)}
                                    reloadData={onClick.bind(null, account.type_name, 0)}/>
            )}
            {isAccountModalShowing && (
                <AccountDetailsModal onClose={() => setAccountModalShow(false)}
                                     reloadData={onClick.bind(null, account.type_name)}
                                     account={props.account.data}/>
            )}
            <tr className={className ? `${styles.childRow} ${className}` : styles.childRow}
                onClick={() => {
                    setIsExpanded(!isExpanded);
                }}>
                <td data-for={account.type_name}
                    data-tip={account.is_active ? 'Деактивировать аккаунт' : 'Активировать аккаунт'}>
                    <Checkbox checked={isActive}
                              className={styles.checkbox}
                              onChange={onChange.bind(null)}/>
                    <span className={`${styles.expandToggle} ${isExpanded && styles.expandedToggle}`}>
                        {children[0] && (isExpanded ? 'Свернуть' : 'Развернуть')}
                    </span>
                </td>
                <td>
                    <Link onClick={() => setAccountModalShow(true)}>
                        {accountName}
                    </Link>
                </td>
                <td>{rubs(account.hard_limit)} / {rubs(account.balance)}</td>
                <td colSpan={4}>
                    <div>
                        {users[0] ? users.map((user, index) => {
                            return (
                                <a key={index}
                                   href={`#/clients/${user.id}/billing`}
                                   className={styles.userLink}>
                                    {user.info || user.id}
                                </a>
                            );
                        }) : (
                            <Link className={styles.deleteButton}
                                  onClick={() => setRemoveModalShow(true)}>
                                [Удалить]
                            </Link>
                        )}
                    </div>
                </td>
            </tr>
            {isExpanded && children && children.map((child, index) => {
                return (
                    <TrParentAccount className={styles.childRowLimit}
                                     key={'parentAccount' + index}
                                     account={child}
                                     index={index}
                                     data={props.data}/>
                );
            })}
        </>
    );
};
