import * as React from 'react';

import { EMPTY_DATA } from '../../constants';
import { Format, IFormattedItem, wikiFormatter } from '../../utils/wikiFormatter';
import { Copy } from '../Copy';
import * as style from './index.css';

export interface ISimpleError {
    error: any;
    data?: {
        label: string;
    };
    className?: string;
    highlightBackendCluster?: boolean;
}

export const RESPONSE_STATUSES = {
    CODE_200: 200,
    CODE_400: 400,
    CODE_401: 401,
    CODE_403: 403,
    CODE_404: 404,
    CODE_418: 418,
    CODE_500: 500,
    CODE_502: 502,
    CODE_503: 503,
};

const STATUSES_MAP = {
    400: 'Неверный запрос',
    401: 'Не авторизирован',
    403: 'Нет доступа',
    404: 'Не найдено',
    500: 'Внутренняя ошибка сервера',
    502: 'Ошибочный шлюз',
    503: 'Сервис недоступен',
};

const SPACES = 1;

function newWikiObj (format: Format = Format.Bold, content: string | null, title = ''): IFormattedItem {
    return {
        wikiFormat: format,
        content: content ?? EMPTY_DATA,
        title: title,
    };
}

function SafeStringify(obj) {
    if(obj === null) {return null;}

    try {
        return JSON.stringify(obj, null, SPACES);
    } catch (e) {
        return null;
    }
}

export const SimpleError = ({ error, data, className = '', highlightBackendCluster }: ISimpleError) => {
    const special_info = error?.response?.error_details?.special_info;
    const status = error?.response?.status;
    const back = error?.response?.config?.headers?.saasnew || null;
    const ccpgf = error?.response?.data?.error?.indexOf?.('cannot construct processor generator for') || false;
    const endPoint = error?.response?.config?.url;
    const admbuild = error?.response?.config?.headers?.admbuild;
    const reqid = endPoint ? new URLSearchParams(endPoint.slice(endPoint.indexOf('?'))).get('reqid') : null;
    const statusName = `${status} ${STATUSES_MAP[status] ? STATUSES_MAP[status] : ''}`;

    const endpointStringLength = 40;

    const formattedArr = [
        newWikiObj(Format.Bold, endPoint, 'Endpoint'),
        newWikiObj(Format.Bold, statusName, 'Status'),
        newWikiObj(Format.Bold, reqid, 'ReqId'),
        newWikiObj(Format.Bold, back, 'Back'),
        newWikiObj(Format.Bold, admbuild, 'Версия админки'),
        newWikiObj(Format.Bold, SafeStringify(error?.request?.options), 'Options'),
        newWikiObj(Format.DropDownJson, SafeStringify(error?.response), 'Посмотреть ответ целиком'),
        newWikiObj(Format.DropDownJson, SafeStringify(error?.request?.body), 'Body запроса'),
    ];

    return <div className={`${style.simple_error} ${className}`}>
        {error?.response ?
            <Copy text={wikiFormatter(formattedArr)}
                  internalTooltipId={'Error'}
                  className={style.copyText}>
                {'Скопировать для трекера'}
            </Copy>
            : null}
        {status
            ? <div className={style.status_title}>
                <h2>{status}</h2>
                {STATUSES_MAP[status]
                    ? <h3>{`: ${STATUSES_MAP[status]}`}</h3>
                    : null}
                <div>
                    {
                        ccpgf ?
                            <h2>Нет такой ручки в выбранном
                                окружении: {error?.response.data.error?.substr(ccpgf + endpointStringLength)}</h2>
                            : null
                    }
                </div>
                <div>
                    {highlightBackendCluster
                        ? <h2>Кластер бекенда: {error?.request?.options?.backend_cluster}</h2>
                        : null
                    }
                </div>
            </div>
            : null}
        {
            error?.message
                ? <div><strong>{data?.label || 'Сообщение'}</strong>: {error?.message}</div>
                : data?.label
                    ? <div><strong>{data?.label}</strong></div>
                    : null
        }
        <div className={style.data}>
            {SafeStringify(error?.response)}
        </div>
        {back
            ? <div className={style.back}>
                back: {back}
            </div>
            : null}
        {endPoint
            ? <div className={style.host}>
                endpoint: {endPoint}
            </div>
            : null}
        {
            special_info
                ? <div className={style.host}>
                    special_info: {
                        (typeof special_info === 'object'
                        && JSON.stringify(special_info)
                        || special_info.toString())
                    }
                </div>
                : null
        }
    </div>;
};
