import moment from 'moment';
import React from 'react';

import { ONE_SECOND } from '../../../constants';
import { UserInfoHandler } from '../../../models/user';
import FormatDate, { prettySeconds } from '../../../ui/FormatDate';
import { Link } from '../../../ui/Link';
import Select from '../../../ui/Select';
import * as styleTable from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { REQUESTS, SUPPORT_REQUESTS } from '../request';
import style from './index.css';

interface ICallsFeedState {
    isLoading: boolean;
    error: Error | null;
    users: any;
    tags: any[];
    selectedTag: string;
    options: any[];
}

const TEN = 10;
const FIFTEEN = 15;
const LIMIT = 600; //call duration is longer then 10minutes

export class CallsFeed extends React.Component<any, ICallsFeedState> {
    state: ICallsFeedState = {
        isLoading: false,
        error: null,
        users: {},
        tags: [],
        selectedTag: '',
        options: [],
    };
    request = new Request2({ requestConfigs: SUPPORT_REQUESTS });
    timeout;

    componentDidMount() {
        this.getData();
    }

    update() {
        clearTimeout(this.timeout);
        this.timeout = setTimeout(() => {
            this.getData();
        }, TEN * ONE_SECOND);
    }

    getData() {
        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.ARCHIVE, {
                queryParams: {
                    since: Math.floor(+moment().subtract(FIFTEEN, 'minutes') / ONE_SECOND),
                },
            }).then(response => {
                const tags = response?.tags?.filter(tag => tag?.call?.exit === 0) || [];
                const optionsSet = new Set();
                tags.forEach(el => {
                    optionsSet.add(el.tag_name);
                });
                const options = [...optionsSet].map((el: string) => ({ text: el, value: el }));
                const users = response?.users?.reduce((_p, _c) => {
                    if (!_p.hasOwnProperty(_c.id)) {
                        _p[_c.id] = _c;
                    }

                    return _p;
                }, {});
                this.setState(() => {
                    return {
                        error: null,
                        isLoading: false,
                        tags,
                        options,
                        users,
                    };
                }, () => {
                    this.update();
                });
            })
                .catch(error => {
                    this.setState({
                        isLoading: false,
                        error,
                    }, () => {
                        this.update();
                    });
                });
        });
    }

    componentWillUnmount() {
        this.request.abort();
        clearTimeout(this.timeout);
    }

    selectTag(selectedTag) {
        this.setState({
            selectedTag,
        });
    }

    render() {
        const { isLoading, error, users, tags, options, selectedTag } = this.state;

        return <div className={`${style.component}`}>
            <div>Всего звонков: {isLoading
                ? <Spin size={'s'}/>
                : <strong>{tags.length}</strong>} <span className={style.updated}>
                | обновлено: <FormatDate value={Date.now()} withSecond/></span></div>
            <div>
                <Select options={[{ text: 'Все очереди', value: '' }].concat(options)}
                        initialValues={[selectedTag]}
                        onSelect={this.selectTag.bind(this)}
                        placeholder={'очередь'}/>
            </div>
            <div>
                {error && <SimpleError error={error}/>}
                <table className={styleTable.table}>
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>длительность</th>
                            <th>очередь</th>
                            <th>оператор</th>
                            <th>клиент</th>
                        </tr>
                    </thead>
                    <tbody>
                        {
                            tags
                                ?.filter(el => {
                                    return !selectedTag || selectedTag === el.tag_name;
                                })
                                ?.map((tag, index) => {
                                    const performer_id = tag.tag_details?.performer;
                                    const client_id = tag.object_id;
                                    const duration = Date.now() / ONE_SECOND - tag.call?.connect;

                                    const client = UserInfoHandler.getPrintName.call(users[client_id]);

                                    return <tr key={index} className={`${duration >= LIMIT ? style.long_call : ''}`}>
                                        <td>{index + 1}</td>
                                        <td>
                                            <div>
                                                <strong>
                                                    {prettySeconds(duration)}
                                                </strong>
                                            </div>
                                            <div className={style.connect}>
                                                <FormatDate value={tag.call?.connect * ONE_SECOND} withSecond/>
                                            </div>
                                        </td>
                                        <td>
                                            <div className={style.status}>{tag.tag_details?.status}</div>
                                            <div>{tag.tag_name}</div>
                                        </td>
                                        <td>
                                            <Link href={`#/clients/${performer_id}/request_aggregator`}
                                                  target={'_blank'}>
                                                {users[performer_id]?.username || performer_id}
                                            </Link>
                                        </td>
                                        <td>
                                            <Link href={`#/clients/${client_id}/info`}
                                                  target={'_blank'}>{client || client_id}</Link>
                                        </td>
                                    </tr>;
                                })
                        }
                    </tbody>
                </table>
            </div>
        </div>;
    }
}
