import React, { useEffect, useState } from 'react';

import { useRequestHandler } from '../../../../hooks/useRequestHandler';
import { Button } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import { Input } from '../../../../ui/Input';
import { Request2 } from '../../../../utils/request';
import { deepCopy } from '../../../../utils/utils';
import { FormConstructor } from '../../../FormConstructor';
import { REQUESTS, SUPPORT_REQUESTS } from '../../request';
import {
    FEATURE_INITIAL,
    FEATURE_STANDARD_SCHEME,
    GVARS_PREFIX,
    MODEL_INITIAL,
    MODEL_STANDARD_SCHEME,
} from '../constants';
import style from '../index.css';
import { FormPreview } from './FormPreview';

export interface IOpenElementModalProps {
    onClose: () => void;
    entity: string;
    data: any;
    updateData: () => void;
}

const SCHEME_DISTRIBUTOR = {
    'models': MODEL_STANDARD_SCHEME,
    'features': FEATURE_STANDARD_SCHEME,
};

const INITIAL_DATA_DISTRIBUTOR = {
    'models': MODEL_INITIAL,
    'features': FEATURE_INITIAL,
};

export const OpenElementModal = React.memo((props: IOpenElementModalProps) => {
    const { onClose, entity, data, updateData } = props;

    const [formData, setFormData] = useState();
    const [isFormValid, setFormValid] = useState();
    const [newId, setNewId] = useState('');
    const [isAdding, setAdding] = useState(false);

    const onFormChange = (data, isValid) => {
        setFormData(data);
        setFormValid(isValid);
    };

    const request = new Request2({ requestConfigs: SUPPORT_REQUESTS });

    const getAddingOptions = React.useMemo(() => {
        const gvarCopy = deepCopy(data ?? {});

        gvarCopy[entity][newId] = formData;

        return {
            requestName: REQUESTS.UPSERT_SETTINGS,
            requestOptions: {
                body: {
                    settings: [
                        {
                            setting_key: GVARS_PREFIX,
                            setting_value: JSON.stringify(gvarCopy),
                        },
                    ],
                },
            },
        };
    }, [isAdding]);

    const onAddData = () => {
        setAdding(false);
        updateData();
        onClose();
    };

    const [
        isAddingLoading, ,
        addingError,
        addElement,
    ] = useRequestHandler<any[]>(request, getAddingOptions, onAddData.bind(null));

    const onSave = () => {
        setAdding(true);
        addElement();
    };

    useEffect(() => {
        return () => request.abort();
    }, []);

    return <Window onClose={onClose.bind(null)} title={'Добавить элемент'} collapseError={addingError}>
        <div className={style.add_wrapper}>
            <div>
                <FormConstructor schema={SCHEME_DISTRIBUTOR[entity]}
                                 hideChanges={true}
                                 initialData={INITIAL_DATA_DISTRIBUTOR[entity]}
                                 onChange={onFormChange.bind(null)}/>

                <Input value={newId}
                       onChange={setNewId.bind(null)}
                       required={true}
                       placeholder={'ID нового поля без пробелов'}/>

                <Button disabled={!isFormValid}
                        isLoading={isAddingLoading}
                        className={style.add_button}
                        onClick={onSave.bind(null)}>
                    Добавить
                </Button>
            </div>

            <div>
                <FormPreview entity={entity} item={formData}/>
            </div>
        </div>
    </Window>;
});
