import React, { useMemo, useState } from 'react';

import { Input } from '../../../../ui/Input';
import Select from '../../../../ui/Select';
import { deepCopy } from '../../../../utils/utils';
import { Copy } from '../../../Copy';
import { FormConstructor } from '../../../FormConstructor';
import { AVAILABLE_CITY_OPTIONS, INFO_ITEM_TITLES, INFO_ITEMS, OPTIONS_SCHEMES } from '../constants';
import style from '../index.css';

interface ICitiesFormProps {
    fullData: any;
    cityData: any;
    onChange: (data, isValid, wasChanged) => void;
}

export const CitiesForm = React.memo((props: ICitiesFormProps) => {
    const options = useMemo(() => {
        return props.cityData?.sections
            ?.filter(el => AVAILABLE_CITY_OPTIONS.includes(el.id))
            ?.map(el => {
                return {
                    value: el.id,
                    description: el.name ?? '',
                };
            });
    }, []);

    const [selectedOption, setSelectedOption] = useState<string>('');
    const [filter, setFilter] = useState<string>('');

    const selectedItem = useMemo(() => {
        return props.cityData?.sections?.find(el => el.id === selectedOption);
    }, [selectedOption]);

    const onFormChange = (newData, isValid, wasChanged) => {
        const data = deepCopy(props.cityData);
        const changedIndex = data?.sections?.findIndex(el => el.id === selectedOption);
        data.sections[changedIndex] = { ...data.sections[changedIndex], ...newData };

        props.onChange(data, isValid, wasChanged);
    };

    return <div className={style.cities_form}>
        <div>
            <Select options={options}
                    onSelect={setSelectedOption.bind(null)}
                    className={style.form_constructor}/>

            {selectedOption
                ? <FormConstructor schema={OPTIONS_SCHEMES[selectedOption ?? ''] ?? {}}
                                   initialData={selectedItem ?? {}}
                                   className={style.form_constructor}
                                   onChange={onFormChange.bind(null)}/>
                : null
            }
        </div>

        <div>
            {selectedOption && selectedOption !== 'promo'
                ? <div>
                    <Input value={filter}
                           onChange={setFilter.bind(null)}
                           placeholder={'Фильтр по значениям'}
                           className={style.filter}/>

                    {INFO_ITEMS.map(el => {
                        return <div className={style.info_item} key={el}>
                            <div className={style.title}>{INFO_ITEM_TITLES[el]}</div>

                            {Object.keys(props.fullData[el])?.map(item => {
                                if (!filter || item.includes(filter)) {
                                    return <div key={item} className={style.item}>
                                        <Copy text={item}>{item}</Copy>
                                    </div>;
                                }

                                return <span key={item}/>;

                            })}
                        </div>;
                    })}
                </div>
                : null
            }
        </div>
    </div>;
});
