import React, { useEffect, useState } from 'react';

import { useRequestHandler } from '../../../hooks/useRequestHandler';
import { Button, ButtonTypes } from '../../../ui/Button';
import { Collapse2 } from '../../../ui/Collapse2';
import Select, { IOptionInfo } from '../../../ui/Select';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Request2 } from '../../../utils/request';
import { deepCopy } from '../../../utils/utils';
import { FormConstructor } from '../../FormConstructor';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { REQUESTS, SUPPORT_REQUESTS } from '../request';
import { OpenElementModal } from './childComponents/AddElementModal';
import { CitiesForm } from './childComponents/CitiesForm';
import { DeleteElementModal } from './childComponents/DeleteElementModal';
import { FormBuilder } from './childComponents/FormBuilder';
import { FormPreview } from './childComponents/FormPreview';
import { availableEntityOptions, availableToAddOptions, GVARS_PREFIX } from './constants';
import style from './index.css';

export const EditorB2C = React.memo(() => {
    const request = new Request2({ requestConfigs: SUPPORT_REQUESTS });

    const [data, setData] = useState();
    const [entityOptions, setEntityOptions] = useState<IOptionInfo[]>([]);
    const [selectedEntity, setSelectedEntity] = useState<string>('');
    const [selectedInstance, setSelectedInstance] = useState<string>('');
    const [selectedItem, setSelectedItem] = useState<any>();
    const [itemCopy, setItemCopy] = useState<any>();
    const [formChanged, setFormChanged] = useState<boolean>(false);
    const [openAddModal, setOpenAddModal] = useState<boolean>(false);
    const [openDeleteModal, setOpenDeleteModal] = useState<boolean>(false);
    const [saving, setSaving] = useState<boolean>(false);

    const options = React.useMemo(() => {
        return {
            requestName: REQUESTS.GET_GLOBAL_SETTINGS,
            requestOptions: {
                queryParams: {
                    prefix: GVARS_PREFIX,
                },
            },
        };
    }, []);

    const responseHandler = (data) => {
        const stringValue = data.settings?.[0]?.setting_value;
        const jsonValue = isValidJSONString(stringValue) ? JSON.parse(stringValue) : null;

        jsonValue ? getEntityOptions(jsonValue) : null;
        setData(jsonValue);
        getEntityOptions(jsonValue);

        return data;
    };

    const getEntityOptions = (data) => {
        const allOptions = Object.keys(data);
        const allowedOptions = allOptions.filter(opt => availableEntityOptions.includes(opt));
        const entityOptions = allowedOptions.map(el => {
            return { value: el };
        });
        setEntityOptions(entityOptions);
    };

    const [isLoading, , error, getData] = useRequestHandler<any[]>(request, options, responseHandler);

    const instanceOptions = React.useMemo(() => {
        const instanceData = data?.[selectedEntity ?? ''] ?? {};
        const instanceEntries = Object.entries(instanceData);

        return instanceEntries.map((el: [string, any]) => {
            return {
                description: el[1]?.name ?? el[0],
                value: el[0],
            };
        });
    }, [selectedEntity]);

    const onFormChange = (newData, isValid, changed) => {
        setItemCopy(newData);
        setFormChanged(changed);
    };

    const setEntity = (entity) => {
        setSelectedEntity(entity);
        if (selectedInstance) {
            setSelectedItem(null);
            setItemCopy(null);
        }
    };

    const setInstance = (instance) => {
        const selectedItem = deepCopy(data?.[selectedEntity]?.[instance] ?? {});
        setSelectedInstance(instance);
        setSelectedItem(Object.keys(selectedItem)?.length ? selectedItem : null);
    };

    const onSaveButtonClick = () => {
        setSaving(true);
        updateData();
    };

    const getUpsertOptions = React.useMemo(() => {
        const gvarCopy = deepCopy(data ?? {});

        gvarCopy[selectedEntity]?.[selectedInstance]
            ? gvarCopy[selectedEntity][selectedInstance] = itemCopy
            : null;

        return {
            requestName: REQUESTS.UPSERT_SETTINGS,
            requestOptions: {
                body: {
                    settings: [
                        {
                            setting_key: GVARS_PREFIX,
                            setting_value: JSON.stringify(gvarCopy),
                        },
                    ],
                },
            },
        };
    }, [saving]);

    const onUpdateData = () => {
        setFormChanged(false);
        setSaving(false);
        getData();
    };

    const onDeleteData = () => {
        setOpenDeleteModal(false);
        setSelectedInstance('');
        setSelectedItem(null);
    };

    const [
        isUpdatingLoading, ,
        updatingError,
        updateData,
    ] = useRequestHandler<any[]>(request, getUpsertOptions, onUpdateData.bind(null));

    useEffect(() => {
        getData();

        return () => request.abort();
    }, []);

    const isCities = selectedEntity === 'cities';

    return <div>
        {error || updatingError
            ? <Collapse2 title={'Уупс, что-то пошло не так'}
                         children={<SimpleError error={error || updatingError}/>}
                         headerClassname={style.error_title}/>
            : null
        }

        {isLoading ? <Spin/> : null}

        {data
            ? <div className={`${style.wrapper} ${isCities ? style.two_rows : style.three_rows}`}>
                <div className={style.settings}>
                    <Select options={entityOptions}
                            onSelect={setEntity.bind(null)}
                            placeholder={'Что будем править?'}
                            className={style.entity_select}
                            disabled={Boolean(isLoading || error)}/>

                    <Select options={instanceOptions}
                            onSelect={setInstance.bind(null)}
                            placeholder={'Выберите экземпляр'}
                            className={style.instance_select}
                            disabled={Boolean(isLoading || error || !selectedEntity)}/>

                    {selectedEntity && availableToAddOptions.includes(selectedEntity)
                        ? <Button onClick={setOpenAddModal.bind(null, true)}
                                  className={style.add_button}>
                            Добавить элемент в {selectedEntity}
                        </Button>
                        : null
                    }

                    {selectedEntity && availableToAddOptions.includes(selectedEntity) && selectedInstance
                        ? <Button onClick={setOpenDeleteModal.bind(null, true)}
                                  className={style.add_button}
                                  colorType={ButtonTypes.negative}
                                  basic={true}>
                            Удалить элемент из {selectedEntity}
                        </Button>
                        : null
                    }
                </div>

                <div className={style.form}>
                    {selectedItem
                        ? <>
                            {isCities
                                ? <CitiesForm cityData={selectedItem}
                                              fullData={data}
                                              onChange={onFormChange.bind(null)}/>
                                : <FormConstructor schema={FormBuilder(selectedItem) ?? {}}
                                                   initialData={selectedItem ?? {}}
                                                   onChange={onFormChange.bind(null)}
                                                   className={style.form_constructor}/>
                            }

                            <Button colorType={ButtonTypes.positive}
                                    className={!isCities ? style.save_button : style.short_save_button}
                                    isLoading={isUpdatingLoading}
                                    disabled={!formChanged}
                                    onClick={onSaveButtonClick.bind(null)}>
                                Сохранить изменения
                            </Button>
                        </>
                        : null
                    }
                </div>

                <div className={style.preview}>
                    {selectedItem
                        ? <FormPreview entity={selectedEntity} item={itemCopy ?? selectedItem}/>
                        : null
                    }
                </div>
            </div>
            : null
        }

        {openAddModal
            ? <OpenElementModal onClose={setOpenAddModal.bind(null, false)}
                                entity={selectedEntity}
                                data={data}
                                updateData={getData.bind(null)}/>
            : null
        }

        {openDeleteModal
            ? <DeleteElementModal data={data}
                                  entity={selectedEntity}
                                  instance={selectedInstance}
                                  updateData={getData.bind(null)}
                                  onClose={onDeleteData.bind(null)}/>
            : null
        }
    </div>;
});
