import React from 'react';

import { INTERFACE_ADMIN_REQUESTS, REQUESTS } from '../../../../../decorators/InterfaceAdminProvider/request';
import { Button } from '../../../../ui/Button';
import { Confirm } from '../../../../ui/FullModal';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import LS from '../../../../utils/localStorage/localStorage';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import { controlType } from '../../../FormConstructor/types';
import { HOST } from '../../../Migrate/constants';
import style from './index.css';

const SCHEME = {
    'billing.session_debt_threshold_override': {
        'type': controlType.numeric,
        'display_name': 'при смене режимов',
        'description': `Возможный долг при смене режимов (в копейках️) session_debt_threshold_override`,
        'order': 1,
    },
    'billing.debt_threshold_on_perform': {
        'type': controlType.numeric,
        'display_name': 'за прошлые поездки',
        'description': `Возможный долг за прошлые поездки при бронировании (в копейках️) debt_threshold_on_perform`,
        'order': 1,
    },
    'billing.session_debt_threshold_deposit': {
        'type': controlType.numeric,
        'display_name': 'непогашенный депозит',
        'description': 'Возможный непогашенный депозит для тарифов с предсписанием (в копейках️) session_debt_threshold_deposit',
        'order': 1,
    },
};

const THRESHOLD_KEY = 'billing';

export class DebtThreshold extends React.Component {
    state = {
        isValid: false,
        isChanged: false,
        form: {},
        isLoading: true,
        initialData: {},
        confirmIsOpen: false,
        error: null,
    };

    request = new Request2({ requestConfigs: INTERFACE_ADMIN_REQUESTS });
    ls = new LS();

    componentDidMount() {
        this.getData();
    }

    getData() {
        this.request.exec(REQUESTS.GET_GLOBAL_SETTINGS, {
            queryParams: {
                prefix: THRESHOLD_KEY,
            },
            ...this.ls.envIsProdPre() && {
                headers: {
                    'saasnew': HOST.ADMIN,
                },
            },
        })
            .then(response => {

                const data = response.settings.filter(el => SCHEME[el.setting_key]);
                const initialData = Object.entries(data).reduce((_p, _c: any[]) => {
                    _p[_c[1].setting_key] = _c[1].setting_value;

                    return _p;
                }, {});

                this.setState({
                    isLoading: false,
                    initialData,
                });

            })
            .catch(error => {
                this.setState({
                    error,
                    isLoading: false,
                });
            });
    }

    onFormChange(form, isValid, isChanged) {
        this.setState({
            form,
            isValid,
            isChanged,
        });
    }

    shouldComponentUpdate(nextProps: Readonly<{}>, nextState: Readonly<{}>, nextContext: any): boolean {
        return !isObjectEqual(this.props, nextProps) || !isObjectEqual(this.state, nextState);
    }

    openConfirm(confirmIsOpen) {
        this.setState({
            confirmIsOpen,
        });
    }

    update() {
        this.setState({
            isLoading: true,
        }, () => {
            let updateDataRequest;
            const prop = (host?: string) => ({
                body: {
                    settings: Object.entries(this.state.form).map(el => {
                        return {
                            setting_key: el[0],
                            setting_value: el[1],
                        };
                    }),
                },
                ...host && {
                    headers: {
                        'saasnew': host,
                    },
                },
            });

            if (this.ls.envIsProdPre()) {
                updateDataRequest = Promise.all([
                    this.request.exec(REQUESTS.UPSERT_SETTINGS, prop(HOST.PRESTABLE)),
                    this.request.exec(REQUESTS.UPSERT_SETTINGS, prop(HOST.ADMIN)),
                ]);
            } else {
                updateDataRequest = this.request.exec(REQUESTS.UPSERT_SETTINGS, prop());
            }

            updateDataRequest.then(() => {
                this.setState({
                    isLoading: false,
                    confirmIsOpen: false,
                });
            })
                .catch(error => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });

    }

    render() {
        const json_space = 2;
        const question = `Обновить пороги?` + `<br><pre>${JSON.stringify(this.state.form, null, json_space)}</pre>`;

        return <div className={style.component}>
            {
                this.state.confirmIsOpen
                    ? <Confirm error={this.state.error}
                               accept={this.update.bind(this)}
                               isWorking={this.state.isLoading}
                               question={question}
                               onClose={this.openConfirm.bind(this, false)}/>
                    : null
            }
            <FormConstructor schema={SCHEME}
                             className={style.form}
                             initialData={this.state.initialData}
                             onChange={this.onFormChange.bind(this)}/>
            <Button onClick={this.openConfirm.bind(this, true)}
                    disabled={!this.state.isValid || !this.state.isChanged}
                    isLoading={this.state.isLoading}>Обновить</Button>
        </div>;
    }
}
