import * as React from 'react';

import { Button } from '../../../../ui/Button';
import { FileListView } from '../../../../ui/FileListView';
import { Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Request2 } from '../../../../utils/request';
import { generateuuid4 } from '../../../../utils/utils';
import TagModal from '../../../TagModal';
import { OBJECT_TYPES } from '../../../TagModal/component';
import { INCIDENT_REQUESTS, REQUESTS } from '../request';
import { IIncident } from '../types';
import { IncidentAttachPhoto } from './IncidentAttachPhoto';

interface IRegisterPhotoModalProps {
    onClose: (success: boolean, closeAll?: boolean) => void;
    carId?: string | null;
    incidentTask?: string;
    incident?: IIncident;
}

interface IRegisterPhotoModalState {
    isLoading: boolean;
    loadingError: Error | null;
    files: File[];
    isCreatingTagModalOpen: boolean;
    incidentAttachTags: null | Record<string, any>[];
    photoIds: string[];
    tagWasCreated: boolean;
}

export class RegisterPhotoModal extends React.Component<IRegisterPhotoModalProps, IRegisterPhotoModalState> {
    state: IRegisterPhotoModalState = {
        isLoading: false,
        loadingError: null,
        files: [],
        isCreatingTagModalOpen: false,
        incidentAttachTags: null,
        photoIds: [],
        tagWasCreated: false,
    };
    request = new Request2({ requestConfigs: INCIDENT_REQUESTS });

    getCarTags() {
        this.setState({ isLoading: true, loadingError: null }, () => {
            this.request.exec(REQUESTS.GET_CAR_TAGS, {
                queryParams: {
                    car_id: this.props.carId ?? '',
                },
            })
                .then(response => {
                    const allTags = response.records ?? [];
                    const dtpTags = allTags.filter(tag => tag.tag.toLowerCase().includes('dtp'));

                    if (dtpTags.length) {
                        this.registerPhoto(dtpTags);
                    } else {
                        this.setState({
                            isCreatingTagModalOpen: true,
                            isLoading: false,
                        });
                    }
                })
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });

        });
    }

    afterTagCreated(data, response) {
        this.setState({ isCreatingTagModalOpen: false, tagWasCreated: true });
        const tag_id = response?.tagged_objects?.[0]?.tag_id?.[0];
        this.registerPhoto([{ tag_id }]);
    }

    registerPhoto(dtpTags: Record<string, any>[]) {
        const { carId, incident } = this.props;
        const { files } = this.state;

        this.setState({ isLoading: true, loadingError: null }, () => {
            const photoIds = files.map(() => generateuuid4());

            const photos = files.map((file, index) => {
                return {
                    marker: `incident/${incident ? incident.incident_id : 'incident_id'}`,
                    uuid: photoIds[index],
                    md5: '',
                    origin: 'incident',
                    meta_data: {},
                };
            });

            const body = {
                tag_ids: dtpTags.map(dtpTag => dtpTag.tag_id),
                photos,
            };

            this.request.exec(REQUESTS.REGISTER_INCIDENT_PHOTO, { body })
                .then((response) => {
                    const images = response?.data?.images ?? [];
                    const uploadedImgIds = images.map(img => img.image_id);

                    const filesUploadRequests = files.map((file: File, index) => {
                        const type = file.type;
                        const reader = new FileReader();
                        reader.readAsArrayBuffer(file);
                        reader.onload = () => {
                            return this.request.exec(REQUESTS.UPLOAD_INCIDENT_PHOTO, {
                                queryParams: {
                                    car_id: carId ?? '',
                                    photo_id: photoIds[index],
                                },
                                file: reader.result,
                                headers: {
                                    'Content-Type': type,
                                },
                            });
                        };
                    });

                    Promise.all(filesUploadRequests)
                        .then(() => {
                            this.setState({
                                incidentAttachTags: dtpTags,
                                photoIds: uploadedImgIds,
                            }, () => {
                                !dtpTags?.length ? this.props.onClose(true) : null;
                                this.setState({ isLoading: false });
                            });
                        })
                        .catch(loadingError => {
                            this.setState({
                                loadingError,
                                isLoading: false,
                            });
                        });
                })
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });

        });
    }

    componentWillUnmount() {
        this.request.abort();
    }

    openFiles() {
        const file: HTMLInputElement = document.createElement('input');
        file.type = 'file';
        file.multiple = true;
        file.style.display = 'none';
        document.body.appendChild(file);
        file.click();
        file.onchange = (e: any) => {
            e.stopPropagation();
            e.preventDefault();
            const files = e.target.files;
            const renamedFiles = [...files].map(file => {
                const blob = file.slice(0, file.size, file.type);

                return new File([blob], generateuuid4(), { type: file.type });
            });

            this.setState({ files: renamedFiles });
        };
    }

    removeItem(index: number) {
        const files = [...this.state.files];
        files.splice(index, 1);
        this.setState({ files });
    }

    filterTags(tags) {
        return tags?.filter(el => el.name.includes('dtp_flow_photo'));
    }

    closeAttach() {
        this.setState({ incidentAttachTags: null });
        this.props.onClose(true, true);
    }

    render() {
        const { onClose, incidentTask, incident } = this.props;
        const {
            loadingError,
            isLoading,
            files,
            isCreatingTagModalOpen,
            incidentAttachTags,
            photoIds,
            tagWasCreated,
        } = this.state;

        return <Window error={loadingError} onClose={onClose.bind(this)} title={'Загрузка фото'}>
            <Button basic disabled={isLoading} onClick={this.openFiles.bind(this)}>
                Выбрать
            </Button>

            {files?.length
                ? <FileListView files={files} removeItem={this.removeItem.bind(this)}/>
                : <h4>Фото не выбраны</h4>
            }

            <div className={coreStyle.button_container}>
                <Button disabled={isLoading || !files?.length}
                        onClick={this.getCarTags.bind(this)}>
                    Загрузить
                </Button>
            </div>

            {isCreatingTagModalOpen
                ? <TagModal objectId={{ type: OBJECT_TYPES.CAR, id: this.props.carId ?? '' }}
                            onClose={() => this.setState({ isCreatingTagModalOpen: false })}
                            filterTags={this.filterTags.bind(this)}
                            initialData={{ comment: incidentTask ?? '' }}
                            callback={this.afterTagCreated.bind(this)}
                            title={'Создайте тег для прикрепления фото'}/>
                : null
            }

            {incidentAttachTags?.length
                ? <IncidentAttachPhoto onClose={this.closeAttach.bind(this)}
                                       tags={incidentAttachTags}
                                       photoIds={photoIds}
                                       tagWasCreated={tagWasCreated}
                                       incident_id={incident?.incident_id ?? ""}/>
                : null
            }
        </Window>;
    }
}
