import * as React from 'react';

import { Dict } from '../../../../../types';
import { ONE_SECOND } from '../../../../constants';
import { Button } from '../../../../ui/Button';
import DatePicker from '../../../../ui/DatePicker';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import { PictureBlocksViewer } from '../../../../ui/PictureBlocksViewer';
import { Request2 } from '../../../../utils/request';
import { deepCopy } from '../../../../utils/utils';
import Spin from '../../../Spin';
import PerformModal from '../PerformModal/component';
import { INCIDENT_REQUESTS, REQUESTS } from '../request';
import { IIncident } from '../types';
import * as style from './index.css';
import { RegisterPhotoModal } from './RegisterPhotoModal';

interface IAllIncidentPhotoModalProps {
    onClose: (success: boolean) => void;
    incident?: IIncident;
    carId: string | null;
}

interface IAllIncidentPhotoModalState {
    isLoading: boolean;
    loadingError: Error | null;
    allImages: Dict<any>;
    filters: { since: number | null; until: number | null };
    isPerformModalOpen: boolean;
    initialData: Dict<any> | null;
    attachingImageId: number | null;
    showAttachSuccess: boolean;
    isOpenLoadPhotoModal: boolean;
}

export default class AllIncidentPhotoModal extends
    React.Component<IAllIncidentPhotoModalProps, IAllIncidentPhotoModalState> {
    state: IAllIncidentPhotoModalState = {
        isLoading: false,
        loadingError: null,
        allImages: [],
        filters: { since: null, until: null },
        isPerformModalOpen: false,
        initialData: null,
        attachingImageId: null,
        showAttachSuccess: false,
        isOpenLoadPhotoModal: false,
    };
    request = new Request2({ requestConfigs: INCIDENT_REQUESTS });

    componentDidMount(): void {
        this.getPhotos();
    }

    componentWillUnmount() {
        this.request.abort();
    }

    getPhotos() {
        const { carId } = this.props;
        const { filters } = this.state;
        const { since, until } = filters;
        this.setState({ isLoading: true, loadingError: null }, () => {

            const queryParams = {
                car_id: carId,
                since, until,
            };

            this.request.exec(REQUESTS.GET_PHOTOS, { queryParams })
                .then((response => {

                    const imagesDescArray: Dict<any>[] = response?.images ? Object.values(response.images) : [];

                    const allImages: Dict<any>[] = imagesDescArray.reduce((result: any, value: any) => {

                        let imagesArr = value.map(valueItem => {
                            const { history_user_id, tag, timestamp, images } = valueItem;

                            return images.map(image => {
                                image.history_user_id = history_user_id;
                                image.tag = tag;
                                image.timestamp = timestamp;

                                return image;
                            });
                        });

                        imagesArr = imagesArr.reduce((result, image) => {
                            result.push(...image);

                            return result;
                        }, []);

                        result.push(...imagesArr);

                        return result;
                    }, []);

                    this.setState({ isLoading: false, allImages });
                }))
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    onFilterChange(type: string, value) {
        const filters = deepCopy(this.state.filters);
        filters[type] = value;
        this.setState({ filters });
    }

    onAttachPhotoClick(image: Dict<any>) {
        const { incident } = this.props;

        const initialData = {
            incident_id: incident?.incident_id,
            transition: {
                transition_id: 'handle_attached_photos',
                actions: [
                    {
                        action_type: {
                            action_type: 'upsert',
                            image_id: image.image_id,
                        },
                    },
                ],
            },
        };

        this.setState({ initialData, attachingImageId: image.image_id }, () => {
            this.openPerformModal();
        });
    }

    openPerformModal() {
        this.setState({ isPerformModalOpen: true });
    }

    closePerformModal(success: boolean) {
        const SHOW_MESSAGE_TIMEOUT = 3000;

        if (success) {
            this.setState({ initialData: null, showAttachSuccess: true, isPerformModalOpen: false }, () => {
                setTimeout(() => {
                    this.setState({ attachingImageId: null, showAttachSuccess: false });
                }, SHOW_MESSAGE_TIMEOUT);
            });
        } else {
            this.setState({ initialData: null, attachingImageId: null, isPerformModalOpen: false });
        }
    }

    openLoadPhotoModal() {
        this.setState({ isOpenLoadPhotoModal: true });
    }

    closeLoadPhotoModal(success = false, closeAll = false) {
        this.setState({ isOpenLoadPhotoModal: false }, () => {
            if (success && !closeAll) {
                this.getPhotos();
            } else if (closeAll) {
                this.props.onClose(true);
            }
        });
    }

    render() {
        const { onClose, incident, carId } = this.props;
        const {
            isLoading, loadingError, allImages, filters, isPerformModalOpen,
            initialData, showAttachSuccess, attachingImageId, isOpenLoadPhotoModal,
        } = this.state;
        const { since, until } = filters;

        const secIncident = incident?.links?.find(el => el.ticket_key?.includes('DRIVESECURITY'));

        return <Window error={loadingError}
                       title={'Фото по машине'}
                       className={style.photo_modal}
                       onClose={onClose.bind(this)}>
            <div className={style.controls}>
                <DatePicker placeholder={'С'}
                            value={since}
                            onChange={this.onFilterChange.bind(this, 'since')}
                            className={style.date_picker}/>
                <DatePicker placeholder={'По'}
                            value={until}
                            onChange={this.onFilterChange.bind(this, 'until')}
                            className={style.date_picker}/>
                <Button basic onClick={this.getPhotos.bind(this)}>Поиск</Button>
                <Link className={style.update_link} onClick={this.getPhotos.bind(this)}>Обновить</Link>
            </div>
            {isLoading
                ? <Spin size={'l'}/>
                : <PictureBlocksViewer pictures={allImages.map(image => {
                    const { url, timestamp, history_user_id, image_id } = image;

                    const info = <div className={style.all_image_info}>
                        <div>
                            <FormatDate value={timestamp * ONE_SECOND}/>
                            <div>
                                <Link href={`#/clients/${history_user_id}/info`} target={'_blank'}>Автор</Link>
                            </div>
                            {showAttachSuccess && attachingImageId === image_id
                                ? <div className={style.success_attach}>Фото прикреплено</div>
                                : null}
                        </div>
                        <div>
                            {incident
                                ? <div>
                                    <Link onClick={this.onAttachPhotoClick.bind(this, image)}>Прикрепить</Link>
                                </div>
                                : null}
                            <div>
                                {image_id}
                            </div>
                        </div>
                    </div>;

                    return { link: url, info: info };
                })}/>}
            <div className={style.load_photo_link}>
                <Link onClick={this.openLoadPhotoModal.bind(this)}>Загрузить фото</Link>
            </div>
            {isPerformModalOpen && incident
                ? <PerformModal onClose={this.closePerformModal.bind(this)}
                                incident={incident}
                                initialData={initialData}/>
                : null}
            {
                isOpenLoadPhotoModal
                    ? <RegisterPhotoModal carId={carId}
                                          onClose={this.closeLoadPhotoModal.bind(this)}
                                          incident={incident}
                                          incidentTask={secIncident?.ticket_key}/>
                    : null
            }
        </Window>;
    }
}
