import * as React from 'react';

import { Dict } from '../../../../../types';
import { CancelButton, SaveButton } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Request2 } from '../../../../utils/request';
import { CAR_INCIDENT_REQUESTS } from '../../../CarCard/CarIncidents/request';
import { FormConstructor } from '../../../FormConstructor';
import Spin from '../../../Spin';
import { REQUESTS } from '../request';

interface ICreateIncidentModalProps {
    onClose: (success: boolean) => void;
}

interface ICreateIncidentModalState {
    isLoading: boolean;
    loadingError: Error | null;
    isSaveLoading: boolean;
    savingError: Error | null;
    schema: Dict<any> | null;
    formData: Dict<any>;
    isFormValid: boolean;
}

export default class CreateIncidentModal extends React.Component<ICreateIncidentModalProps, ICreateIncidentModalState> {
    state: ICreateIncidentModalState = {
        isLoading: false,
        loadingError: null,
        isSaveLoading: false,
        savingError: null,
        schema: null,
        formData: {},
        isFormValid: false,
    };
    request = new Request2({ requestConfigs: CAR_INCIDENT_REQUESTS });

    componentDidMount(): void {
        this.setState({ isLoading: true, loadingError: null }, () => {
            this.request.exec(REQUESTS.GET_CREATE_INCIDENTS_SCHEMA, { queryParams: { scheme: true } })
                .then((response => {
                    this.setState({ isLoading: false, schema: response?.request_data?.structure ?? null });
                }))
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    componentWillUnmount() {
        this.request.abort();
    }

    onFormChange(formData: Dict<any>, isFormValid: boolean) {
        this.setState({ formData, isFormValid });
    }

    onSaveClick() {
        const { formData } = this.state;

        this.setState({ isSaveLoading: true, savingError: null }, () => {
            this.request.exec(REQUESTS.CREATE_INCIDENT, { body: formData })
                .then(() => {
                    this.setState({ isSaveLoading: false });
                    this.props.onClose(true);
                })
                .catch(savingError => {
                    this.setState({ isSaveLoading: false, savingError });
                });
        });
    }

    render() {
        const { onClose } = this.props;
        const { loadingError, isLoading, schema, isSaveLoading, savingError, isFormValid } = this.state;

        return <Window title={'Создание инцидента'} error={loadingError || savingError} onClose={onClose.bind(this)}>
            {isLoading
                ? <Spin/>
                : <>
                    <FormConstructor schema={schema ?? {}}
                                     onChange={this.onFormChange.bind(this)}/>
                    <div className={coreStyle.button_container}>
                        <CancelButton onClick={onClose.bind(this, false)}/>
                        <SaveButton isLoading={isSaveLoading}
                                    disabled={!isFormValid}
                                    onClick={this.onSaveClick.bind(this)}/>
                    </div>
                </>}
        </Window>;
    }
}
