import * as React from 'react';

import { ONE_SECOND } from '../../../../constants';
import { CAR_STATUS, CarInfoHandler } from '../../../../models/car';
import { SessionHistoryInfoHandler } from '../../../../models/session';
import { UserInfoHandler } from '../../../../models/user';
import { Button, ButtonTypes } from '../../../../ui/Button';
import { FormatDateInString } from '../../../../ui/FormatDate';
import { Confirm, Window } from '../../../../ui/FullModal';
import { isValidJSONString } from '../../../../utils/isValidJSONString';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import { SimpleError } from '../../../SimpleError';
import { CenteredSpin } from '../../../Spin';
import style from '../index.css';
import { INCIDENT_REQUESTS, REQUESTS } from '../request';
import { IIncident } from '../types';

interface IEvaInformModalProps {
    onClose: (success?: boolean) => void;
    incident: IIncident | null;
    constantsIFaceTag: (ids: string) => Promise<{ iface_tag_implementations: Record<string, any> }>;
}

interface IEvaInformModalState {
    loadingError: Error | null;
    tagModalOpen: boolean;
    initialData: Record<string, any>;
    pricesList: Record<string, any>;
    formData: Record<string, any>;
    isLoading: boolean;
    disableCloseModal: boolean;
    iface_tag_implementations: Record<string, any>;
    schema: Record<string, any>;
    isEmailTagAttaching: boolean;
    isFineTagAttaching: boolean;
    emailTagError: Error | null;
    fineTagError: Error | null;
    isFormValid: boolean;
    isWarningOpen: boolean;
    isPorsche: boolean;
}

const tagDistricts = {
    moscow: 'fine_eva_msc',
    moscow_region: 'fine_eva_msc_obl',
    spb: 'fine_eva_spb',
    spb_region: 'fine_eva_spb_obl',
    kzn: 'fine_eva_kazan',
    sochi: 'fine_eva_sochi',
    porsche: 'fine_eva_porsche',
};
const gosDepartments = {
    moscow: 'МАДИ',
    other: 'ГИБДД',
};
const OPTIONAL_FIELDS = ['decree_number', 'drunk', 'foreign_license', 'case'];
const prefix = 'incident.evacuation.amount_to_pay_by_district';
const EMAIL_TAG = 'email_evacuation';
const EMAIL_CONST_TAG = 'user_mail_notification_tag';
const FINE_CONST_TAG = 'fixed_sum_tag';

export default class EvacInformModal extends React.Component<IEvaInformModalProps, IEvaInformModalState> {
    state: IEvaInformModalState = {
        loadingError: null,
        tagModalOpen: false,
        initialData: {},
        pricesList: {},
        formData: {},
        isLoading: false,
        disableCloseModal: false,
        iface_tag_implementations: {},
        schema: {},
        isEmailTagAttaching: false,
        isFineTagAttaching: false,
        emailTagError: null,
        fineTagError: null,
        isFormValid: true,
        isWarningOpen: false,
        isPorsche: false,
    };
    request = new Request2({ requestConfigs: INCIDENT_REQUESTS });

    componentDidMount() {
        this.getTagImplementations(() => this.getInitialData());
    }

    componentWillUnmount() {
        this.request.abort();
    }

    getTagImplementations(callback?: () => void) {
        this.setState({
            isLoading: true,
        }, () => {
            const ids = `${EMAIL_CONST_TAG},${FINE_CONST_TAG}`;

            this.props.constantsIFaceTag(ids)
                .then((response) => {
                    const schemaList = response.iface_tag_implementations || {};
                    this.setState({
                        iface_tag_implementations: {
                            [EMAIL_TAG]: schemaList[EMAIL_TAG],
                            [FINE_CONST_TAG]: schemaList[FINE_CONST_TAG],
                        },
                        schema: {
                            ...this.setSchema(schemaList[EMAIL_TAG]),
                            is_plus_user: schemaList[FINE_CONST_TAG]?.is_plus_user,
                        },
                        isLoading: false,
                    }, () => {
                        callback?.();
                    });
                })
                .catch((loadingError) => {
                    this.setState({
                        loadingError,
                        isLoading: false,
                    });
                });
        });
    }

    setSchema(schema: Record<string, any>) {
        const structure = schema?.template_args?.structure || {};
        const newSchema = Object.keys(structure).reduce((prev, key) => {
            prev[key] = { ...structure[key] };
            if (!OPTIONAL_FIELDS.includes(key)) {
                prev[key].required = true;
            }

            if (key === 'evac_town' || key === 'fine_amount') {
                prev[key].read_only = true;
            }

            return prev;
        }, {});

        return { ...newSchema, session_id: { ...schema?.session_id, read_only: true } };
    }

    getInitialData() {
        const { incident } = this.props;
        const { userInfo, contexts, car_id, session_id } = incident as IIncident;

        this.setState({ isLoading: true, loadingError: null }, () => {
            Promise.all([
                this.request.exec(REQUESTS.GET_CAR_INFO, { queryParams: { car_id } }),
                this.request.exec(REQUESTS.GET_INCIDENTS_SETTINGS, { queryParams: { prefix } }),
                this.request.exec(REQUESTS.GET_SESSION, { queryParams: { session_id } }),
            ])
                .then(response => {
                    const [car, incidentInfo, session] = response;
                    const pricesList = isValidJSONString(incidentInfo?.settings?.[0]?.setting_value)
                        ? JSON.parse(incidentInfo?.settings?.[0]?.setting_value)
                        : {};
                    const district = contexts?.[0]?.evacuation_district;
                    const evac_date_time = FormatDateInString({
                        value: contexts?.[0]?.evacuation_instant * ONE_SECOND,
                    });

                    const initialData = {
                        session_id,
                        user_name: UserInfoHandler.getName.call(userInfo),
                        finish_time: this.getInitialDates(session),
                        car_model: CarInfoHandler.getModelName.call(car),
                        car_number: CarInfoHandler.getNumber.call(car),
                        geocoded_finish: contexts?.[0]?.evacuation_address,
                        fine_amount: pricesList[district].toString(),
                        sts_number: CarInfoHandler.getSts.call(car),
                        evac_town: district,
                        gos_department: gosDepartments[district] || gosDepartments.other,
                        evac_date_time,
                        is_plus_user: false,
                    };
                    this.setState({
                        tagModalOpen: true,
                        isLoading: false,
                        initialData,
                        pricesList,
                        formData: initialData,
                    });
                })
                .catch((loadingError) => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    getInitialDates(session) {
        let last_ts = SessionHistoryInfoHandler.getFinish.call(session);

        if (!SessionHistoryInfoHandler.isSessionCompiled.call(session)) {
            const events = SessionHistoryInfoHandler.getEvents.call(session);
            const index = events.findLastIndex((el) => el.tag_name === CAR_STATUS.old_state_riding);
            last_ts = index !== -1 ? events[index + 1]?.timestamp * ONE_SECOND : undefined;
        }

        return FormatDateInString({ value: last_ts });
    }

    getInfo(tag: string) {
        const { formData, iface_tag_implementations: schema, isPorsche } = this.state;
        const { incident } = this.props;
        const data: Record<string, any> = {
            session_id: incident?.session_id,
            object_id: incident?.user_id,
            priority: 0,
        };

        if (tag === EMAIL_TAG) {
            data.attachments = [];
            data.tag = EMAIL_TAG;
            data.template_args = {};
            Object.keys(formData).map((key) => {
                if (schema?.[tag]?.template_args?.structure?.[key]) {
                    data.template_args[key] = formData[key];
                }
            });
        } else {
            data.car_number = formData?.car_number;
            data.is_plus_user = formData?.is_plus_user;
            data.links = [{
                uri: incident?.links?.find((el) => el.type === 'startrek_ticket')?.ticket_url ?? '',
                type: 'st',
            }];
            data.tag = isPorsche
                ? tagDistricts.porsche
                : tagDistricts[incident?.contexts?.[0]?.evacuation_district];
        }

        return {
            object_id: incident?.user_id,
            data,
        };
    }

    attachEmailTag(callback?: () => void) {
        this.setState({ isEmailTagAttaching: true, disableCloseModal: true, emailTagError: null }, () => {
            const { object_id, data } = this.getInfo(EMAIL_TAG);

            this.request.exec(REQUESTS.ATTACH_USER_TAG, { queryParams: { object_id }, body: data })
                .then(response => {
                    this.setState({ isEmailTagAttaching: false }, () => {
                        callback?.();
                    });
                })
                .catch(emailTagError => {
                    this.setState({ emailTagError, isEmailTagAttaching: false, disableCloseModal: false });
                });
        });
    }

    attachFineTag() {
        this.setState({ isFineTagAttaching: true, fineTagError: null }, () => {
            const { object_id, data } = this.getInfo(FINE_CONST_TAG);

            this.request.exec(REQUESTS.ATTACH_USER_TAG, { queryParams: { object_id }, body: data })
                .then(response => {
                    this.setState({ isFineTagAttaching: false, disableCloseModal: false }, () => {
                        this.props.onClose();
                    });
                })
                .catch(fineTagError => {
                    this.setState({ fineTagError, isFineTagAttaching: false });
                });
        });
    }

    attachTags() {
        const { emailTagError, fineTagError } = this.state;
        if (fineTagError && !emailTagError) {
            this.attachFineTag();
        } else {
            this.attachEmailTag(() => this.attachFineTag());
        }
    }

    onChangeForm(formData: Record<string, any>, isFormValid: boolean) {
        const { pricesList } = this.state;
        const porschePrice = pricesList.porsche.toString();
        const isPorsche = !!formData.case;

        if (isPorsche && formData.fine_amount !== porschePrice) {
            this.setState({ initialData: { ...formData, fine_amount: porschePrice } });
        } else if (!isPorsche && formData.fine_amount === porschePrice) {
            this.setState({ initialData: {
                ...formData,
                fine_amount: pricesList[formData.evac_town]?.toString(),
            } });
        }

        this.setState({ formData, isFormValid, isPorsche });
    }

    closeModal() {
        const { onClose } = this.props;
        this.setState({ isWarningOpen: this.state.disableCloseModal }, () => {
            !this.state.isWarningOpen && onClose && onClose();
        });
    }

    closeWarning() {
        this.setState({ isWarningOpen: false });
    }

    accept() {
        const { onClose } = this.props;
        this.setState({ isWarningOpen: false, disableCloseModal: false }, () => {
            onClose && onClose();
        });
    }

    render() {
        const {
            loadingError, initialData, isLoading, schema, isFineTagAttaching,
            isEmailTagAttaching, emailTagError, fineTagError, isFormValid, isWarningOpen,
        } = this.state;
        const hasError = emailTagError || fineTagError;

        return <Window error={loadingError} onClose={this.closeModal.bind(this)} title={'Эвакуация — оповестить и списать'}>
            {isLoading
                ? <CenteredSpin/>
                : <FormConstructor schema={schema}
                                   initialData={initialData}
                                   onChange={this.onChangeForm.bind(this)}/>}
            {hasError && <SimpleError error={emailTagError || fineTagError}/>}
            {isWarningOpen && <Confirm onClose={this.closeWarning.bind(this)}
                                       question={<span>Если вы закроете окно сейчас, введенные данные не сохранятся.
                                           <br/>Вы уверены, что хотите закрыть окно?</span>}
                                       error={null}
                                       accept={this.accept.bind(this)}/>}
            <div className={style.button_container}>
                <Button onClick={this.attachTags.bind(this)}
                        disabled={!isFormValid}
                        colorType={hasError ? ButtonTypes.warning : undefined}>
                    {isEmailTagAttaching || isFineTagAttaching
                        ? <CenteredSpin/>
                        : hasError ? 'Попробовать еще раз' : 'Оповестить и списать'}
                </Button>
            </div>
        </Window>;
    }
}
