import React from 'react';

import { Dict } from '../../../../../types';
import { ONE_SECOND } from '../../../../constants';
import { Button } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import { Copy } from '../../../Copy';
import { FormConstructor } from '../../../FormConstructor';
import { controlType, ISchemaItem } from '../../../FormConstructor/types';
import { IIncident, INCIDENT_LINK } from '../types';

interface IIncidentToTelegramProps {
    onClose: () => void;
    incident: IIncident;
}

interface IIncidentToTelegramState {
    schema: Dict<ISchemaItem>;
    data: any;
}

export class IncidentToTelegram extends React.Component<IIncidentToTelegramProps, IIncidentToTelegramState> {
    state = {
        schema: {} as Dict<ISchemaItem>,
        data: {},
    };

    componentDidMount() {
        this.buildSchema();
    }

    getTicket() {
        return this.props.incident?.links?.find(link => {
            return link.type === INCIDENT_LINK.STARTREK_TICKET && link.ticket_key;
        }) || {};
    }

    getCarInfo() {
        const carInfo = { number: null, model: null, vin: null };
        carInfo.number = this.props.incident?.carInfo?.number;
        carInfo.model = this.props.incident?.carInfo?.modelInfo?.name;

        this.props.incident?.contexts?.forEach(el => {
            if (el.hasOwnProperty('vehicle_info')) {
                carInfo.vin = el?.vehicle_info?.vin;
            }
        });

        return carInfo;
    }

    getFIO() {
        const info = this.props.incident?.userInfo || {};

        return `${info.last_name || ''} ${info.first_name || ''} ${info?.pn || ''}`;
    }

    getAddressObj() {
        const obj = {
            addr: null,
        };
        this.props.incident?.contexts.forEach(el => {
            if (el.hasOwnProperty('evacuation_address')) {
                obj.addr = el.evacuation_address;
            }

            if (el.hasOwnProperty('incident_address')) {
                obj.addr = el.incident_address;
            }
        });

        return obj;
    }

    getComment() {
        let comment = '';
        this.props.incident?.contexts.forEach(el => {
            if (el.hasOwnProperty('comment')) {
                comment = el.comment;
            }
        });

        return comment;
    }

    addItemToSchema(schema, key, __description) {
        schema[__description] = {
            'type': controlType.bool,
            'display_name': `${__description} ${key}`,
            default: true,
        };
    }

    buildSchema() {
        const schema: any = {};
        const ticket: any = this.getTicket();

        if (ticket.ticket_url) {
            this.addItemToSchema(schema, ticket?.ticket_url, '✔️ Таска: ');
        }

        const carInfo: any = this.getCarInfo();
        if (carInfo.number) {
            this.addItemToSchema(
                schema,
                `${carInfo?.number}; ${carInfo?.model}${carInfo?.vin ? `; VIN:${carInfo?.vin}` : ''}`,
                '🚗 Авто: ',
            );
        }

        const fio = this.getFIO();
        if (fio.trim()) {
            this.addItemToSchema(schema, fio, '👤 ФИО: ');
        }

        const addressObj: any = this.getAddressObj();
        if (addressObj.addr) {
            this.addItemToSchema(schema, addressObj.addr, '📍 Адрес: ');
        }

        if (this.props.incident?.initiated_at) {
            this.addItemToSchema(schema, `${new Date(this.props.incident?.initiated_at * ONE_SECOND)}`, '⏱ Время: ');
        }

        const comment = this.getComment();
        if (comment) {
            this.addItemToSchema(schema, comment, '🤫 Комментарий: ');
        }

        this.setState({
            schema,
        });
    }

    updateData(data) {
        this.setState({
            data,
        });
    }

    buildText() {
        let text = `${this.props.incident?.incident_type}:\n`;
        Object.entries(this.state.data).forEach(el => {
            if (el[1]) {
                const _: any = this.state.schema[el[0]];
                text += `${_.display_name}\n`;
            }
        });

        return text;
    }

    render() {
        const { incident, onClose } = this.props;
        const { schema } = this.state;

        return <Window onClose={onClose.bind(this)} title={incident.incident_type}>
            {schema && <FormConstructor schema={schema} onChange={this.updateData.bind(this)}/>}
            <div>
                <Copy text={this.buildText()}><Button>Копировать</Button></Copy>
            </div>
        </Window>;
    }
}
