import * as React from 'react';

import { Dict } from '../../../../../types';
import { ONE_SECOND } from '../../../../constants';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import { PictureBlocksViewer } from '../../../../ui/PictureBlocksViewer';
import { Request2 } from '../../../../utils/request';
import { CAR_INCIDENT_REQUESTS } from '../../../CarCard/CarIncidents/request';
import AllIncidentPhotoModal from '../AllIncidentPhotoModal/component';
import PerformModal from '../PerformModal/component';
import { IIncident } from '../types';
import * as style from './index.css';

interface IIncidentPhotoModalProps {
    loadIncidents: (initial?: boolean) => void;
    onClose: () => void;
    incident: IIncident;
    carId?: string | null;
}

interface IIncidentPhotoModalState {
    isLoading: boolean;
    loadingError: Error | null;
    isPerformModalOpen: boolean;
    isOpenAllIncidentPhotoModal: boolean;
    initialData: Dict<any> | null;
}

export default class IncidentPhotoModal extends React.Component<IIncidentPhotoModalProps, IIncidentPhotoModalState> {
    state: IIncidentPhotoModalState = {
        isLoading: false,
        loadingError: null,
        isPerformModalOpen: false,
        isOpenAllIncidentPhotoModal: false,
        initialData: null,
    };
    request = new Request2({ requestConfigs: CAR_INCIDENT_REQUESTS });

    componentWillUnmount() {
        this.request.abort();
    }

    onEditPhotoClick(image: Dict<any>) {
        const { incident } = this.props;

        const initialData = {
            incident_id: incident.incident_id,
            transition: {
                transition_id: 'handle_attached_photos',
                actions: [
                    {
                        action_type: {
                            action_type: 'upsert',
                            attachment_code: image.attachment_code,
                            image_id: image.image_id,
                        },
                    },
                ],
            },
        };

        this.setState({ initialData }, () => {
            this.openPerformModal();
        });
    }

    onRemovePhotoClick(image: Dict<any>) {
        const { incident } = this.props;

        const initialData = {
            incident_id: incident.incident_id,
            transition: {
                transition_id: 'handle_attached_photos',
                actions: [
                    {
                        action_type: {
                            action_type: 'remove',
                            image_id: image.image_id,
                        },
                    },
                ],
            },
        };

        this.setState({ initialData }, () => {
            this.openPerformModal();
        });
    }

    openPerformModal() {
        this.setState({ isPerformModalOpen: true });
    }

    closePerformModal(success = false) {
        this.setState({ isPerformModalOpen: false, initialData: null }, () => {
            if (success) {
                const { loadIncidents } = this.props;
                loadIncidents();
            }
        });
    }

    openAllIncidentPhotoModal() {
        this.setState({ isOpenAllIncidentPhotoModal: true });
    }

    closeAllIncidentPhotoModal() {
        this.setState({ isOpenAllIncidentPhotoModal: false }, () => {
            this.props.loadIncidents();
        });
    }

    render() {
        const { onClose, incident, carId } = this.props;
        const { isOpenAllIncidentPhotoModal, isPerformModalOpen, initialData } = this.state;
        const photoIncidentImages = incident?.images ?? {};

        return <Window title={'Фото по машине'} onClose={onClose.bind(this)}>
            {Object.entries(photoIncidentImages).map(entry => {
                const [tag, images] = entry;

                return <div key={tag}>
                    <h4>{tag} ({images.length}):</h4>
                    <PictureBlocksViewer pictures={images.map(image => {
                        const {
                            image_url,
                            last_modified,
                            originator_id,
                            attachment_code,
                            attachment_code_description,
                        } = image;

                        const info = <div className={style.image_info_container}>
                            <div className={style.attachment_code}>
                                <span className={style.code}>{attachment_code ?? ''}</span>
                                <span className={style.description}> {attachment_code_description ?? ''}</span>
                            </div>
                            <div className={style.image_info}>
                                <div>
                                    <FormatDate value={last_modified * ONE_SECOND}/>
                                    <div>
                                        <Link href={`#/clients/${originator_id}/info`} target={'_blank'}>Автор</Link>
                                    </div>
                                </div>
                                <div>
                                    <div>
                                        <Link onClick={this.onEditPhotoClick.bind(this, image)}>Изменить</Link>
                                    </div>
                                    <div>
                                        <Link onClick={this.onRemovePhotoClick.bind(this, image)}>Открепить</Link>
                                    </div>
                                </div>
                            </div>
                        </div>;

                        return { link: image_url, info: info };
                    })}/>
                </div>;
            })}

            <Link onClick={this.openAllIncidentPhotoModal.bind(this)} className={style.attach_photo_link}>Прикрепить
                фото</Link>
            {isPerformModalOpen
                ? <PerformModal initialData={initialData}
                                onClose={this.closePerformModal.bind(this)}
                                incident={incident}/>
                : null}
            {isOpenAllIncidentPhotoModal
                ? <AllIncidentPhotoModal carId={carId ?? ''}
                                         incident={incident}
                                         onClose={this.closeAllIncidentPhotoModal.bind(this)}/>
                : null}
        </Window>;
    }
}
