import * as React from 'react';
import ReactTooltip from 'react-tooltip';

import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import CarInfo from '../../../../models/car';
import { SessionHistoryInfoHandler } from '../../../../models/session';
import { UserInfoHandler } from '../../../../models/user';
import FormatDate from '../../../../ui/FormatDate';
import { JsonModal } from '../../../../ui/FullModal/JsonModal';
import { Link } from '../../../../ui/Link';
import { LabelStatus, TLabel } from '../../../../ui/Table';
import * as styleTable from '../../../../ui/Table/index.css';
import { Request2 } from '../../../../utils/request';
import { CAR_INCIDENT_REQUESTS } from '../../../CarCard/CarIncidents/request';
import CarNumber from '../../../CarNumber';
import { Copy } from '../../../Copy';
import EvacInformModal from '../EvacInformModal';
import getClaimNumber from '../getClaimNumber';
import { IncidentToTelegram } from '../IncidentToTelegram';
import PerformModal from '../PerformModal/component';
import { IIncident, INCIDENT_LINK } from '../types';
import IncidentPhotoModal from './IncidentPhotoModal';
import * as style from './index.css';

const TOOLTIP = 'TOOLTIP';

interface IIncidentsTableProps {
    incidents: IIncident[];
    loadIncidents: (initial?: boolean) => void;
    hideColumns: {
        date?: boolean; type?: boolean; status?: boolean;
        session?: boolean; car?: boolean; user?: boolean;
        links?: boolean; edit?: boolean; photo?: boolean;
    } | null;
    carId?: string | null;
}

interface IIncidentsTableState {
    isLoading: boolean;
    loadingError: Error | null;
    infoModalIncident: IIncident | null;
    tgInfoModalIncident: IIncident | null;
    performIncident: IIncident | null;
    photoIncidentId: string | null;
    isPhotoModalOpen: boolean;
    isPerformModalOpen: boolean;
    isEvacInformModalOpen: boolean;
}

export default class IncidentsTable extends React.Component<IIncidentsTableProps, IIncidentsTableState> {
    state: IIncidentsTableState = {
        isLoading: false,
        loadingError: null,
        infoModalIncident: null,
        tgInfoModalIncident: null,
        performIncident: null,
        photoIncidentId: null,
        isPhotoModalOpen: false,
        isPerformModalOpen: false,
        isEvacInformModalOpen: false,
    };
    request = new Request2({ requestConfigs: CAR_INCIDENT_REQUESTS });

    componentWillUnmount() {
        this.request.abort();
    }

    openInfoModal(infoModalIncident: IIncident) {
        this.setState({ infoModalIncident });
    }

    openTelegramModal(tgInfoModalIncident: IIncident) {
        this.setState({ tgInfoModalIncident });
    }

    closeInfoModal() {
        this.setState({ infoModalIncident: null });
    }

    openPerformModal(performIncident: IIncident) {
        this.setState({ performIncident, isPerformModalOpen: true });
    }

    closePerformModal() {
        this.setState({ performIncident: null, isPerformModalOpen: false });
    }

    buildLinks(incident: IIncident) {
        const links = incident?.links ?? [];

        return links.map((link, index) => {
            const { type, comment } = link;

            let linkComponent: React.ReactElement | null;

            switch (type) {
            case INCIDENT_LINK.STARTREK_TICKET:
                const { ticket_key, ticket_url } = link;
                linkComponent =
                        <Link target={'_blank'} href={ticket_url}>Тикет{ticket_key ? ` (${ticket_key})` : ''}</Link>;
                break;
            case INCIDENT_LINK.TAG:
                const { tag_id, tag_entity_type } = link;
                linkComponent =
                        <Link disable
                              target={'_blank'}
                              href={tag_id}>Тег{tag_entity_type ? ` (${tag_entity_type})` : ''}</Link>;
                break;
            case INCIDENT_LINK.DOCUMENT:
                const { document_type, document_url } = link;
                linkComponent =
                        <Link disable
                              target={'_blank'}
                              href={document_url}>Документ{document_type ? ` (${document_type})` : ''}</Link>;
                break;
            case INCIDENT_LINK.PHOTO:
                linkComponent = null;
                break;
            default:
                linkComponent = <Link disable>Неизвестная ссылка (тип {type})</Link>;
                break;
            }

            return <div key={index}>
                {linkComponent}
                {comment ? <div className={style.link_comment}>{comment}</div> : null}
            </div>;
        });
    }

    openPhotoModal(photoIncidentId: string) {
        this.setState({ isPhotoModalOpen: true, photoIncidentId });
    }

    closePhotoModal() {
        this.setState({ isPhotoModalOpen: false, photoIncidentId: null });
    }

    openEvaInformModal(performIncident: IIncident, isEvacInformModalOpen: boolean) {
        this.setState({ performIncident, isEvacInformModalOpen });
    }

    render() {
        const { incidents, loadIncidents, carId } = this.props;
        const hideColumns = this.props.hideColumns ?? {};
        const {
            infoModalIncident, isPerformModalOpen, isPhotoModalOpen, performIncident, tgInfoModalIncident,
            isEvacInformModalOpen,
        } = this.state;
        const { date, type, status, session, car, user, links, edit } = hideColumns;
        const photoIncident: any = this.state.photoIncidentId !== null
            ? incidents.filter(incident => incident.incident_id === this.state.photoIncidentId)?.[0]
            : null;

        return <div>
            <ReactTooltip id={TOOLTIP} html/>
            {incidents?.length
                ? <table className={styleTable.table}>
                    <thead>
                        <tr>
                            <th>#</th>
                            {!date ? <th>Дата создания</th> : null}
                            {!type ? <th>Тип</th> : null}
                            {!status ? <th>Статус</th> : null}
                            {!session ? <th/> : null}
                            {!car ? <th>Авто</th> : null}
                            {!user ? <th>Пользователь</th>
                                : null}
                            <th>Ссылки</th>
                            <th/>
                            <th/>
                            <th/>
                            {!edit ? <th/> : null}
                        </tr>
                    </thead>
                    <tbody>
                        {incidents.map((incident, index) => {
                            const {
                                incident_id, initiated_at, incident_type, incident_status,
                                session_id, car_id, user_id, carInfo, userInfo, sessionInfo,
                            } = incident;

                            const _incidentStatusLabel = incidentStatusLabel(incident_status);

                            const claim_number = getClaimNumber(incident);

                            return <tr key={incident_id ?? index}>
                                <td>{index + 1}</td>
                                {!date
                                    ? <td>
                                        <FormatDate value={initiated_at * ONE_SECOND}/>
                                    </td>
                                    : null}
                                {!type
                                    ? <td>{incident_type || EMPTY_DATA}</td>
                                    : null}
                                {!status
                                    ? <td>
                                        <div>{_incidentStatusLabel}</div>
                                        {claim_number &&
                                    <div title={'ClaimNumber'}><b>CN: </b><Copy>{claim_number}</Copy></div>}
                                    </td>
                                    : null}
                                {!session
                                    ? <td>{session_id
                                        ? <Link target={'_blank'}
                                                href={`#/session/${session_id}/billing?back=${encodeURIComponent(location.hash)}`}>
                                            {sessionInfo && SessionHistoryInfoHandler.getFinish.call(sessionInfo)
                                                ?
                                                <FormatDate value={SessionHistoryInfoHandler.getFinish
                                                    .call(sessionInfo)}/>
                                                : 'Сессия'}
                                        </Link>
                                        : EMPTY_DATA}
                                    </td>
                                    : null}
                                {!car
                                    ? <td>
                                        <div>
                                            {carInfo
                                                ?
                                                <CarNumber carInfo={carInfo as typeof CarInfo}
                                                           externalTooltipId={TOOLTIP}/>
                                                : null}
                                        </div>
                                        {car_id
                                            ? <Link target={'_blank'} href={`#/cars/${car_id}/info`}>

                                                {carInfo && (carInfo?.modelInfo?.name || carInfo?.model_id)
                                                    ? carInfo?.modelInfo?.name || carInfo?.model_id
                                                    : 'Авто'}
                                            </Link>
                                            : EMPTY_DATA}
                                    </td>
                                    : null}
                                {!user
                                    ? <td>{user_id
                                        ? <Link target={'_blank'} href={`#/clients/${user_id}/info`}>
                                            {userInfo && UserInfoHandler.getPrintNameWithoutPn
                                                .call(userInfo) || 'Пользователь'}
                                        </Link>
                                        : EMPTY_DATA}
                                    </td>
                                    : null}
                                {!links
                                    ? <td>
                                        {this.buildLinks(incident)}
                                    </td>
                                    : null}
                                <td>
                                    <Link onClick={this.openTelegramModal.bind(this, incident)}>telegram</Link>
                                </td>
                                <td>
                                    <Link onClick={this.openPhotoModal.bind(this, incident.incident_id)}>
                                    Фото {incident.images && Object.values(incident?.images ?? {})?.length
                                            ? `(${Object.values(incident?.images ?? {})
                                                .reduce((result: number, imagesArray) => {
                                                    result += imagesArray?.length ?? 0;

                                                    return result;
                                                }, 0)})`
                                            : null}
                                    </Link>
                                </td>
                                <td>
                                    <Link onClick={this.openInfoModal.bind(this, incident)}>Info</Link>
                                </td>
                                {!edit
                                    ? <td>
                                        <div>
                                            <Link onClick={this.openPerformModal.bind(this, incident)}>Изменить</Link>
                                        </div>
                                        {incident_type === 'Эвакуация'
                                            && <div>
                                                <Link onClick={this.openEvaInformModal.bind(this, incident, true)}>
                                                    Оповестить и списать
                                                </Link>
                                            </div>}
                                    </td>
                                    : null}
                            </tr>;
                        })}
                    </tbody>
                </table>
                : <h4>Инцидентов не найдено</h4>}
            {infoModalIncident
                ? <JsonModal obj={infoModalIncident}
                             title={'Инцидент'}
                             onClose={this.closeInfoModal.bind(this)}/>
                : null}
            {tgInfoModalIncident && <IncidentToTelegram incident={tgInfoModalIncident}
                                                        onClose={this.openTelegramModal.bind(this, null)}/>}
            {isEvacInformModalOpen && <EvacInformModal incident={performIncident}
                                                       onClose={this.openEvaInformModal.bind(this, null, false)}/>}
            {isPerformModalOpen
                ? <PerformModal onClose={this.closePerformModal.bind(this)} incident={performIncident}/>
                : null}
            {isPhotoModalOpen
                ? <IncidentPhotoModal loadIncidents={loadIncidents.bind(this)}
                                      incident={photoIncident}
                                      carId={carId}
                                      onClose={this.closePhotoModal.bind(this)}/>
                : null}
        </div>;
    }
}

export const incidentStatusLabel = (incident_status) => {
    return incident_status === 'Закрыт'
        ? <TLabel status={LabelStatus.NEGATIVE_DARK} text={incident_status}/>
        : incident_status === 'Отменен'
            ? <TLabel status={LabelStatus.WARNING} text={incident_status}/>
            : incident_status === 'Тикет создан'
                ? <TLabel status={LabelStatus.POSITIVE} text={incident_status}/>
                : <TLabel status={LabelStatus.DEFAULT} text={incident_status}/>
                || EMPTY_DATA;
};
