import * as React from 'react';

import { Dict } from '../../../../../types';
import { Button, CancelButton, SaveButton } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Request2 } from '../../../../utils/request';
import { Copy } from '../../../Copy';
import { FormConstructor } from '../../../FormConstructor';
import Spin from '../../../Spin';
import getClaimNumber from '../getClaimNumber';
import { INCIDENT_REQUESTS, REQUESTS } from '../request';
import { IIncident } from '../types';

interface IPerformModalProps {
    onClose: (success?: boolean) => void;
    incident: IIncident | null;
    initialData?: Dict<any> | null;
}

interface IPerformModalState {
    isLoading: boolean;
    loadingError: Error | null;

    performSchema: Dict<any>;
    performData: Dict<any>;

    isPerforming: boolean;
    performError: Error | null;

    claim_number: string | null;
}

export default class PerformModal extends React.Component<IPerformModalProps, IPerformModalState> {
    state: IPerformModalState = {
        isLoading: false,
        loadingError: null,

        performSchema: {},
        performData: {},

        isPerforming: false,
        performError: null,

        claim_number: null,
    };
    request = new Request2({ requestConfigs: INCIDENT_REQUESTS });

    componentDidMount(): void {
        this.setState({ isLoading: true, loadingError: null }, () => {
            const { incident } = this.props;
            const incident_id = incident?.incident_id ?? '';

            this.request.exec(REQUESTS.GET_PERFORM_INCIDENTS_SCHEMA, {
                queryParams: {
                    scheme: true,
                    incident_id,
                },
            })
                .then((response => {
                    const performSchema = response?.request_data?.structure ?? {};
                    this.setState({ performSchema });

                    this.setState({ isLoading: false });
                }))
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    componentWillUnmount() {
        this.request.abort();
    }

    onPerformFormChange(performData: Dict<any>) {
        this.setState({ performData, claim_number: null });
    }

    performIncident() {
        const { onClose } = this.props;
        const { performData } = this.state;

        this.setState({ isPerforming: true, performError: null }, () => {
            this.request.exec(REQUESTS.PERFORM_INCIDENTS, {
                body: performData,
            })
                .then((response) => {
                    let claim_number = null;
                    if (performData?.transition?.transition_id === 'apply_renins_kasko_claim') {
                        claim_number = getClaimNumber(response);
                    }

                    !claim_number && onClose(true);
                    this.setState({ isPerforming: false, claim_number });
                })
                .catch(performError => {
                    this.setState({ performError, isPerforming: false });
                });
        });
    }

    render() {
        const { onClose, initialData } = this.props;
        const { loadingError, isLoading, performSchema, isPerforming, performError, claim_number } = this.state;

        return <Window error={loadingError || performError} onClose={onClose.bind(this)} title={'Изменить инцидент'}>
            {isLoading
                ? <Spin size={'l'}/>
                : <>
                    <FormConstructor schema={performSchema}
                                     initialData={initialData ?? null}
                                     onChange={this.onPerformFormChange.bind(this)}/>

                    {
                        claim_number
                            ? <div className={coreStyle.button_container}>
                                <b>ClaimNumber: </b> <Copy>{claim_number}</Copy>
                                <Button onClick={onClose.bind(this)}>Закрыть</Button>
                            </div>
                            :
                            <div className={coreStyle.button_container}>
                                <CancelButton onClick={onClose.bind(this)}/>
                                <SaveButton disabled={isPerforming} onClick={this.performIncident.bind(this)}/>
                            </div>
                    }
                </>}
        </Window>;
    }
}
