import * as React from 'react';

import { Dict, LSSettingItems } from '../../../../types';
import { ONE_SECOND } from '../../../constants';
import { Button } from '../../../ui/Button';
import DatePicker from '../../../ui/DatePicker';
import IDSelect from '../../../ui/IDSelect/component';
import { GLOBAL_SEARCH_OBJECTS } from '../../../ui/IDSelect/constants';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import Select, { IOptionInfo } from '../../../ui/Select';
import LS from '../../../utils/localStorage/localStorage';
import { Request2 } from '../../../utils/request';
import { deepCopy } from '../../../utils/utils';
import IncidentForm from '../../IncidentForm';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import AllIncidentPhotoModal from './AllIncidentPhotoModal/component';
import CreateIncidentModal from './CreateIncidentModal/component';
import IncidentsTable from './IncidentsTable/component';
import * as style from './index.css';
import { INCIDENT_REQUESTS, REQUESTS } from './request';
import { IIncident, INCIDENT_LINK } from './types';

const EMPTY_FIELDS_OPTIONS: IOptionInfo[] = [
    {
        value: 'car_id',
        text: 'Авто',
    },
    {
        value: 'user_id',
        text: 'Пользователь',
    },
    {
        value: 'session_id',
        text: 'Сессия',
    },
];

interface IIncidentsFeedProps {
    isFiltersOn?: {
        type?: boolean; status?: boolean; since?: boolean; until?: boolean;
        carId?: boolean; userId?: boolean; sessionId?: boolean; isOnlyEmpty?: boolean;
    };
    initialFiltersData?: {
        type?: string[]; status?: string[]; since?: number; until?: number;
        carId?: string; userId?: string; sessionId?: string; isOnlyEmpty?: string[];
    };
    hideColumns?: {
        date?: boolean; type?: boolean; status?: boolean;
        session?: boolean; car?: boolean; user?: boolean;
        links?: boolean; edit?: boolean; photo?: boolean; fullIncidentButton?: boolean;
    };
}

interface IIncidentsFeedState {
    isLoading: boolean;
    loadingError: Error | null;
    incidents: IIncident[];
    incidentTypesOptions: IOptionInfo[];
    incidentStatusesOptions: IOptionInfo[];
    incidentTypes: string[];
    incidentStatuses: string[];
    filteredIncidents: IIncident[];
    isOnlyEmpty: string[];
    filters: Dict<any>;
    isCreateIncidentModalOpen: boolean;
    isPhotoModalOpen: boolean;
    allImages: Dict<any>[];
    isCreateFullIncidentModalOpen: boolean;
}

export default class IncidentsFeed extends React.Component<IIncidentsFeedProps, IIncidentsFeedState> {
    state: IIncidentsFeedState = {
        isLoading: false,
        loadingError: null,
        incidents: [],
        incidentTypesOptions: [],
        incidentStatusesOptions: [],
        incidentTypes: [],
        incidentStatuses: [],
        filteredIncidents: [],
        isOnlyEmpty: [],
        filters: {
            since: null,
            until: null,
            session_id: null,
            car_id: null,
            user_id: null,
        },
        isCreateIncidentModalOpen: false,
        isPhotoModalOpen: false,
        allImages: [],
        isCreateFullIncidentModalOpen: false,
    };
    request = new Request2({ requestConfigs: INCIDENT_REQUESTS });
    ls = new LS();

    componentDidMount(): void {
        let incidentTypes: string[] = this.ls.get(LSSettingItems.incident_types_filter) ?? [];
        const initialFiltersData: any = this.props.initialFiltersData ?? null;

        incidentTypes = initialFiltersData?.type ? initialFiltersData?.type : incidentTypes;
        const incidentStatuses = initialFiltersData?.status ? initialFiltersData?.status : [];
        const isOnlyEmpty = initialFiltersData?.isOnlyEmpty ? initialFiltersData?.isOnlyEmpty : [];
        const filters = {
            since: initialFiltersData?.since ?? null,
            until: initialFiltersData?.until ?? null,
            session_id: initialFiltersData?.sessionId ?? null,
            car_id: initialFiltersData?.carId ?? null,
            user_id: initialFiltersData?.userId ?? null,
        };
        this.setState({ incidentTypes, incidentStatuses, isOnlyEmpty, filters }, () => {
            this.loadIncidents(true);
        });
    }

    componentWillUnmount() {
        this.request.abort();
    }

    loadIncidents(initial = false) {
        this.setState({ isLoading: initial, loadingError: null }, () => {
            const { filters } = this.state;
            const { since = null, until = null, car_id = null, user_id = null, session_id = null } = filters;
            const queryParams = {
                since: typeof since === 'number' ? Math.floor(since / ONE_SECOND) : null,
                until: typeof until === 'number' ? Math.floor(until / ONE_SECOND) : null,
                car_id,
                user_id,
                session_id,
            };

            this.request.exec(REQUESTS.GET_INCIDENTS, { queryParams: queryParams ?? {} })
                .then((response => {
                    let incidents: IIncident[] = response?.incidents ?? [];
                    const incidentTypes = Array.from(new Set(incidents.map(incident => incident.incident_type)));
                    const incidentTypesOptions = incidentTypes.map(type => {
                        return { value: type };
                    });
                    const incidentStatuses = Array.from(new Set(incidents.map(incident => incident.incident_status)));
                    const incidentStatusesOptions = incidentStatuses.map(status => {
                        return { value: status };
                    });

                    const car_models = response?.car_models ?? [];
                    const cars = response?.cars ?? [];
                    const sessions = response?.sessions ?? [];
                    const users = response?.users ?? [];
                    const attachment_code_description_mapping = response?.attachment_code_description_mapping ?? {};
                    const allImages = response?.images ?? [];

                    incidents = incidents.map(incident => {
                        const carInfo = cars?.find(car => car?.id === incident?.car_id) ?? null;
                        if (carInfo) {
                            carInfo.modelInfo = car_models
                                ?.find(car_model => car_model?.code === carInfo?.model_id) ?? null;
                        }

                        const userInfo = users?.find(user => user?.id === incident?.user_id) ?? null;
                        const sessionInfo = sessions
                            ?.find(session => session?.session_id === incident?.session_id) ?? null;
                        const incidentLinks = incident.links ?? [];
                        const incidentImages = incidentLinks
                            .filter(link => link.type === INCIDENT_LINK.PHOTO)
                            .map(photoLink => {
                                const imageId = photoLink.image_id;
                                const commonImageWithUrl = allImages.find(image => image.image_id === imageId);

                                if (commonImageWithUrl) {
                                    commonImageWithUrl.attached = true;
                                    photoLink.image_url = commonImageWithUrl.url;
                                    photoLink.marker = commonImageWithUrl.marker;
                                }

                                const { attachment_code } = photoLink;
                                photoLink.attachment_code_description = attachment_code_description_mapping
                                    ?.[attachment_code ?? ''] ?? '';

                                return photoLink;
                            });

                        incident.carInfo = carInfo;
                        incident.userInfo = userInfo;
                        incident.sessionInfo = sessionInfo;
                        incident.images = incidentImages.reduce((result: Dict<any>, image) => {
                            const { marker = 'others' } = image;

                            if (result[marker]) {
                                result[marker].push(image);
                            } else {
                                result[marker] = [image];
                            }

                            return result;
                        }, {});

                        return incident;
                    });

                    this.setState({
                        isLoading: false,
                        incidents,
                        incidentTypesOptions,
                        incidentStatusesOptions,
                        allImages,
                    }, () => {
                        const filteredIncidents = this.filterIncidents();
                        this.setState({ filteredIncidents });
                    });

                }))
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    //Method to filter incidents on front if no cgi in API
    filterIncidents() {
        let filteredIncidents: IIncident[] = this.state.incidents;

        if (this.state.isOnlyEmpty?.length) {
            filteredIncidents = filteredIncidents.filter(incident => {
                return this.state.isOnlyEmpty.every(isOnlyEmptyField => !incident?.[isOnlyEmptyField]);
            });
        }

        if (this.state.incidentTypes?.length) {
            filteredIncidents = filteredIncidents
                .filter(incident => this.state.incidentTypes.includes(incident.incident_type));
        }

        if (this.state.incidentStatuses?.length) {
            filteredIncidents = filteredIncidents
                .filter(incident => this.state.incidentStatuses.includes(incident.incident_status));
        }

        return filteredIncidents;
    }

    onIncidentTypeChange(incidentTypes: string[]) {
        this.ls.set(LSSettingItems.incident_types_filter, incidentTypes);
        this.setState({ incidentTypes }, () => {
            this.setState({ filteredIncidents: this.filterIncidents() });
        });
    }

    onIncidentStatusChange(incidentStatuses: string[]) {
        this.setState({ incidentStatuses }, () => {
            this.setState({ filteredIncidents: this.filterIncidents() });
        });
    }

    onEmptyFieldsChange(isOnlyEmpty: string[]) {
        this.setState({ isOnlyEmpty }, () => {
            this.setState({ filteredIncidents: this.filterIncidents() });
        });
    }

    onFilterChange(type: 'car_id' | 'user_id' | 'session_id' | 'since' | 'until', value: string) {
        const filters = deepCopy(this.state.filters);
        filters[type] = value;
        this.setState({ filters }, () => {
            this.loadIncidents(true);
        });
    }

    buildFilters() {
        const { incidentTypesOptions, incidentStatusesOptions, incidentTypes, incidentStatuses } = this.state;

        const isFiltersOn = this.props.isFiltersOn ?? null;
        const {
            type = false, status = false, since = false, until = false,
            carId = false, userId = false, sessionId = false, isOnlyEmpty = false,
        } = isFiltersOn ?? {};

        return <>
            <div className={style.incident_table_controls}>
                {!isFiltersOn || type
                    ? <Select placeholder={'Тип инцидента'}
                              options={incidentTypesOptions}
                              initialValues={incidentTypes}
                              multiSelect
                              onSelect={this.onIncidentTypeChange.bind(this)}
                              containerClassName={style.incident_table_control}/>
                    : null}
                {!isFiltersOn || status
                    ? <Select placeholder={'Статус инцидента'}
                              options={incidentStatusesOptions}
                              initialValues={incidentStatuses}
                              multiSelect
                              onSelect={this.onIncidentStatusChange.bind(this)}
                              containerClassName={style.incident_table_control}/>
                    : null}
                {!isFiltersOn || since
                    ? <DatePicker placeholder={'С'}
                                  onChange={this.onFilterChange.bind(this, 'since')}
                                  value={this.state.filters.since}
                                  className={style.incident_table_control}/>
                    : null}
                {!isFiltersOn || until
                    ? <DatePicker placeholder={'По'}
                                  onChange={this.onFilterChange.bind(this, 'until')}
                                  value={this.state.filters.until}
                                  className={style.incident_table_control}/>
                    : null}
            </div>
            <div className={style.incident_table_controls}>
                {!isFiltersOn || carId
                    ? <IDSelect placeholder={'Авто'}
                                initialValues={this.state.filters.car_id}
                                object={GLOBAL_SEARCH_OBJECTS.cars}
                                onSelect={this.onFilterChange.bind(this, 'car_id')}
                                containerClassName={style.incident_table_control}/>
                    : null}
                {!isFiltersOn || userId
                    ? <IDSelect placeholder={'Пользователь'}
                                initialValues={this.state.filters.user_id}
                                object={GLOBAL_SEARCH_OBJECTS.users}
                                onSelect={this.onFilterChange.bind(this, 'user_id')}
                                containerClassName={style.incident_table_control}/>
                    : null}
                {!isFiltersOn || sessionId
                    ? <Input placeholder={'Сессия'}
                             value={this.state.filters.session_id}
                             onChange={this.onFilterChange.bind(this, 'session_id')}
                             className={style.incident_table_control}/>
                    : null}
                {!isFiltersOn || isOnlyEmpty
                    ? <Select placeholder={'Инциденты БЕЗ'}
                              options={EMPTY_FIELDS_OPTIONS}
                              multiSelect
                              onSelect={this.onEmptyFieldsChange.bind(this)}
                              containerClassName={style.incident_table_control}/>
                    : null}
            </div>
        </>;
    }

    openCreateIncidentModal() {
        this.setState({ isCreateIncidentModalOpen: true });
    }

    closeCreateIncidentModal(success: boolean) {
        this.setState({ isCreateIncidentModalOpen: false }, () => {
            if (success) {
                this.loadIncidents(true);
            }
        });
    }

    openCreateFullIncidentModal() {
        this.setState({ isCreateFullIncidentModalOpen: true });
    }

    closeCreateFullIncidentModal(success: boolean) {
        this.setState({ isCreateFullIncidentModalOpen: false }, () => {
            if (success) {
                this.loadIncidents(true);
            }
        });
    }

    openPhotoModal() {
        this.setState({ isPhotoModalOpen: true });
    }

    closePhotoModal() {
        this.setState({ isPhotoModalOpen: false });
    }

    render() {
        const { hideColumns, initialFiltersData } = this.props;
        const {
            isLoading, loadingError, filteredIncidents,
            isCreateIncidentModalOpen, isPhotoModalOpen,
            isCreateFullIncidentModalOpen,
        } = this.state;

        return <div>
            <Button basic className={style.feed_button} onClick={this.openCreateIncidentModal.bind(this)}>Добавить
                инцидент</Button>
            {!hideColumns?.fullIncidentButton
                ? <Button basic className={style.feed_button} onClick={this.openCreateFullIncidentModal.bind(this)}>
                    Добавить подробный инцидент
                </Button>
                : null}
            {!hideColumns?.photo ?
                <Link onClick={this.openPhotoModal.bind(this)}>Фото по машине</Link> : null}
            {this.buildFilters()}
            {isLoading
                ? <Spin/>
                : loadingError
                    ? <SimpleError error={loadingError} data={{ label: 'Ошибка при загрузке инцидентов' }}/>
                    : <IncidentsTable carId={initialFiltersData?.carId ?? null}
                                      hideColumns={hideColumns ?? null}
                                      loadIncidents={this.loadIncidents.bind(this)}
                                      incidents={filteredIncidents}/>}
            {isCreateIncidentModalOpen
                ? <CreateIncidentModal onClose={this.closeCreateIncidentModal.bind(this)}/>
                : null}
            {isCreateFullIncidentModalOpen
                ? <IncidentForm onClose={this.closeCreateFullIncidentModal.bind(this)}/>
                : null}
            {isPhotoModalOpen
                ? <AllIncidentPhotoModal carId={initialFiltersData?.carId ?? null}
                                         onClose={this.closePhotoModal.bind(this)}/>
                : null}
        </div>;
    }
}
