import React, { useState } from 'react';

import { ONE_SECOND } from '../../../../constants';
import { Button, CancelButton } from '../../../../ui/Button';
import Checkbox from '../../../../ui/Checkbox';
import { Collapse2 } from '../../../../ui/Collapse2';
import DatePicker from '../../../../ui/DatePicker';
import { Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import Select from '../../../../ui/Select';
import { Request2 } from '../../../../utils/request';
import { rubs } from '../../../../utils/rubs';
import { SimpleError } from '../../../SimpleError';
import { REQUESTS, SUPPORT_REQUESTS } from '../../request';
import style from '../index.css';
import { IOfferInfo } from '../types';

interface ICalculateRentModalProps {
    onWindowClose: () => void;
    offer: IOfferInfo;
    userId: string | undefined;
}

export const CalculateRentModal = React.memo((props: ICalculateRentModalProps) => {
    const request = new Request2({ requestConfigs: SUPPORT_REQUESTS });
    const paymentPlanOptions = getPaymentPlanOptions(props.offer);
    const mileageOptions = getMileageOptions(props.offer);
    const franchiseOptions = getFranchiseOptions(props.offer);

    const [paymentPlan, setPaymentPlan] = useState<string | null>(null);
    const [mileage, setMileage] = useState<number | null>(null);
    const [franchise, setFranchise] = useState<number | null>(null);
    const [childSeat, setChildSeat] = useState<boolean>(false);
    const [dateSince, setDateSince] = useState<number>(+new Date());
    const [dateUntil, setDateUntil] = useState<number>(+new Date());
    const [isLoading, setLoading] = useState<boolean>(false);
    const [error, setError] = useState<Error | null>(null);
    const [finalCost, setFinalCost] = useState<number | null>(null);

    function showParams() {
        setLoading(true);
        const headers = props.userId ? { UserIdDelegation: props.userId } : null;

        request.exec(REQUESTS.GET_OFFERS, {
            queryParams: {
                offer_name: props.offer?.price_constructor_id,
            },
            body: {
                variables: {
                    'mileage': mileage,
                    'child_seat': childSeat,
                    'franchise': franchise,
                    'payment_plan': paymentPlan,
                    'since': dateSince / ONE_SECOND,
                    'until': dateUntil / ONE_SECOND,
                },
            },
            headers,
        })
            .then(response => {
                const cost = response?.offers?.[0]?.payment_plan?.cost;
                cost && setFinalCost(cost);

                setLoading(false);
                setError(null);
            })
            .catch(error => {
                setLoading(false);
                setError(error);
            });
    }

    const disableButton = !paymentPlan || mileage === null || franchise === null;

    return <Window onClose={props.onWindowClose.bind(null)} title={'Калькулятор цен'}>
        {error
            ? <Collapse2 title={'Ошибочка!'} children={<SimpleError error={error} />} />
            : null
        }

        <Select options={paymentPlanOptions}
                onSelect={setPaymentPlan.bind(null)}
                placeholder={'План выплат'}
                className={style.calc_item}
                required={true}/>

        <Select options={mileageOptions}
                onSelect={setMileage.bind(null)}
                placeholder={'Пробег'}
                className={style.calc_item}
                required={true}/>

        <Select options={franchiseOptions}
                onSelect={setFranchise.bind(null)}
                placeholder={'Франшиза виновника'}
                className={style.calc_item}
                required={true}/>

        <DatePicker value={dateSince}
                    onChange={setDateSince.bind(null)}
                    placeholder={'Дата начала аренды'}
                    className={style.calc_item}
                    required={true}/>

        <DatePicker value={dateUntil}
                    onChange={setDateUntil.bind(null)}
                    placeholder={'Дата завершения аренды'}
                    className={style.calc_item}
                    required={true}/>

        <div className={style.calc_placeholder}>
            <div>Детское кресло:</div>
            <Checkbox checked={childSeat} onChange={setChildSeat.bind(null)}/>
        </div>

        {finalCost
            ? <div className={style.final_cost}>
                Итоговая сумма: {rubs(finalCost)}
            </div>
            : null
        }

        <div className={coreStyle.button_container}>
            <CancelButton onClick={props.onWindowClose.bind(null)}/>
            <Button onClick={showParams.bind(null)} isLoading={isLoading} disabled={disableButton}>
                Посчитать!
            </Button>
        </div>
    </Window>;
});

function getPaymentPlanOptions(offer) {
    return offer?.payment_plan?.values?.map(el => {
        return { value: el.value, text: el.title };
    });
}

function getMileageOptions(offer) {
    const item = offer?.primary?.find(el => el.id === 'mileage') ?? {};

    return item?.values?.map(el => {
        return { value: el };
    });
}

function getFranchiseOptions(offer) {
    const item = offer?.primary?.find(el => el.id === 'franchise') ?? {};

    return item?.values?.map(el => {
        return { value: el };
    });
}
