import React, { useState } from 'react';

import { EMPTY_DATA, ONE_DAY, ONE_SECOND } from '../../../../constants';
import { Window } from '../../../../ui/FullModal';
import { JsonModal } from '../../../../ui/FullModal/JsonModal';
import { Link } from '../../../../ui/Link';
import { rubs } from '../../../../utils/rubs';
import style from '../index.css';
import { IDebugInfoItem, IOfferInfo } from '../types';
import { CalculateRentModal } from './CalculateRentModal';

interface IOfferInfoProps {
    selectOffer: () => void;
    debugData: IDebugInfoItem;
    selectedOffer: IOfferInfo;
    userId: string | undefined;
    modelInfo: any;
}

export const OfferInfo = React.memo((props: IOfferInfoProps) => {
    const { selectOffer, selectedOffer, modelInfo, userId } = props;

    const [carDetailsOpen, setCarDetailsOpen] = useState(false);
    const [offerDetailsOpen, setOfferDetailsOpen] = useState(false);
    const [rentCalculateOpen, setRentCalculateOpen] = useState(false);
    const [debugInfoOpen, setdebugInfoOpen] = useState(false);

    const prebill = selectedOffer?.prebill ?? [];
    const hasDebug = Object.keys(props.debugData)?.length;

    function getMinimalOrderDuration() {
        const periodInSec = props.selectedOffer?.minimal_period ?? 0;

        return periodInSec / ONE_DAY * ONE_SECOND ?? EMPTY_DATA;
    }

    function getMaximumOrderDuration() {
        const periodInSec = props.selectedOffer?.maximal_period ?? 0;

        return periodInSec / ONE_DAY * ONE_SECOND ?? EMPTY_DATA;
    }

    return <div>
        <div className={style.back_link} onClick={selectOffer.bind(null, null)}>
            <span className={style.link_label}>ᐸ</span> К офферам
        </div>

        <div className={style.offer_info_wrapper}>
            <div className={style.info_block}>
                <CurrentOfferVisual offer={selectedOffer}/>

                <div className={style.car_info}>
                    <div className={style.subtitle}>Машина</div>
                    <table className={style.car_info_table}>
                        <tbody>
                            {modelInfo?.specifications?.length
                                ? modelInfo.specifications.map(item => {
                                    return <tr key={item.name} className={style.car_info_item}>
                                        <td><b>{item.name}:</b></td>
                                        <td>{item.value}</td>
                                    </tr>;
                                })
                                : null
                            }
                        </tbody>
                    </table>
                </div>

                {selectedOffer?.badges?.length
                    ? <>
                        <div className={style.subtitle}>Бейджи</div>
                        <div>
                            {selectedOffer.badges.map(badge => {
                                return Badge(badge.text);
                            })}
                        </div>
                    </>
                    : null
                }

                {selectedOffer?.car_description
                    ? <Link onClick={setCarDetailsOpen.bind(null, true)}
                            className={style.car_details_link}>
                        Все характеристики и комплектация {LINK_LABEL}
                    </Link>
                    : null
                }

                <Link onClick={setRentCalculateOpen.bind(null, true)}
                      className={style.car_details_link}>
                    Посчитать стоимость аренды {LINK_LABEL}
                </Link>

                {hasDebug
                    ? <Link onClick={setdebugInfoOpen.bind(null, true)}
                            className={style.car_details_link}>
                        Debug Session Info {LINK_LABEL}
                    </Link>
                    : null
                }
            </div>

            <div className={style.info_block}>
                <div className={style.order_info}>
                    <div className={style.item}>
                        <div className={style.subtitle}>
                            Период аренды
                        </div>
                        <div className={style.values}>
                            {getMinimalOrderDuration()} - {getMaximumOrderDuration()}
                        </div>
                        <div className={style.subvalue}>
                            Количество дней, на которое можно взять машину
                        </div>
                    </div>

                    {selectedOffer?.primary?.length
                        ? selectedOffer.primary.map(item => {
                            return <div className={style.item} key={item.id}>
                                <div className={style.subtitle}>
                                    {item.title}
                                </div>
                                <div className={style.values}>
                                    {item.value} {item.unit ?? ''}
                                </div>
                                {FREE_LABEL}
                                <div className={style.subvalue}>
                                    {item.subtitle}
                                </div>
                            </div>;
                        })
                        : EMPTY_DATA
                    }

                    {selectedOffer?.detailed_description
                        ? <div className={style.item}>
                            <div className={`${style.subtitle} ${style.link_title}`}
                                 onClick={setOfferDetailsOpen.bind(null, true)}>
                                Остальные условия {LINK_LABEL}
                            </div>
                            <div className={style.subvalue}>
                                Детали по аренде, возврату и всё такое
                            </div>
                        </div>
                        : null
                    }
                </div>
            </div>

            <div className={style.info_block}>
                <div className={style.item}>
                    <div className={style.subtitle}>
                        Детализация
                    </div>
                    <table>
                        <tbody>
                            {prebill.map((item, index) => {
                                const isLast = index === prebill.length - 1;

                                return <tr key={item.type} className={isLast ? style.total : ''}>
                                    <td className={style.prebill_string}>
                                        <div>{item.title}</div>
                                        <div className={style.subvalue}>{item.details}</div>
                                    </td>
                                    <td className={style.prebill_string}>
                                        <div>{rubs(item.cost)}</div>
                                    </td>
                                </tr>;
                            })}
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        {carDetailsOpen
            ? <Window onClose={setCarDetailsOpen.bind(null, false)}
                      title={'Технические характеристики'}
                      className={style.details_window}>
                <iframe srcDoc={selectedOffer?.car_description} className={style.iframe}/>
            </Window>
            : null
        }

        {offerDetailsOpen
            ? <Window onClose={setOfferDetailsOpen.bind(null, false)}
                      title={'Детали оффера'}
                      className={style.details_window}>
                <iframe srcDoc={selectedOffer?.detailed_description} className={style.iframe}/>
            </Window>
            : null
        }

        {rentCalculateOpen
            ? <CalculateRentModal onWindowClose={setRentCalculateOpen.bind(null, false)}
                                  userId={userId}
                                  offer={selectedOffer}/>
            : null
        }

        {debugInfoOpen
            ? <JsonModal obj={props.debugData} onClose={setdebugInfoOpen.bind(null, false)}/>
            : null
        }
    </div>;
});

const CurrentOfferVisual = React.memo((props: any) => {
    return <div className={style.chosen_item}>
        <div>
            <img src={props.offer.offer_image} alt={props.offer.name} className={style.car_photo}/>
        </div>
        <div className={style.description}>
            <div className={style.description_item_name}>{props.offer.name}</div>
            <div className={style.description_item}>{props.offer.subname}</div>
        </div>
    </div>;
});

const Badge = (badgeName: string) => {
    return <div className={style.badge} key={badgeName}>{badgeName}</div>;
};

const FREE_LABEL = <div className={style.free}>✓ Включено в стоимость</div>;
const LINK_LABEL = <span className={style.link_label}>ᐳ</span>;
