import React from 'react';

import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import IDSelect from '../../../ui/IDSelect/component';
import { Input } from '../../../ui/Input';
import { NoInformation } from '../../../ui/NoInformation';
import Select, { IOptionInfo } from '../../../ui/Select';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { REQUESTS, SUPPORT_REQUESTS } from '../request';
import { OfferInfo } from './ChildComponents/OfferInfo';
import style from './index.css';
import {
    CITY_TAG,
    CLASS_TAG,
    IDebugInfoItem,
    ILongtermOffersData,
    IOfferInfo,
    LONGTERM_BUILDER,
    OFFERS_FILTERS,
} from './types';

interface ILongtermOffersState {
    isLoading: boolean;
    error: Error | null;
    data: ILongtermOffersData;
    offers: IOfferInfo[] | null;
    selectedOffer: IOfferInfo | null;

    appBuild: string;
    userId: string;

    filters: {
        newCar: boolean;
        city: number | null;
        carClass: number | null;
    };

    cityOptions: IOptionInfo[];
    classOptions: IOptionInfo[];
}

export class LongtermOffers extends React.Component<any, any> {
    state: ILongtermOffersState = {
        isLoading: false,
        error: null,
        data: {} as ILongtermOffersData,
        offers: null,
        selectedOffer: null,

        appBuild: '103789',
        userId: '',

        filters: {
            newCar: false,
            city: null,
            carClass: null,
        },

        cityOptions: [],
        classOptions: [],
    };

    request = new Request2({
        requestConfigs: SUPPORT_REQUESTS,
    });

    getInfo() {
        const headers: any = { IC_AppBuild: this.state.appBuild };

        if (this.state.userId) {
            headers.UserIdDelegation = this.state.userId;
        }

        this.setState({
            isLoading: true,
            error: null,
        }, () => {
            this.request.exec(REQUESTS.GET_OFFERS, {
                queryParams: {
                    offer_builder_type: LONGTERM_BUILDER,
                },
                headers,
            })
                .then(response => {
                    this.setState({
                        isLoading: false,
                        data: response,
                        cityOptions: this.getSelectOptions(response.filters ?? [], CITY_TAG),
                        classOptions: this.getSelectOptions(response.filters ?? [], CLASS_TAG),
                    }, () => this.filterItems());
                })
                .catch(error => {
                    this.setState({
                        data: null,
                        cityOptions: [],
                        isLoading: false,
                        error,
                    });
                });
        });
    }

    getSelectOptions(data, tag) {
        const currentTags = data.filter(el => el.parent_objects?.includes(tag));

        return currentTags.map(el => {
            return { value: el.id, text: el.text };
        });
    }

    onFilterChange(filter, value) {
        this.setState({
            filters: {
                ...this.state.filters,
                [filter]: value,
            },
        }, () => {
            filter !== 'userId' && this.filterItems();
        });
    }

    filterItems() {
        let offers = this.state.data.offers?.slice() ?? [];
        const { newCar, city, carClass } = this.state.filters;

        offers = offers.filter(offer => {
            const correctCity = city ? offer.filters.includes(city) : true;
            const correctClass = carClass ? offer.filters.includes(carClass) : true;
            const correctMileage = newCar ? offer.new : true;

            return correctCity && correctMileage && correctClass;
        });

        this.setState({ offers });
    }

    selectOffer(selectedOffer) {
        this.setState({ selectedOffer });
    }

    onUserIdChange(userId) {
        this.setState({ userId });
    }

    onInputChange(appBuild) {
        this.setState({ appBuild });
    }

    render() {
        const {
            isLoading,
            error,
            filters,
            selectedOffer,
            data,
            cityOptions,
            classOptions,
            userId,
            appBuild,
        } = this.state;
        const { newCar } = filters;

        const offersData = this.state.offers || [];
        const model = selectedOffer?.object_model ?? '';
        const modelInfo = data?.models?.[model];
        const disableFilters = !this.state.offers;
        const allDebugInfo = this.state.data?.debug_session_info?.long_term_autohide ?? [];
        const debugData: IDebugInfoItem = allDebugInfo.find(el => {
            return el.name === selectedOffer?.name || el.name === selectedOffer?.constructor_id;
        }) ?? {} as IDebugInfoItem;

        return <div>
            {selectedOffer
                ? <OfferInfo modelInfo={modelInfo}
                             selectedOffer={selectedOffer}
                             debugData={debugData}
                             userId={userId}
                             selectOffer={this.selectOffer.bind(this)}/>
                : <div>
                    <div className={style.filters_wrapper}>

                        <div className={style.filter_item}>
                            <IDSelect onSelect={this.onUserIdChange.bind(this)}
                                      containerClassName={style.filter}
                                      placeholder={'Пользователь'}/>
                        </div>

                        <div className={style.filter}>
                            <Input value={appBuild}
                                   onChange={this.onInputChange.bind(this)}
                                   required={true}
                                   placeholder={'App Build'}/>
                        </div>

                        <div className={style.filter_item}>
                            <Button onClick={this.getInfo.bind(this)}
                                    children={`Подобрать ${userId ? 'для пользователя' : ''}`}
                                    basic={true}
                                    disabled={!appBuild}/>
                        </div>

                        <div className={style.separator}/>

                        <Select options={cityOptions}
                                onSelect={this.onFilterChange.bind(this, OFFERS_FILTERS.city)}
                                placeholder={'Город'}
                                containerClassName={style.filter}
                                disabled={disableFilters}/>

                        <Select options={classOptions}
                                onSelect={this.onFilterChange.bind(this, OFFERS_FILTERS.carClass)}
                                placeholder={'Класс'}
                                containerClassName={style.filter}
                                disabled={disableFilters}/>

                        <div className={style.filter_item}>
                            <span className={style.black}>Новая: </span>
                            <Checkbox checked={newCar}
                                      disabled={disableFilters}
                                      onChange={this.onFilterChange.bind(this, OFFERS_FILTERS.newCar)}/>
                        </div>

                    </div>

                    {isLoading
                        ? <Spin/>
                        : error
                            ? <SimpleError error={error}/>
                            : offersData?.length
                                ? <div className={style.items_wrapper}>
                                    {offersData?.map((offer) => {
                                        return <OfferItem offer={offer}
                                                          key={offer.constructor_id}
                                                          onClick={this.selectOffer.bind(this, offer)}/>;
                                    })}
                                </div>
                                : <div className={style.no_info}><NoInformation/></div>
                    }
                </div>
            }
        </div>;
    }
}

const OfferItem = React.memo((props: any) => {
    return <div className={style.item} onClick={props.onClick?.bind(null)}>
        <div>
            <img src={props.offer.offer_image} alt={props.offer.name} className={style.car_photo}/>
        </div>
        <div className={style.description}>
            <div className={style.description_item}>{props.offer.name}</div>
            <div className={style.description_item}>{props.offer.subname}</div>
        </div>
    </div>;
});
