import * as React from 'react';
import { connect } from 'react-redux';

import { Dict } from '../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../constants';
import { Button, ButtonTypes } from '../../../ui/Button';
import DatePicker from '../../../ui/DatePicker';
import FormatDate from '../../../ui/FormatDate';
import { Confirm, Window } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import Select from '../../../ui/Select';
import * as styleTable from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import { Translate } from '../../../utils/translate';
import { makeObjPlain } from '../../../utils/utils';
import { IStore } from '../../App/store';
import { Copy } from '../../Copy';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import * as style from './index.css';
import { CORE_DRIVE_LENS_REQUESTS as requestConfigs, REQUESTS } from './request';

interface ISearchCallsProps extends IStore {
    userId: string;
}

interface ISearchCallsState {

}

export default class SearchCalls extends React.Component<ISearchCallsProps, ISearchCallsState> {
    state: ISearchCallsState = {};
    request = new Request2({
        requestConfigs,
    });
    t = this.props.Lang && new Translate(this.props.Lang) || {} as Translate;

    render() {
        return <div>
            <h3>Поиск обращений по номеру телефона</h3>
            <GetCalls/>
        </div>;
    }
}

const COMMON_LOADING = 'isCommonLoading';
const MORE_LOADING = 'isMoreLoading';

interface ICall {
    duration: number;
    duration_print: string;
    status: string;
    time_connect: number;
    time_enter: number;
    time_exit: number;
    request: { id: string; type: string; queue: string; origin: string };
    tags: ICallTags[];
    operators: { id: string; print_name: string; username: string }[];
    user: {
        id: string;
        is_phone_verified: boolean;
        phone: string;
        print_name: string;
        status: string;
        uid: number;
        username: string;
    };
    data: {
        message: string;
        track_url: string | null;
        chat_url: string | null;
    };
}

interface ICallTags {
    tag_id: string;
    tag_type: string;
    tag_description: string[];
    comment: string;
}

interface IGetCallsProps {
    phoneNumber?: string;
    fixedNumber?: boolean;
    userId?: string;
    agentIdFilter?: string;
    since?: any;
}

interface IGetCallsState {
    [key: string]: any;

    agentUsername: string;
    phoneNumber: string;
    data: ICall[] | null;
    [COMMON_LOADING]: boolean;
    error: any;
}

export class GetCalls extends React.Component <IGetCallsProps, IGetCallsState> {
    state = {
        phoneNumber: this.props.phoneNumber || '',
        agentUsername: '',
        agentIdFilter: this.props.agentIdFilter || '',
        userUidFilter: '',
        userIdFilter: this.props.userId || '',
        data: [],
        next_cursor: '',
        [COMMON_LOADING]: false,
        moreSettings: false,
        filters: {
            date: {
                since: this.props.since || '',
                until: '',
            },
        },
        error: null,
        [MORE_LOADING]: false,
    };
    request = new Request2({
        requestConfigs,
    });

    componentDidMount() {
        if (this.state.userIdFilter || this.props.agentIdFilter) {
            this.getData(COMMON_LOADING);
        }
    }

    componentDidUpdate(prevProps: Readonly<IGetCallsProps>): void {
        if (prevProps.phoneNumber !== this.props.phoneNumber) {
            this.setState({ phoneNumber: this.props.phoneNumber || '' });
        }
    }

    getData(loading: string) {
        this.setState({
            [loading]: true,
        }, () => {
            const data: Dict<any> = {
                agent_username: this.state.agentUsername,
                uid: this.state.userUidFilter,
                user_id: this.state.userIdFilter,
                agent_user_id: this.state.agentIdFilter,
                since: +this.state.filters.date.since / ONE_SECOND,
                until: +this.state.filters.date.until / ONE_SECOND,
                ...(loading === MORE_LOADING) && { next_cursor: this.state.next_cursor },
            };
            !this.props.userId && (data.phone_number = this.state.phoneNumber);

            this.request.exec(REQUESTS.GET_REQUEST_AGGREGATOR, { queryParams: data })
                .then(response => {
                    this.setState({
                        [loading]: false,
                        error: null,
                        data: loading === COMMON_LOADING ? response.data : this.state.data.concat(response.data),
                        next_cursor: response.next_cursor,
                    });
                })
                .catch((error) => {
                    this.setState({ error, [loading]: false });
                });
        });
    }

    onSearchInputChange(key: string, value: string) {
        this.setState({ [key]: value });
    }

    changeDate(type: string, value: string) {
        const filters: any = this.state.filters;
        filters.date[type] = value;
        this.setState({ filters });
    }

    changeMoreSettingShow() {
        this.setState({ moreSettings: !this.state.moreSettings });
    }

    isFilterDataSetInvalid() {
        const isAlone = (values: any[]) => values.filter(value => value).length <= 1;

        const isUserSetValid = isAlone([this.state.phoneNumber, this.state.userIdFilter, this.state.userUidFilter])
            || (isAlone([this.state.phoneNumber, this.state.userUidFilter])
                && isAlone([this.state.userIdFilter, this.state.userUidFilter])
                && this.props.userId);
        const isAgentSetValid = isAlone([this.state.agentUsername, this.state.agentIdFilter]);

        if (isUserSetValid && isAgentSetValid) {
            return '';
        }

        return <div className={style.invalid_alert}>Данные для поиска не валидны!
            {!isUserSetValid
            && <div>Только одно поле по пользователю: номера телефона, id пользователя, uid пользователя</div>}
            {!isAgentSetValid
            && <div>Только одно поле по оператору: логин оператора, id оператора</div>}
        </div>;
    }

    render() {
        const data: any = this.state.data;
        const invalidQuestion = this.isFilterDataSetInvalid();

        return <div>
            <div className={style.filter_container}>
                <div className={style.filters}>
                    <Input value={this.state.phoneNumber}
                           placeholder={'Номер телефона'}
                           disabled={this.props.fixedNumber}
                           className={style.filter_input}
                           onChange={this.onSearchInputChange.bind(this, 'phoneNumber')}/>
                    <Input value={this.state.agentUsername}
                           placeholder={'Оператор'}
                           className={style.filter_input}
                           onChange={this.onSearchInputChange.bind(this, 'agentUsername')}/>
                    <DatePicker placeholder={'От'}
                                className={style.filter_input}
                                value={this.state.filters.date.since}
                                onChange={this.changeDate.bind(this, 'since')}/>
                    <DatePicker placeholder={'До'}
                                className={style.filter_input}
                                value={this.state.filters.date.until}
                                onChange={this.changeDate.bind(this, 'until')}/>
                    <div className={style.filter_input}>
                        <Button disabled={this.state[COMMON_LOADING]}
                                onClick={this.getData.bind(this, COMMON_LOADING)}>Поиск</Button>
                    </div>
                </div>
                {!this.props.fixedNumber && <div>
                    <div className={style.settings_more}>
                        <Link onClick={this.changeMoreSettingShow.bind(this)}>
                            {!this.state.moreSettings ? 'Подробные настройки' : 'Скрыть'}
                        </Link>
                    </div>
                    {this.state.moreSettings
                    && <div className={style.filters}>
                        <Input value={this.state.userUidFilter}
                               placeholder={'UID Пользователя'}
                               className={style.filter_input}
                               onChange={this.onSearchInputChange.bind(this, 'userUidFilter')}/>
                        <Input value={this.state.userIdFilter}
                               placeholder={'ID Пользователя'}
                               className={style.filter_input}
                               onChange={this.onSearchInputChange.bind(this, 'userIdFilter')}/>
                        <Input value={this.state.agentIdFilter}
                               placeholder={'ID Оператора'}
                               className={style.filter_input}
                               onChange={this.onSearchInputChange.bind(this, 'agentIdFilter')}/>
                    </div>}
                </div>}
            </div>
            {invalidQuestion}
            {!this.state[COMMON_LOADING]
                ? this.state.error
                    ? <SimpleError error={this.state.error} data={{ label: 'Не удалось загрузить обращения' }}/>
                    : data
                        ? data.length
                        && <Calls2 calls={data}
                                   fixedNumber={!!(this.props.fixedNumber)}
                                   isCommonLoading={this.state[COMMON_LOADING]}
                                   isMoreLoading={this.state[MORE_LOADING]}
                                   nextCursor={this.state.next_cursor}
                                   update={this.getData.bind(this, COMMON_LOADING)}
                                   getMoreData={this.getData.bind(this, MORE_LOADING)}/>
                        || <h3>Обращений не найдено</h3>
                        : <h3>Нет данных</h3>
                : <Spin/>}
        </div>;
    }
}

interface IFilterTypes {
    [key: string]: boolean;

    phone: boolean;
    messenger: boolean;
    push: boolean;
    mail: boolean;
    other: boolean;
}

interface IFilter {
    types: IFilterTypes;
    topics: string;
}

interface ICallsState {
    filters: IFilter;
    isOpenedMoreInfoModal: boolean;
    currentCallIndex: number;

    showTags: boolean;
    userRequest: any;
    isTagLoading: boolean;
}

interface ICallsProps {
    filterTypes?: any;
    calls: ICall[];
    fixedNumber: boolean;
    isCommonLoading?: boolean;
    getMoreData: () => {};
    isMoreLoading: boolean;
    nextCursor?: any;
    update?: () => {};
    saasAuth?: any;
}

export class Calls extends React.Component <ICallsProps, ICallsState> {
    state = {
        filters: {
            types: this.props.filterTypes || {
                phone: true,
                messenger: true,
                push: true,
                mail: true,
                other: true,
            },
            topics: '',
        },
        isOpenedMoreInfoModal: false,
        currentCallIndex: -1,
        isCommonLoading: true,

        showTags: false,
        userRequest: '',
        isTagLoading: false,
    };
    filterTypes = {
        phone: { title: 'Телефон' },
        messenger: { title: 'Мессенджер' },
        push: { title: 'Пуш-уведомление' },
        mail: { title: 'Почта' },
        other: { title: 'Неизвестная очередь' },
    };

    isActiveTypeFilter(type: string): boolean {
        const types: IFilterTypes = this.state.filters.types;

        return types[type];
    }

    changeActiveTypeFilter(type: string): void {
        const filters: IFilter = this.state.filters;
        filters.types[type] = !filters.types[type];
        this.setState({ filters });
    }

    changeTopics(text: string) {
        const filters: IFilter = this.state.filters;
        filters.topics = text;
        this.setState({ filters });
    }

    getArticle(item: ICall) {
        const { tags, data = {} as { message: string } } = item;
        const { message = '' } = data;

        let article = '';
        if (tags.length) {
            article = tags.reduce((res: string[], curr: ICallTags) => {
                res.push(`${curr.tag_description.join(': ')}${curr.comment && `(${curr.comment})`}`);

                return res;
            }, []).join(', ');
        }

        if (message) {
            article = message;
        }

        return article;
    }

    openMoreInfoModal(currentCallIndex: number) {
        this.setState({ currentCallIndex, isOpenedMoreInfoModal: true });
    }

    closeMoreInfoModal() {
        this.setState({ currentCallIndex: -1, isOpenedMoreInfoModal: false });
    }

    filterCall(call: ICall) {
        const filterTypes: IFilterTypes = this.state.filters.types;

        const filterByQueue = (item: ICall) => {
            const type = item.request && item.request.type;

            return filterTypes.hasOwnProperty(type) ? filterTypes[type] : true;
        };

        const filterByArticle = (item: ICall) => {
            const article = this.getArticle(item);

            return this.state.filters.topics
                ? article.toLowerCase().includes(this.state.filters.topics.toLowerCase())
                : true;
        };

        return filterByQueue(call) && filterByArticle(call);
    }

    getFilterType(type: string) {
        const filterTypes: any = this.filterTypes;
        const filterType: any = filterTypes[type];

        return filterType && filterType.title;
    }

    getUserInfo(item: ICall) {
        const { id, username, print_name, status, phone } = item.user || {} as ICall;

        const displayName = print_name || username || phone || EMPTY_DATA;
        const link = (id && `#/clients/${id}/request_aggregator`) || null;

        return <div className={style.agent_username}>{
            link
                ? <a target={`_blank`}
                     href={link}>
                    {displayName}
                </a>
                : <span>
                    {displayName}
                </span>
        }
        {status && <div className={style.user_status}>({status})</div>}
        </div>;
    }

    showTag(userRequest: any, e: KeyboardEvent) {
        e.preventDefault();
        e.stopPropagation();
        this.setState({
            showTags: true,
            userRequest,
        });
    }

    onCloseTags() {
        this.setState({
            showTags: false,
            userRequest: '',
            isTagLoading: false,
        });
        this.props.update && this.props.update();
    }

    render() {
        if (this.props.isCommonLoading) {
            return <Spin/>;
        }

        const calls = this.props.calls && this.props.calls.length
            && this.props.calls.filter(this.filterCall.bind(this));

        const currentCallMore = calls && calls[this.state.currentCallIndex]
            && makeObjPlain(calls[this.state.currentCallIndex]);

        return <div className={style.calls_view}>
            {
                this.state.showTags
                && <RequestTags showTags={this.state.showTags}
                                userRequest={this.state.userRequest}
                                isLoading={this.state.isTagLoading}
                                onCloseTags={this.onCloseTags.bind(this)}/>
            }
            <div className={style.filter_container}>
                <div className={style.filters}>
                    {Object.keys(this.filterTypes).map(key => {
                        return <div key={key}
                                    title={this.getFilterType(key)}
                                    className={`${style.call_type} ${style[`type_${key}`]} ${this.isActiveTypeFilter(key) ? style.active : style.disable}`}
                                    onClick={this.changeActiveTypeFilter.bind(this, key)}>
                        </div>;
                    })}
                    <Input placeholder={'Тема'}
                           className={style.filter_input}
                           value={this.state.filters.topics}
                           onChange={this.changeTopics.bind(this)}/>
                </div>
            </div>
            {calls && calls.length
                ? <>
                    <table className={styleTable.table}>
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Тип</th>
                                <th>Дата</th>
                                <th/>
                                <th>Тема</th>
                                <th/>
                                <th>Оператор</th>
                                <th>Очередь</th>
                                {!this.props.fixedNumber && <th>Пользователь</th>}
                                <th/>
                            </tr>
                        </thead>
                        <tbody>
                            {
                                calls.map((item: ICall, index: number) => {
                                    const article = this.getArticle(item);
                                    const type = item.request && item.request.type || EMPTY_DATA;
                                    const queue = item.request && item.request.queue || EMPTY_DATA;

                                    const status = item.status
                                        ? <td className={style[`status_${item.status}`]}>
                                            {item.status}
                                        </td>
                                        : <td>{EMPTY_DATA}</td>;

                                    return <tr key={index}>
                                        <td>{index + 1}</td>
                                        <td>
                                            <div title={this.getFilterType(type)}
                                                 className={`${style[`type_${type}`]} ${style.req_icon}`}/>
                                        </td>
                                        <td className={style.enter_exit_time}>
                                            <div className={style.date}>
                                                <FormatDate value={item.time_enter * ONE_SECOND} onlyDate={true}/>
                                            </div>
                                            <div className={style.time}>
                                                <FormatDate value={item.time_enter * ONE_SECOND} onlyTime={true}/>
                                                {item.time_enter !== item.time_exit
                                            && <>
                                                -
                                                <FormatDate value={item.time_exit * ONE_SECOND} onlyTime={true}/>
                                            </>}
                                            </div>
                                        </td>
                                        <td>{item.duration_print || item.duration || EMPTY_DATA}</td>
                                        <td className={style.article}>
                                            {
                                                type === 'phone'
                                            && <a href={'#'}
                                                  className={!article && style.need_to_tagged || ''}
                                                  onClick={this.showTag.bind(this, item)}>{article || '[РАЗМЕТИТЬ]'}</a>
                                            || article
                                            }
                                        </td>
                                        {status}
                                        <td>
                                            {
                                                item.operators && item.operators.length
                                            && item.operators.map((operator: any, operatorIndex: number) => {
                                                return <div key={`${index}_${operatorIndex}_${operator.username}`}>
                                                    <a target={`_blank`}
                                                       className={style.agent_username}
                                                       href={`https://staff.yandex-team.ru/${operator.username}`}>
                                                        {operator.print_name
                                                            ? <>
                                                                <span className={style.agent_name}>
                                                                    {operator.print_name}
                                                                </span>
                                                            </>
                                                            : <span className={style.agent_name}>
                                                                {operator.username}
                                                            </span>}
                                                    </a>
                                                </div>;
                                            }) || EMPTY_DATA
                                            }
                                        </td>
                                        <td>{item.data && item.data.track_url
                                    && <a href={item.data.track_url}
                                          target={'_blank'}>{queue}</a>
                                    || item.data && item.data.chat_url
                                    && <a href={item.data.chat_url}
                                          target={'_blank'}>{queue}</a>
                                    || queue || EMPTY_DATA}</td>
                                        {!this.props.fixedNumber
                                    && <td>
                                        {this.getUserInfo(item)}
                                    </td>}
                                        <td>
                                            {
                                            //TODO access
                                            // !checkSaasRoles({
                                            //     saasAuth: this.props.saasAuth,
                                            //     checkingRole: [TEAM_ADM_LEVEL_1, TEAM_ADM_LEVEL_2, TEAM_ADM_LEVEL_3],
                                            //     exect: true
                                            // })
                                            // &&
                                                <span className={style.more}
                                                      onClick={this.openMoreInfoModal.bind(this, index)}>
                                    Подробнее
                                                </span>
                                            }

                                        </td>
                                    </tr>;
                                })
                            }
                        </tbody>
                    </table>
                    {this.props.nextCursor
                        ? <div className={style.more_calls_container}>
                            {this.props.isMoreLoading
                                ? <Spin/>
                                : <Link onClick={this.props.getMoreData.bind(null)}>Ещё</Link>
                            }
                        </div>
                        : null}
                </>
                : <h3>Обращений не найдено</h3>
            }
            {this.state.isOpenedMoreInfoModal
            && <Window onClose={this.closeMoreInfoModal.bind(this)}>
                <div className={style.more_info_call}>
                    <table>
                        <thead>
                            <tr>
                                <th>Ключ</th>
                                <th>Значение</th>
                            </tr>
                        </thead>
                        <tbody>
                            {calls
                        && currentCallMore && Object.entries(currentCallMore)
                                .sort((a, b) => {
                                    return a[0].localeCompare(b[0]);
                                }).map((entry: any) => {
                                    const TIME_KEY = 'time';
                                    const TRACK_URL_KEY = 'track_url';
                                    const CHAT_URL_KEY = 'chat_url';

                                    let [key, value] = entry;
                                    key.includes(TIME_KEY) && (value = <FormatDate value={value * ONE_SECOND}/>);
                                    key === `data.${TRACK_URL_KEY}` && value
                                && (value = <a href={value}
                                               title={'Прослушать звонок'}
                                               target={'_blank'}>Прослушать</a>);

                                    key === `data.${CHAT_URL_KEY}` && value
                                && (value =
                                    <a href={value || ''}
                                       target={'_blank'}>{value}</a>);

                                    return <tr key={key}>
                                        <td>{key}</td>
                                        <td>{value || EMPTY_DATA}</td>
                                    </tr>;
                                })}
                        </tbody>
                    </table>
                </div>
            </Window>}
        </div>;
    }
}

export const Calls2 = connect((store: IStore) =>
    ({ BlockRules: store.AdminUser && store.AdminUser.blockRules }), {})(Calls);

export class RequestTags extends React.Component<any, any> {
    render() {
        return <Window title={'Разметка звонка'} onClose={this.props.onCloseTags} className={'light-shadow'}>
            <_RequestTags {...this.props}/>
        </Window>;
    }
}

export class _RequestTags extends React.Component<any, any> {
    state: any = {
        categories: [],
        theme: [],
        tags: [],
        tag_id: [],
        comment: '',
        original_user_id: null,
        original_user_trait: this.props.userRequest && this.props.userRequest.phone || null,
        isLoading: false,
    };
    schema = {};
    _meta = {};
    request = new Request2({
        requestConfigs,
    });

    componentDidMount(): void {
        this.getCategories();
    }

    componentDidUpdate(prevProps: Readonly<any>): void {
        if (typeof this.props.userRequest === 'object' && !this.state.categories.length) {
            this.getCategories();
        }
    }

    shouldComponentUpdate(nextProps: Readonly<any>, nextState: Readonly<any>, nextContext: any): boolean {
        return this.props.showTags !== nextProps.showTags
            || this.props.isLoading !== nextProps.isLoading
            || JSON.stringify(this.props.userRequest) !== JSON.stringify(nextProps.userRequest)
            || JSON.stringify(this.state) !== JSON.stringify(nextState);
    }

    getCategories() {
        const props = this.props;

        if (props.userRequest
            && props.userRequest.request
            && props.userRequest.request.origin) {

            this.request.exec(REQUESTS.GET_REQUEST_TAGS_CATEGORIES,
                { queryParams: { request_origin: props.userRequest.request.origin, tag_type: 'new' } })
                .then(response => {
                    this.setState({
                        categories: response.data,
                    });
                });
        }

        this.getTags(props);
    }

    getTags(props: any) {
        props.userRequest
        && props.userRequest.request
        && this.request.exec(REQUESTS.GET_REQUEST_TAGS, {
            queryParams: {
                request_origin: props.userRequest.request.origin,
                request_id: props.userRequest.request.id,
            },
        })
            .then(response => {
                this.setState({
                    tags: response.tags,
                });
            });
    }

    onChange(type: string, value: any) {
        this.setState({ [type]: value });
    }

    submit() {
        if (!this.state.original_user_trait && !this.state.original_user_id) {
            this.setState({
                question: [
                    <div key={'0'}>Поля id/телефон не заполнены.</div>,
                    <div key={'1'}>Уверены, что хотите продолжить?</div>,
                ],
                isWorking: false,
                confirmIsOpen: true,
                accept: () => {
                    this.submitHandler();
                },
            });
        } else {
            this.submitHandler();
        }
    }

    submitHandler() {
        this.setState({
            isLoading: true,
        }, () => {
            const userRequest = this.props.userRequest || {};
            const tag = this.state.categories
                .filter((_cat: any) => _cat.id === this.state.tag_id)[0];
            this.request.exec(REQUESTS.GET_REQUEST_TAGS_CATEGORIES_ASSIGN, {
                body: {
                    tag_id: this.state.tag_id,
                    tag_type: tag && tag.type,
                    request_origin: userRequest.request && userRequest.request.origin,
                    request_id: userRequest.request && userRequest.request.id,
                    original_user_trait: this.state.original_user_trait || null,
                    original_user_id: this.state.original_user_id || null,
                    comment: this.state.comment,
                },
            })
                .then((response) => {
                    this.setState({
                        isLoading: false,
                        confirmIsOpen: false,
                        tags: [response.data.entry || {}, ...this.state.tags],
                    });
                    this.props.onCloseTags();
                })
                .catch(() => {
                    this.setState({
                        isLoading: false,
                    });
                    this.props.onCloseTags();
                });
        });
    }

    onCloseModal() {
        this.setState({
            confirmIsOpen: false,
        });
    }

    removeTag(tag: any, e: KeyboardEvent) {
        e.stopPropagation();
        e.preventDefault();
        this.setState({
            question: [
                <div key={'0'}>Удалить тег </div>,
                <div key={'1'}>{tag.tag_description && tag.tag_description.join(':') || tag.tag_description}?</div>,
            ],
            isWorking: false,
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    const { entry_id } = tag;
                    this.request.exec(REQUESTS.REMOVE_REQUEST_TAGS, { body: { entry_id } })
                        .then(() => {
                            this.setState({
                                isWorking: false,
                                confirmIsOpen: false,
                                tags: this.state.tags.filter((_t: any) => _t.entry_id !== entry_id),
                            });
                        })
                        .catch(() => {
                            this.setState({
                                isWorking: false,
                            });
                        });
                });
            },
            confirmIsOpen: true,
        });
    }

    openUser(href: string) {
        location.href = href;
    }

    render() {
        const phone = this.props.userRequest && this.props.userRequest.phone;
        const user = this.props.userRequest && this.props.userRequest.user && this.props.userRequest.user.print_name;
        const user_id = this.props.userRequest && this.props.userRequest.user && this.props.userRequest.user.id;

        const themes = (this.state.categories && this.state.categories
            .reduce((_p: any[], _c: any) => {
                if (!_p.includes(_c.description[0])) {
                    _p.push(_c.description[0]);
                }

                return _p;
            }, [])
            || [])
            .map((_el: any) => {
                return { text: _el, value: _el };
            });

        const items = this.state.categories && this.state.categories.reduce((_p: any, _c: any) => {
            if (Array.isArray(this.state.theme) && !this.state.theme.length || !this.state.theme) {
                _p.push({ text: _c.description[1], value: _c.id, description: _c.description[0] });
            }

            if (this.state.theme && _c.description[0] === this.state.theme) {
                _p.push({ text: _c.description[1], value: _c.id });
            }

            return _p;
        }, []) || [];

        return <div className={'request-sidebar'}>
            <h4><Copy>{phone}</Copy> {user
            && <a href={`#/clients/${user_id}/info`}
                  onClick={this.openUser.bind(this, `#/clients/${user_id}/info`)}>{user}</a>}
            </h4>
            {
                this.props.isLoading
                    ? <Spin/>
                    : <div>
                        <div className={'tree_filter'}>
                            <TreeSuggest title={'Эксперимент!!! Фильтр / Дерево'}
                                         onSelect={this.onChange.bind(this, 'tag_id')}/>
                        </div>
                        <Select options={themes}
                                className={'control-element'}
                                placeholder={'Тема'}
                                onSelect={this.onChange.bind(this, 'theme')}
                                initialValues={this.state.theme}/>
                        <Select options={items}
                                className={'control-element'}
                                placeholder={'Тег'}
                                onSelect={this.onChange.bind(this, 'tag_id')}
                                initialValues={this.state.tag_id}/>
                        <Input value={this.state.comment}
                               placeholder={'Комментарий'}
                               className={'control-element'}
                               onChange={this.onChange.bind(this, 'comment')}/>

                        <Input value={this.state.original_user_id}
                               placeholder={'id-пользователя'}
                               className={'control-element'}
                               onChange={this.onChange.bind(this, 'original_user_id')}/>

                        <Input value={this.state.original_user_trait}
                               placeholder={'телефон'}
                               className={'control-element'}
                               onChange={this.onChange.bind(this, 'original_user_trait')}/>

                        <Button colorType={ButtonTypes.negative}
                                onClick={this.props.onCloseTags.bind(this)}
                                disabled={this.state.confirmIsOpen}>Закрыть</Button>
                        <Button disabled={this.state.isLoading || this.state.confirmIsOpen}
                                onClick={this.submit.bind(this)}>Классифицировать</Button>
                        {
                            !this.props.withoutTags
                            && <div className={'tbl-requests-tags'}>
                                <h4>Теги</h4>
                                {this.state.tags.length
                                    ? <table className={'tbl-order request-tags'}>
                                        <tbody>
                                            {
                                                this.state.tags.map((_tag: any, _i: number) => {
                                                    return <tr key={_i}>
                                                        <td>{_i + 1}</td>
                                                        <td>
                                                            <FormatDate value={_tag.submitted_at * ONE_SECOND}
                                                                        withSecond={true}/>
                                                        </td>
                                                        <td>
                                                            {_tag.tag_description && _tag.tag_description.join(':')}
                                                        </td>
                                                        <td>{_tag.comment}</td>
                                                        <td>{_tag.performer && _tag.performer.print_name}</td>
                                                        <td><a href={'#'}
                                                               onClick={this.removeTag.bind(this, _tag)}>Удалить</a>
                                                        </td>
                                                    </tr>;
                                                })
                                            }
                                        </tbody>
                                    </table>
                                    : <div>нет тегов</div>
                                }
                            </div>
                        }
                    </div>
            }
            {
                this.state.confirmIsOpen
                && <Confirm isWorking={this.state.isWorking}
                            accept={this.state.accept.bind(this)}
                            question={this.state.question}
                            error={null}
                            onClose={this.onCloseModal.bind(this)}/>
            }
        </div>;
    }
}

class TreeSuggest extends React.Component<any, any> {
    state: any = {
        filter: '',
        suggestIsOpen: false,
        isLoading: false,
        data: null,
        error: null,
    };
    request = new Request2({
        requestConfigs,
    });
    timer: any;

    onChange(filter: string) {
        const TIMEOUT = 200;
        this.setState({ filter }, () => {
            clearTimeout(this.timer);
            this.timer = setTimeout(() => {
                this.getTree(false);
            }, TIMEOUT);
        });
    }

    onSelect(item: any) {
        this.setState({
            filter: item.meta_info.labels && item.meta_info.labels.ru,
        }, () => {
            this.props.onSelect(item.id);
            this.close();
        });
    }

    close() {
        this.setState({
            suggestIsOpen: false,
        });
    }

    onFocus() {
        this.setState({
            suggestIsOpen: true,
        }, () => {
            this.getTree(true);
        });
    }

    drawTree(data: any, opt: any = {}) {
        const _data = [...data];
        _data.sort((a: any, b: any) => {
            return a.meta_info.order - b.meta_info.order;
        });

        return <>{
            _data
                .map((item: any, index: number) => {
                    const indexLabel = opt.index ? `${opt.index}.${++index}` : ++index;

                    return <>
                        <div className={'tree_suggest_item'}
                             onClick={this.onSelect.bind(this, item)}
                             key={indexLabel}>
                            <span>{indexLabel})</span> {item.meta_info.labels && item.meta_info.labels.ru || EMPTY_DATA}
                        </div>
                        <div className={'children'}>
                            {item.children.length
                                ? this.drawTree(item.children, { index: indexLabel })
                                : null}
                        </div>
                    </>;
                })
        }</>;
    }

    getTree(withLoader?: boolean) {
        this.setState({
            isLoading: withLoader,
        }, () => {
            this.request.exec(REQUESTS.GET_CLASSIFICATION_TREE, { queryParams: { filter: this.state.filter } })
                .then(response => {
                    this.setState({
                        isLoading: false,
                        error: null,
                        data: response || {},
                    });
                })
                .catch(err => {
                    this.setState({
                        isLoading: false,
                        error: err,
                        data: null,
                    });
                });
        });
    }

    render() {
        return <div>
            <div tabIndex={0} onFocus={this.onFocus.bind(this)}>
                <Input value={this.state.filter}
                       placeholder={this.props.title}
                       onChange={this.onChange.bind(this)}/>
            </div>
            {
                this.state.suggestIsOpen
                    ? <>
                        <span className={`close ${this.state.suggestIsOpen ? 'suggest_is_open' : ''}`}
                              onClick={this.close.bind(this)}>X</span>
                        <div className={'tree_suggest_block'}>
                            {
                                this.state.isLoading
                                    ? <Spin size={'l'}/>
                                    : this.state.error
                                        ? <SimpleError data={{ label: 'Tree' }} error={this.state.error}/>
                                        : <div>
                                            {this.state.data
                                        && this.state.data.children
                                        && Array.isArray(this.state.data.children)
                                        && this.state.data.children.length
                                        && this.drawTree(this.state.data.children)
                                        || <h3>Нет записей</h3>
                                            }
                                        </div>
                            }
                        </div>
                    </>
                    : null
            }
        </div>;
    }
}
