import produce, { Draft } from 'immer';
import React from 'react';
import Scrollbars from 'react-custom-scrollbars';

import { ISearchTags, ITagItem, ITagItemDescription } from '../../../../types';
import { EMPTY_DATA, TIMERS } from '../../../constants';
import { BasicUserInfo, UserInfoHandler } from '../../../models/user';
import FormatDate from '../../../ui/FormatDate';
import { Confirm } from '../../../ui/FullModal';
import { Link } from '../../../ui/Link';
import { LabelStatus, TLabel } from '../../../ui/Table';
import * as tblStyle from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { REQUESTS, SUPPORT_REQUESTS } from '../request';
import * as style from './index.css';

enum ALERT_STATUS {
    ACTIVE = 'active',
    MUTE = 'mute'
}

interface IState {
    isLoading: boolean;
    data: ISearchTags | null;
    error: any;
    lastUpdate: string;

    showConfirm: boolean;
    actionError: any;
    question: string;
    isWorking: boolean;
    accept: () => void;
}

export class SupportAlerts extends React.Component<any, IState> {
    state = {
        isLoading: false,
        data: {} as ISearchTags,
        error: null,
        lastUpdate: new Date().toLocaleString(),

        showConfirm: false,
        actionError: null,
        question: '',
        isWorking: false,
        accept: () => {
        },
    };
    request = new Request2({ requestConfigs: SUPPORT_REQUESTS });
    timeout: any;

    perform(tag_id: string) {
        this.setState(produce(this.state, (draft: Draft<IState>) => {
            draft.showConfirm = true;
            draft.actionError = null;
            draft.question = 'Взять тег в работу?';
            draft.isWorking = false;
            draft.accept = () => {
                this.setState(produce(this.state, draft => {
                    draft.isWorking = true;
                    this.request.exec(REQUESTS.SET_PERFORM, {
                        body: {
                            tag_id,
                        },
                    },
                    )
                        .then(resp => {
                            this.setState(produce(this.state, draft => {
                                draft.actionError = null;
                                draft.isWorking = false;
                                draft.showConfirm = false;
                            }));
                            this.getData(false);
                        })
                        .catch(actionError => {
                            this.setState(produce(this.state, draft => {
                                draft.actionError = actionError;
                                draft.isWorking = false;
                            }));
                        });
                }));
            };
        }));
    }

    changeState(state: ALERT_STATUS, tag) {
        this.setState(produce(this.state, (draft: Draft<IState>) => {
            draft.showConfirm = true;
            draft.actionError = null;
            draft.question = `Перевести состояние тега в [${state}]`;
            draft.isWorking = false;
            draft.accept = () => {

                this.setState(produce(this.state, draft => {
                    draft.isWorking = true;
                    const body = Object.assign({}, tag, { status: state });
                    delete body['performer'];
                    this.request.exec(REQUESTS.UPDATE_TAG, {
                        body,
                    })
                        .then(() => {
                            this.getData(false);
                            this.setState(produce(this.state, draft => {
                                draft.actionError = null;
                                draft.isWorking = false;
                                draft.showConfirm = false;
                            }));
                        })
                        .catch((actionError) => {
                            this.setState(produce(this.state, draft => {
                                draft.actionError = actionError;
                                draft.isWorking = false;
                            }));
                        });
                }));
            };
        }));
    }

    componentWillUnmount(): void {
        clearTimeout(this.timeout);
    }

    runCheck() {
        clearTimeout(this.timeout);
        this.timeout = setTimeout(() => {
            this.getData(false);
        }, TIMERS.CHECK_SUPPORT_ALERTS);
    }

    getData(isLoading = true) {
        this.setState(produce(this.state, (draft: Draft<IState>) => {
            draft.isLoading = isLoading;
        }), () => {
            this.request.exec<ISearchTags>(REQUESTS.GET_TAGS)
                .then(response => {
                    this.setState(produce(this.state, (draft: Draft<IState>) => {
                        draft.isLoading = false;
                        draft.data = response || {};
                        draft.error = null;
                        draft.lastUpdate = new Date().toLocaleString();
                    }));
                    this.runCheck();
                })
                .catch(error => {
                    this.setState(produce(this.state, (draft: Draft<IState>) => {
                        draft.isLoading = false;
                        draft.data = {} as ISearchTags;
                        draft.error = error;
                        draft.lastUpdate = new Date().toLocaleString();
                    }));
                    this.runCheck();
                });
        });
    }

    componentDidMount(): void {
        this.getData();
    }

    closeConfirm(showConfirm: boolean) {
        this.setState(produce(this.state, draft => {
            draft.showConfirm = !showConfirm;
        }));
    }

    render() {
        const report = this.state.data
            && this.state.data.results
            && this.state.data.results[0]
            && this.state.data.results[0].report || [];
        const tag_descriptions = this.state.data
            && this.state.data.tag_descriptions;

        return <>
            {
                this.state.showConfirm && <Confirm error={this.state.actionError}
                                                   question={this.state.question}
                                                   isWorking={this.state.isWorking}
                                                   accept={this.state.accept && this.state.accept}
                                                   onClose={this.closeConfirm.bind(this, true)}/>
            }
            <div>
                <strong>Последнее обновление:</strong> <FormatDate value={this.state.lastUpdate} withSecond/>,
                (записей: <strong>{report.length}</strong>)
            </div>
            {
                this.state.isLoading
                    ? <Spin/>
                    : this.state.error
                        ? <SimpleError error={this.state.error}/>
                        : <Data report={report}
                                perform={this.perform.bind(this)}
                                changeState={this.changeState.bind(this)}
                                tag_descriptions={tag_descriptions}/>
            }
        </>;
    }
}

interface ICommon {
    perform: () => void;
    changeState: () => void;
}

interface TrTagItem extends ICommon {
    item: BasicUserInfo;
    index: number;
    tagMap: { [key: string]: { displayName: string; comment: string } };
}

interface IData extends ICommon {
    report: BasicUserInfo[];
    tag_descriptions: ITagItemDescription[];
}

const Data = React.memo((props: IData) => {
    const { report, changeState, perform, tag_descriptions } = props;

    const tagMap = tag_descriptions
        && Array.isArray(tag_descriptions)
        && tag_descriptions.reduce((_p, _c) => {
            return { ..._p, [_c.name]: { displayName: _c.display_name, comment: _c.comment } };
        }, {}) || {};

    return <Scrollbars style={{ maxHeight: 'calc(100vh - 187px)' }} autoHide>
        <table className={tblStyle.table}>
            <thead>
                <tr className={'header'}>
                    <th>#</th>
                    <th>имя</th>
                    <th>тег</th>
                    <th>перформ</th>
                    <th>статус</th>
                    <th/>
                </tr>
            </thead>
            <tbody>
                {
                    report
                        .map((item, key) => <TrTagItem item={item}
                                                       key={key}
                                                       index={key}
                                                       tagMap={tagMap}
                                                       changeState={changeState}
                                                       perform={perform}/>,
                        )
                }
            </tbody>
        </table>
    </Scrollbars>;
});

const TrTagItem = React.memo((props: TrTagItem) => {
    const { perform, changeState, item, index, tagMap } = props;
    const uih = UserInfoHandler;
    const href = (id) => `${location.origin}${location.pathname}#/clients/${id}/info`; //
    const tags = uih.getTags.call(item) as ITagItem[];
    const tagStatus = tags.length && tags[0].status || EMPTY_DATA;
    const tagName = tags.length && tags[0].tag;
    const tagId = tags.length && tags[0].tag_id;
    const performer = tags.length && tags[0].performer;
    const { displayName, comment } = tagMap[tagName];

    return <tr>
        <td>
            {index + 1}
        </td>
        <td>
            <div>
                <Link href={href(uih.getId.call(item))} target={'_blank'}>{uih.getPrintName.call(item)}</Link>
            </div>
            <div className={style.comment}>{uih.getPhone.call(item)} [{uih.getStatus.call(item)}]</div>
        </td>
        <td>
            <div className={style.display_name}>
                {displayName}
            </div>
            <div className={style.comment}>{comment}</div>
        </td>
        <td>{performer ? <Link href={href(performer)} target={'_blank'}>исполнитель</Link> : EMPTY_DATA}</td>
        <td>
            <TLabel text={tagStatus}
                    status={tagStatus === ALERT_STATUS.ACTIVE ? LabelStatus.POSITIVE : LabelStatus.NEGATIVE}/>
        </td>
        <td>
            <Link className={style.link}
                  onClick={perform.bind(null, tagId)}
                  disable={!!performer}>в работу</Link>
            <Link className={style.link}
                  onClick={changeState.bind(null, ALERT_STATUS.ACTIVE, tags[0])}
                  disable={tagStatus === ALERT_STATUS.ACTIVE}>старт</Link>
            <Link className={style.link}
                  onClick={changeState.bind(null, ALERT_STATUS.MUTE, tags[0])}
                  disable={tagStatus === ALERT_STATUS.MUTE}>пауза</Link>
        </td>
    </tr>;
});
