import * as React from 'react';

import { DAYS_LABEL } from '../../../constants';
import Checkbox from '../../../ui/Checkbox';
import { Input } from '../../../ui/Input';
import * as style from './index.css';

interface IDistributionTable {
    schedule: number[][][];
    dayNow: number;
    timeNow: number;
    activeDay: number;
    activeTime: number;

    setActiveRange(activeDay: number, activeTime: number): void;

    setFocus(cell: HTMLTableDataCellElement, indexDay: number, indexTime: number): void;
}

export const DistributionTables = (props: IDistributionTable) => {
    const { schedule, dayNow, timeNow, activeDay, activeTime, setActiveRange, setFocus } = props;

    return <table className={style.table}>
        <thead className={style.table_head}>
            <tr>
                <th className={`${style.table_head__cell} ${style.fixed}`}>День</th>
                {schedule?.[0]?.map((_, index) =>
                    <th key={index} className={style.table_head__cell}>{index}:00</th>)}
            </tr>
        </thead>
        <tbody>
            {schedule.length && schedule.map((day, indexDay) =>
                <tr key={indexDay} className={style.column}>
                    <td className={`${style.table_head__cell} ${style.fixed}`}>
                        {DAYS_LABEL[indexDay]}
                    </td>
                    {
                        day?.map((hours, indexTime) =>
                            <td tabIndex={0}
                                className={`${style.cell} 
                                        ${indexDay === dayNow || indexTime === timeNow ? style.cell_now : null}
                                        ${indexDay === activeDay
                                        && indexTime === activeTime ? style.cell_active : null}`}
                                ref={setFocus.bind(null, indexDay, indexTime)}
                                key={indexTime}
                                onClick={setActiveRange.bind(null, indexDay, indexTime)}>
                                {
                                    hours.join('/')
                                }
                            </td>)
                    }
                </tr>)}
        </tbody>
    </table>;
};

interface IActiveRangeInputs {
    companies: string[];
    range: number[];
    setNewRange: (range: number[]) => void;
}

const RANGE_LIMIT = 100;

const useInput = (initialValue: number, index: number, setNewRange, range, autoFillMode) => {
    const [value, setValue] = React.useState(initialValue);
    const onChange = value => {
        setValue(value);
    };

    React.useEffect(() => {
        const newRange = range;
        if(autoFillMode) {
            if(index === 0) {
                newRange[index] = value;
                newRange[range.length - 1] = RANGE_LIMIT - value;
            }

            if(index === range.length - 1) {
                newRange[0] = RANGE_LIMIT - value;
                newRange[index] = value;
            }

            if(index !== 0 && index !== range.length - 1) {
                newRange[index] = value;
            }
        } else {
            newRange[index] = value;
        }

        setNewRange(newRange);
    }, [value]);

    React.useEffect(() => {
        setValue(initialValue);
    }, [initialValue]);

    return {
        value,
        onChange,
    };
};

export const ActiveRangeInputs = (props: IActiveRangeInputs) => {
    const { companies, range, setNewRange } = props;
    const [autoFillMode, setAutoFillMode] = React.useState(true);

    return <div className={style.rangeInfo}>
        <p className={style.checkboxInfo}>Взаимное заполнение AUDIOTELE и YANDEX</p>
        <Checkbox checked={autoFillMode}
                  onChange={setAutoFillMode}
                  className={style.checkbox}/>
        <span>Нагрузка, %</span>
        {companies?.map((company, index) =>
            <Input {...useInput(range?.[index], index, setNewRange, range, autoFillMode)}
                   placeholder={company}
                   key={index}
                   className={style.inputBlock}
                   type={'number'}
                   disabled={autoFillMode && company === 'NEXTCONTACT' ? true : false}/>)
        }
    </div>;
};
