import * as React from 'react';

import { CITIES, ONE_HUNDRED_PERCENT } from '../../../constants';
import { Button } from '../../../ui/Button';
import { Confirm } from '../../../ui/FullModal';
import { Tabs } from '../../../ui/Tabs';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { REQUESTS, SUPPORT_REQUESTS } from '../request';
import { ActiveRangeInputs, DistributionTables } from './DistributionTables';
import * as style from './index.css';

const compare = (a: any, b: any): number => a?.[0].localeCompare(b?.[0]);
const TABS = Object.entries(CITIES)?.map(city => ({ name: city[1].name, link: city[1].short_name_en }));

type IDistributionValue = {
    [key: string]: {
        [key: string]: {
            [key: string]: number;
        };
    };
};

interface ISupportDistributionProps {
}

interface ISupportDistributionState {
    schedule: number[][][];
    city: string;
    companies: string[];
    activeDay: number;
    activeTime: number;
    dayNow: number;
    timeNow: number;
    isLoading: boolean;
    isUpdating: boolean;
    isOpenUpdateConfirm: boolean;
    loadingError: Error | null;
    validated: boolean;
    rangeSum: number;
}

export default class SupportDistribution extends React.Component<ISupportDistributionProps, ISupportDistributionState> {
    state: ISupportDistributionState = {
        companies: [],
        activeDay: INIT_WEEK_DAY(),
        activeTime: new Date().getHours(),
        dayNow: INIT_WEEK_DAY(),
        timeNow: new Date().getHours(),
        schedule: {} as number[][][],
        city: CITIES?.msc_area?.short_name_en,
        isLoading: false,
        isUpdating: false,
        isOpenUpdateConfirm: false,
        loadingError: null,
        validated: true,
        rangeSum: 0,
    };

    request = new Request2({ requestConfigs: SUPPORT_REQUESTS });
    activeCell: HTMLTableDataCellElement;

    componentDidMount() {
        this.getDistributionValues();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getDistributionValues() {
        this.setState({ isLoading: true, loadingError: null }, () => {
            this.request.exec(REQUESTS.GET_LOAD_BALANCE, { queryParams: { city: this.state.city } })
                .then(response => {
                    const { data = {} }: IDistributionValue = response || {};

                    const schedule = Object.entries(data)
                        .sort(compare)
                        .map(day => Object.entries(day?.[1])
                            .sort(compare)
                            .map((hour) => Object.entries(hour?.[1])
                                .sort(compare)
                                .map(company => company?.[1]),
                            ));

                    const companies = Object.keys(Object.entries(data?.[0])?.[0]?.[1]).sort(compare);

                    this.setState({
                        isLoading: false,
                        schedule,
                        companies,
                    }, () => {
                        this.activeCell?.focus();
                    },
                    );
                })
                .catch((loadingError) => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    changeCity(city: string) {
        this.setState({ city }, () => {
            this.getDistributionValues();
        });
    }

    handleChange(range: number[]) {
        const { schedule, companies } = this.state;
        const rangeSum = range.reduce((acc, curVal) => {
            return acc + curVal;
        }, 0);

        if (rangeSum !== ONE_HUNDRED_PERCENT) {
            this.setState({
                validated: false,
                rangeSum: rangeSum,
            });
        } else {
            this.setState({
                validated: true,
            });
        }

        companies?.map((company, index) => {
            schedule[this.state.activeDay][this.state.activeTime][index] = range?.[index];
        },
        );
        this.setState({ schedule });
    }

    toggleConfirm(isOpenUpdateConfirm: boolean): void {
        this.setState({ isOpenUpdateConfirm });
    }

    setActiveRange(activeDay: number, activeTime: number) {
        if (this.state.validated) {
            this.setState({ activeDay, activeTime });
        }
    }

    setFocus(indexDay: number, indexTime: number, cell: HTMLTableDataCellElement): void {
        if (indexDay === this.state.activeDay && indexTime === this.state.activeTime) {
            this.activeCell = cell;
        }
    }

    setDistributionValues() {
        this.setState({ isUpdating: true, isOpenUpdateConfirm: false }, () => {
            const data = this.state.schedule?.reduce((_p, _c, _i) => {
                const h = _c.reduce((__p, __c, __i) => {
                    const SMALLEST_TWO_DIGIT_NUMBER = 10;
                    const _h = __i < SMALLEST_TWO_DIGIT_NUMBER ? `0${__i}` : `${__i}`;

                    return {
                        ...__p, ...{
                            [_h + ':00']: __c.reduce((___p, ___c, ___i) => {
                                return { ...___p, ...{ [this.state.companies[___i]]: ___c } };
                            }, {}),
                        },
                    };
                }, {});

                return { ..._p, ...{ [_i]: h } };
            }, {});

            this.request.exec(REQUESTS.SET_LOAD_BALANCE, {
                body: { data, city: this.state.city },
            })
                .then(() => {
                    this.setState({ isUpdating: false });
                },
                )
                .catch((loadingError: Error) => {
                    this.setState({ loadingError, isUpdating: false });
                });
        });
    }

    render() {
        return <div className={style.distribution}>
            <div className={style.header}>
                <h2 className={style.title}>Распределение нагрузки запросов в поддержку</h2>
                <Tabs tabs={TABS} currentTab={this.state.city} selectTab={this.changeCity.bind(this)}/>
            </div>

            {this.state.isLoading
                ? <Spin/>
                : this.state.loadingError
                    ? <SimpleError error={this.state.loadingError} data={{ label: 'Ошибка при загрузке данных' }}/>
                    : <>
                        {
                            this.state.isOpenUpdateConfirm &&
                            <Confirm question={'Обновить данные о распределении нагрузки?'}
                                     error={this.state.loadingError}
                                     onClose={this.toggleConfirm.bind(this, false)}
                                     accept={this.setDistributionValues.bind(this)}/>
                        }
                        <div className={style.info}>
                            <div className={style.submitBlock}>
                                <Button onClick={this.toggleConfirm.bind(this, true)}
                                        disabled={!this.state.validated}>
                                    Обновить
                                </Button>
                                {!this.state.validated && <span className={style.errorMsg}>
                                    Сумма диапозонов равна {this.state.rangeSum}
                                </span>}
                            </div>
                            <ActiveRangeInputs companies={this.state.companies}
                                               range={this.state.schedule?.[this.state.activeDay]
                                                   ?.[this.state.activeTime] || []}
                                               setNewRange={this.handleChange.bind(this)}/>
                        </div>
                        <div className={style.wrapper}>
                            <div className={style.scroller}>
                                <DistributionTables schedule={this.state.schedule}
                                                    dayNow={this.state.dayNow}
                                                    timeNow={this.state.timeNow}
                                                    activeDay={this.state.activeDay}
                                                    activeTime={this.state.activeTime}
                                                    setActiveRange={this.setActiveRange.bind(this)}
                                                    setFocus={this.setFocus.bind(this)}/>
                            </div>
                        </div>
                    </>}
        </div>;
    }
}

const SUNDAY_INDEX = 6;

const INIT_WEEK_DAY = () => {
    return new Date().getDay() ? new Date().getDay() - 1 : SUNDAY_INDEX;
};
