import React from 'react';

import { Dict, ICar, LSSettingItems, TAG_ACTION } from '../../../../types';
import { CITIES, ONE_SECOND } from '../../../constants';
import { TagRecord, TagRecordHandler } from '../../../models/tag';
import { Button, ButtonTypes } from '../../../ui/Button';
import FormatDate from '../../../ui/FormatDate';
import { TabItem, Tabs } from '../../../ui/Tabs';
import { CacheHelper } from '../../../utils/cache';
import LS from '../../../utils/localStorage/localStorage';
import { Request2 } from '../../../utils/request';
import QueryScheduler, { COUNT_FIRST } from '../../QueryScheduler/QueryScheduler';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { REQUESTS, SUPPORT_REQUESTS } from '../request';
import * as styles from './index.css';
import { TableCars } from './TableCars/component';
import { TRAITS } from './types';

export interface IRichCarEvac extends ICar {
    incorrectMovingTime?: number;
    current_tags?: Dict<string>[];
    tagDate?: number[];
    taskLink?: string;
    authorLink?: string;
}

interface IState {
    tabs: TabItem[];
    msk: IRichCarEvac[];
    spb: IRichCarEvac[];
    kzn: IRichCarEvac[];
    sochi: IRichCarEvac[];
    currentCarsService: Dict<IRichCarEvac[]>;
    currentCarsOther: IRichCarEvac[];
    isLoading: boolean;
    error: Error | null;
    hasOldData: boolean;
    newCacheAvailable: boolean;
    currentTab: string;
    cacheTimestamp?: number;
}

enum CarStatus {
    SERVICE = 'service',
}

const EVACUATION_LOC_TAG = 'impoundment_lot';
const EVACUATION_TAG = 'evacuation';
const POSSIBLE_EVACUATION_TAG = 'possible_evacuation';
const INCORRECT_MOVING_TAG = 'incorrect_moving';
const ANTITRAITS = 'ReportTags,ReportModels';
const CITIES_KEYS = Object.keys(CITIES);
const DEFAULT_TAB = CITIES[CITIES_KEYS[0]].short_name_en;

const stLinkPattern = /https:\/\/st.yandex-team.ru\/[A-za-z]+-[0-9]+/g;

const urlPattern = `#/support/probably-evacuated?city=`;
const REQUEST_KEY = `${SUPPORT_REQUESTS[REQUESTS.GET_CARS].api}_${TRAITS}_A`;

export class SupportEvacuations extends React.Component<any, IState> {
    state: IState = {
        tabs: [],
        msk: [],
        spb: [],
        kzn: [],
        sochi: [],
        currentCarsService: { 'unsorted': [] },
        currentCarsOther: [],
        isLoading: false,
        error: null,
        hasOldData: false,
        newCacheAvailable: false,
        currentTab: '',
    };

    cache = new CacheHelper({
        id: REQUEST_KEY,
        reduceResponse: this.reduceResponse.bind(this),
        onResponse: this.onResponse.bind(this),
        onCache: this.onCache.bind(this),
        onError: this.setError.bind(this),
    });

    request = new Request2({ requestConfigs: SUPPORT_REQUESTS });
    ls = new LS();

    cars: ICar[] = [];
    count = 0;

    setError(error) {
        this.setState({
            error,
            isLoading: false,
        });
    }

    onResponse(response) {
        this.fetchCars(response)
            .then(() => {
                if (this.state.hasOldData) {
                    this.setState({
                        hasOldData: false,
                        newCacheAvailable: true,
                    });
                } else {
                    this.sortCars();
                }

                this.setState({
                    cacheTimestamp: +Date.now(),
                });
            });
    }

    reduceResponse(response) {
        const cars: ICar[] = response?.cars;
        if (cars) {
            return cars.filter((car) => {
                return car.location && car.location.tags && car.location.tags.includes(EVACUATION_LOC_TAG);
            });
        }

        return [];
    }

    onCache(response) {
        this.setState({
            hasOldData: !!response?.response?.length,
            isLoading: true,
            cacheTimestamp: response?.timestamp,
        }, () => {
            this.fetchCars(response?.response)
                .then(() => {
                    this.sortCars();
                });
        });
    }

    fetchCars(response) {
        return new Promise((resolve) => {
            if (response?.length) {
                this.cars = response;

                const queue = this.cars?.map((car) => {
                    return this.request.exec.bind(this.request, REQUESTS.GET_CAR_TAG_HISTORY, {
                        queryParams: {
                            car_id: car.id,
                        },
                    });
                });

                const qs = new QueryScheduler({
                    queue,
                    limit: COUNT_FIRST,
                    onProgress: this.onProgress.bind(this),
                    onSuccess: this.onSuccess.bind(this, { resolvePromise: resolve, isRetry: false, cars: this.cars }),
                });

                qs.run();
            } else {
                this.onSuccess(
                    { resolvePromise: resolve, isRetry: false, cars: this.cars },
                    {
                        queue: [],
                        success: [],
                        failed: [],
                    },
                );
            }
        });
    }

    onProgress() {}

    onSuccess(props: { resolvePromise; isRetry; cars }, data) {
        const { success, failed } = data;
        const failedIndexes = failed.map((error: any) => error.data);

        const successCars: IRichCarEvac[] = [];
        const retryCars: ICar[] = [];

        props.cars.forEach((car, index) => {
            if (!failedIndexes.includes(index)) {
                successCars.push(car);
            } else {
                retryCars.push(car);
            }
        });

        if (successCars.length) {
            let msk: ICar[] = [];
            let spb: ICar[] = [];
            let kzn: ICar[] = [];
            let sochi: ICar[] = [];

            const kznIndex = 2;
            const sochiIndex = 3;

            successCars.forEach((car, index) => {
                if (car.location && car.location.tags && car.location.tags.includes(EVACUATION_LOC_TAG)) {
                    let carData = success[index];

                    if (!carData.objects[car.id]) {
                        carData = success.find((data) => data.objects && data.objects[car.id]);
                    }

                    if (carData) {
                        car = this.enrichCar(car, carData);
                    }

                    if (car.location.tags?.includes(CITIES_KEYS[0])) {
                        msk.push(car);
                    }

                    if (car.location.tags?.includes(CITIES_KEYS[1])) {
                        spb.push(car);
                    }

                    if (car.location.tags?.includes(CITIES_KEYS[kznIndex])) {
                        kzn.push(car);
                    }

                    if (car.location.tags?.includes(CITIES_KEYS[sochiIndex])) {
                        sochi.push(car);
                    }
                }
            });

            if (props.isRetry) {
                msk = msk.concat(this.state.msk);
                spb = spb.concat(this.state.spb);
                kzn = kzn.concat(this.state.kzn);
                sochi = sochi.concat(this.state.sochi);
            }

            //console.log('isRetry: ', props.isRetry, 'msk', this.state.msk, 'new', msk);

            const tabs: TabItem[] = [{
                name: `Москва и МО (${msk.length})`,
                link: CITIES[CITIES_KEYS[0]].short_name_en,
            }, {
                name: `Санкт-Петербург и ЛО (${spb.length})`,
                link: CITIES[CITIES_KEYS[1]].short_name_en,
            }, {
                name: `Казань (${kzn.length})`,
                link: CITIES[CITIES_KEYS[kznIndex]].short_name_en,
            }, {
                name: `Сочи (${sochi.length})`,
                link: CITIES[CITIES_KEYS[sochiIndex]].short_name_en,
            }];

            this.setState({
                tabs,
                msk,
                spb,
                kzn,
                sochi,
                currentTab: this.setTab(tabs),
            }, () => {
                if (!retryCars.length) {
                    props.resolvePromise();
                }
            });
        }

        if (retryCars.length) {
            const LIMIT_RETRIES = 3;
            if (this.count < LIMIT_RETRIES) {
                this.count += 1;
                const queue = retryCars.map((car) => {
                    return this.request.exec.bind(this.request, REQUESTS.GET_CAR_TAG_HISTORY, {
                        queryParams: {
                            car_id: car.id,
                        },
                    });
                });

                const qs = new QueryScheduler({
                    queue,
                    limit: COUNT_FIRST,
                    onProgress: this.onProgress.bind(this),
                    onSuccess: this.onSuccess.bind(this, {
                        resolvePromise: props.resolvePromise,
                        isRetry: true,
                        cars: retryCars,
                    }),
                });

                qs.run();
            } else {
                this.setState({
                    cacheTimestamp: undefined,
                    isLoading: false,
                    error: new Error('Ошибка загрузки дополнительной информации'),
                }, () => {
                    props.resolvePromise();
                });
            }
        }
    }

    getTagData(tag, tags, tagDate) {
        const tagAuthor = TagRecordHandler.getTagAuthor.call(tag);
        const authorLink = TagRecordHandler.getTagAuthorLink.call(tag);
        const tagName = TagRecordHandler.getTagDisplayName.call(tag);
        const priority = TagRecordHandler.getTagPriority.call(tag);
        const timestamp = TagRecordHandler.getTagTimestamp.call(tag);

        tags.push({ tagName, priority, tagAuthor, authorLink });
        tagDate.push(timestamp || '');
    }

    enrichCar(car: ICar, data) {
        let richCar: IRichCarEvac = car;

        if (data.records) {
            if (car.status === CarStatus.SERVICE) {
                const current_tags: Dict<string>[] = [];
                const tagDate: number[] = [];
                let taskLink = '';

                const carTags: string[] = car.tags && car.tags.reduce((acc: string[], tag) => {
                    if (tag.tag === POSSIBLE_EVACUATION_TAG || tag.tag === EVACUATION_TAG) {
                        acc.push(tag.tag_id);
                    }

                    return acc;
                }, []) || [];

                data.records.forEach((tag: TagRecord) => {
                    const tagId: string = TagRecordHandler.getTagId.call(tag);

                    if (carTags.includes(tagId)) {
                        const tagAction = TagRecordHandler.getAction.call(tag);
                        const tagName = TagRecordHandler.getTagName.call(tag);

                        if (tagAction === TAG_ACTION.ADD && tagName == POSSIBLE_EVACUATION_TAG) {
                            this.getTagData(tag, current_tags, tagDate);
                        }

                        if ((tagAction === TAG_ACTION.ADD || tagAction === TAG_ACTION.EVOLVE)
                            && tagName === EVACUATION_TAG) {
                            const previousTag = current_tags
                                && current_tags[current_tags.length - 1]
                                && current_tags[current_tags.length - 1].tagName;

                            if (tagAction === TAG_ACTION.ADD || (tagAction === TAG_ACTION.EVOLVE
                                && (previousTag && previousTag !== EVACUATION_TAG || !previousTag))) {
                                this.getTagData(tag, current_tags, tagDate);
                            }

                            if (tagAction === TAG_ACTION.EVOLVE) {
                                const linkFromComment = TagRecordHandler.getTagComment.call(tag).match(stLinkPattern);
                                taskLink = linkFromComment && linkFromComment[0] || '';
                            }
                        }
                    }
                });

                richCar = {
                    ...richCar,
                    current_tags,
                    tagDate,
                    taskLink,
                };
            } else {
                let incorrectMovingTime: number;

                const incorrectMovingTag = data.records.find((tag) => {
                    return TagRecordHandler.getAction.call(tag) === TAG_ACTION.ADD
                        && TagRecordHandler.getTagName.call(tag) === INCORRECT_MOVING_TAG;
                });

                if (incorrectMovingTag && incorrectMovingTag.hasOwnProperty('timestamp')) {
                    incorrectMovingTime = incorrectMovingTag['timestamp'] * ONE_SECOND;
                    richCar = {
                        ...richCar,
                        incorrectMovingTime,
                    };
                }
            }
        }

        return richCar;
    }

    sortCars() {
        const { currentTab } = this.state;
        const cars = this.state[currentTab];

        let currentCarsService: Dict<IRichCarEvac[]> = {};
        const currentCarsOther: IRichCarEvac[] = [];

        cars?.forEach((car) => {
            if (car.status === CarStatus.SERVICE) {
                const impoundmentLotObj = car.location.areas && car.location.areas.filter((area) => {
                    return area.tags && area.tags[0] === EVACUATION_LOC_TAG;
                });

                const impoundmentLot = impoundmentLotObj[0].title !== '' && impoundmentLotObj[0].title
                    || impoundmentLotObj[0].id;

                if (impoundmentLot && !currentCarsService[impoundmentLot]) {
                    currentCarsService = {
                        ...currentCarsService,
                        [impoundmentLot]: [],
                    };
                }

                if (impoundmentLot) {
                    currentCarsService[impoundmentLot].push(car);
                } else {
                    currentCarsService.unsorted.push(car);
                }
            } else {
                currentCarsOther.push(car);
            }
        });

        this.setState({
            isLoading: false,
            currentCarsService,
            currentCarsOther,
        });
    }

    componentDidMount() {
        this.cache.exec({
            request: this.request.exec.bind(this.request, REQUESTS.GET_CARS, {
                queryParams: {
                    traits: TRAITS,
                    antitraits: ANTITRAITS,
                    filters: '-model_solaris_dedicated_fleet*-block_maintenance',
                },
            }),
            requestKey: REQUEST_KEY,
        });

        this.ls.set(LSSettingItems.evacuationCarsList, undefined);
    }

    componentWillUnmount() {
        this.request.abort();
    }

    componentDidUpdate(prevProps, prevState: IState) {
        if (prevProps.location.search !== this.props.location.search) {
            const queries = new URLSearchParams(this.props.location.search);

            if (!queries.has('city')) {
                this.selectTab(DEFAULT_TAB);
            } else {
                this.state.tabs.forEach((tab) => {
                    if (queries.get('city') === tab.link) {
                        this.selectTab(tab.link);
                    }
                });
            }
        }

    }

    setTab(tabs) {
        let currentTab = DEFAULT_TAB;
        const queries = new URLSearchParams(this.props.location.search);

        tabs.forEach((tab) => {
            if (queries.get('city') === tab.link) {
                currentTab = tab.link;
            }
        });

        return currentTab;
    }

    selectTab(tab) {
        this.setState({
            currentTab: tab,
            isLoading: true,
        }, () => {
            this.sortCars();
        });
    }

    changeHref(tab) {
        location.hash = `${urlPattern}${tab}`;
    }

    render() {
        const {
            tabs,
            currentCarsOther,
            currentCarsService,
            isLoading,
            hasOldData,
            error,
            currentTab,
            newCacheAvailable,
        } = this.state;

        const cacheDate = (
            <FormatDate value={this.state.cacheTimestamp} withSecond/>
        );

        return (
            <>
                {error
                    ? <SimpleError data={{ label: 'Ошибка! Перезагрузите страницу и проверьте подключение' }}
                                   error={error}/>
                    : null
                }
                {isLoading
                    ? <Spin/>
                    : <>
                        <div className={styles.header}>
                            <Tabs tabs={tabs}
                                  currentTab={currentTab}
                                  selectTab={this.changeHref.bind(this)}/>
                            {!error && newCacheAvailable
                                ? <div>
                                    <span className={styles.refresh}>Данные от {cacheDate}</span>
                                    <Button colorType={ButtonTypes.warning}
                                            onClick={() => {
                                                this.setState({
                                                    newCacheAvailable: false,
                                                }, () => {
                                                    this.sortCars();
                                                });
                                            }}>Обновить данные</Button>
                                </div>
                                : (!hasOldData && !newCacheAvailable) || error
                                    ? <span>Данные от {cacheDate}</span>
                                    : <span className={styles.refresh}>
                                        Данные от {cacheDate}. Идет обновление <Spin size={'s'}/>
                                    </span>
                            }
                        </div>
                        <TableCars currentCarsOther={currentCarsOther}
                                   currentCarsService={currentCarsService}/>
                    </>
                }
            </>
        );
    }
}
