import { UA, Utils, WebSocketInterface } from 'jssip';
import React from 'react';
import { connect } from 'react-redux';

import { Button, ButtonTypes } from '../../../ui/Button';
import { Input } from '../../../ui/Input';
import { request_id } from '../../../utils/request';
import { IStore } from '../../App/store';
import { SimpleError } from '../../SimpleError';
import * as style from './index.css';

const SERVER = 'drive-phone.tel.yandex-team.ru';
const PORT = 7443;

enum UA_STATE {
    registered = 'registered',
    connected = 'connected',
    connecting = 'connecting',
    addstream = 'addstream',
    progress = 'progress',
    failed = 'failed',
    ended = 'ended',
    accepted = 'accepted',
}

interface IWebPhoneState {
    ua_state: UA_STATE | null;
    registrationFailed: any;
    ua_state_description: string;
    client: string;
    refer: string;

    incomingLog: any;
    incomingStatus: any;
    direction: string;
}

type IWebPhoneProps = IMapStateToPropsWebPhone

class SupportWebphone extends React.Component<IWebPhoneProps, IWebPhoneState> {
    socket;
    configuration;
    _ua;
    session;

    _soundsControl;
    _localAudio;
    _remoteAudio;

    _localClonedStream;
    state: IWebPhoneState = {
        ua_state: null,
        registrationFailed: null,
        ua_state_description: '',
        client: '',
        refer: '',
        incomingLog: {},
        incomingStatus: '',
        direction: '',
    };

    constructor(props) {
        super(props);
        this._soundsControl = React.createRef();
        this._localAudio = React.createRef();
        this._remoteAudio = React.createRef();
    }

    componentDidMount(): void {
        this.props.yandex_login && this.start();
    }

    componentDidUpdate(prevProps: Readonly<IWebPhoneProps>, prevState: Readonly<IWebPhoneState>, snapshot?: any): void {
        if (this.props.yandex_login !== prevProps.yandex_login) {
            this.props.yandex_login && this.start();
        }
    }

    peerConnectionHandler({ peerconnection }: { peerconnection: RTCPeerConnection }) {
        peerconnection.addEventListener('track', (event: RTCTrackEvent) => {
            const audio = new Audio(); // создадим новое аудио
            const [stream] = event.streams;

            audio.srcObject = stream; // прокидываем в него стрим
            audio.play();
        });
    }

    start() {
        this.socket = new WebSocketInterface(`wss://${SERVER}:${PORT}`);
        // debug.enable('JsSIP:*');
        // debug.enable(false)
        this.socket._onError = (err) => {
            console.info('socket error', err);
        };

        this.configuration = {
            sockets: this.socket,
            session_timers: false,
            authorization_user: this.props.yandex_login,
            uri: `sip:${this.props.yandex_login}@yandex-team.ru`,
            password: `NA`,
        };
        this._ua = new UA(this.configuration);
        this._ua.start();

        this._ua.on('newRTCSession', (data) => {
            const session = data.session;
            this.session = data.session;
            if (session.direction === 'incoming') {
                this.setState({
                    incomingLog: data,
                    direction: session.direction,
                    incomingStatus: 'newRTCSession',
                });
            }

            session.on('icecandidate', (event) => {
                const { candidate } = event;
                const icecandidate = `icecandidate ${JSON.stringify(candidate)}`;
                console.info(icecandidate);
                this.setState({
                    incomingLog: icecandidate,
                });
            });

            session.on('peerconnection', this.peerConnectionHandler);

            session.on('ended', () => {
                this.setState({
                    incomingStatus: 'ended',
                    direction: session.direction,
                });
            });

            session.on('accepted', () => {
                this.setState({
                    incomingStatus: 'accepted',
                    direction: session.direction,
                });
            });

            session.on('hold', () => {
                this.setState({
                    incomingStatus: 'hold',
                    direction: session.direction,
                });
            });

            session.on('unhold', () => {
                this.setState({
                    incomingStatus: 'unhold',
                    direction: session.direction,
                });
            });

            session.on('failed', () => {
                this.setState({
                    incomingStatus: 'failed',
                });
            });
        });

        this._ua.on('connecting', () => {
            this.setState({
                ua_state: UA_STATE.connecting,
                ua_state_description: 'соединяемся',
            });
        });

        this._ua.on('connected', () => {
            this.setState({
                ua_state: UA_STATE.connected,
                ua_state_description: 'соединились',
            });
        });

        this._ua.on('registered', () => {
            this.setState({
                ua_state: UA_STATE.registered,
                ua_state_description: 'можно звонить и принимать звонки',
            });
        });

        this._ua.on('registrationFailed', (data) => {
            console.error('UA registrationFailed', data.cause);
            this.setState({
                registrationFailed: data,
            });
        });
    }

    answer() {
        this.session?.answer(this.options());
    }

    options() {
        return {
            mediaConstraints: {
                audio: true,
                video: false,
            },
            eventHandlers: {
                peerconnection: this.peerConnectionHandler,
            },
            rtcConstraints: {
                optional : [ { googIPv6 : false }],
            },
        };
    }

    cancel() {
        this.session?.terminate();
        Utils.closeMediaStream(this._localClonedStream);
    }

    call() {
        this.session = this._ua.call(this.state.client, {
            'extraHeaders': [`X-recordId: ${request_id()}`],
            pcConfig:
                {
                    hackStripTcp: true, // Какая-то проблема с хром
                    rtcpMuxPolicy: 'negotiate', // multiplexing
                    iceServers: [{ 'urls': ['stun:stun.tel.yandex.net:3478'] }],
                },
            rtcOfferConstraints:
                {
                    offerToReceiveAudio: 1, // Принимаем только аудио
                    offerToReceiveVideo: 0,
                }, ...this.options(),
        });

        this.session.on('connecting', () => {
            this.setState({
                ua_state: UA_STATE.connecting,
                ua_state_description: '...',
            });
            this.playSound('ringback.ogg', true);

            // Тут мы подключаемся к микрофону и цепляем к нему поток
            /* let peerconnection = this.session.connection;
             let localStream = peerconnection.getLocalStreams()[0];

             // Handle local stream
             if (localStream) {
                 // Clone local stream
                 this._localClonedStream = localStream.clone();

                 //@ts-ignore
                 this?._localAudio?.current?.srcObject = this._localClonedStream;
             }

             peerconnection.addEventListener('addstream', (event) => {
                 this.setState({
                     ua_state: UA_STATE.addstream,
                     ua_state_description: '...'
                 });
                 //@ts-ignore
                 this._remoteAudio?.current?.srcObject = event.stream;
             });
            */
        });

        this.session.on('progress', () => {
            console.info('UA session progress');
            this.setState({
                ua_state: UA_STATE.progress,
                ua_state_description: 'В процессе дозвона',
            });
            this.playSound('ringback.ogg', true);
        });

        this.session.on('failed', (data) => {
            this.setState({
                ua_state: UA_STATE.failed,
                ua_state_description: ' Дозвон завершился неудачно, например, абонент сбросил звонок',
            });
            Utils?.closeMediaStream?.(this._localClonedStream);
            this.stopSound();
            this.playSound('rejected.mp3', false);

        });

        this.session.on('ended', () => {
            this.playSound('rejected.mp3', false);
            this.setState({
                ua_state: UA_STATE.ended,
                ua_state_description: 'Сбросили',
            });
            Utils?.closeMediaStream?.(this._localClonedStream);
        });

        this.session.on('accepted', () => {
            console.info('UA session accepted');
            this.setState({
                ua_state: UA_STATE.accepted,
                ua_state_description: 'Звонок принят, можно начинать говорить',
            });
            this.stopSound();
            this.playSound('answered.mp3', false);
        });
    }

    playSound(soundName, loop) {
        /*this._soundsControl.current.pause();
        this._soundsControl.current.currentTime = 0.0;
        this._soundsControl.current.src = 'sounds/' + soundName;
        this._soundsControl.current.loop = loop;
        this._soundsControl.current.play();*/
    }

    stopSound() {
        /*this._soundsControl.current.pause();
        this._soundsControl.current.currentTime = 0.0;*/
    }

    componentWillUnmount(): void {
        Utils?.closeMediaStream?.(this?._localClonedStream);
        this._ua.stop?.();
        this._ua.unregister?.();
    }

    onChangeClient(client) {
        this.setState({
            client,
        });
    }

    onChangeRefer(refer) {
        this.setState({
            refer,
        });
    }

    hold() {
        this.session.hold();
    }

    unhold() {
        this.session.unhold();
    }

    refer() {
        this.session.refer(this.state.refer);
    }

    render() {
        return <div>
            <h1>Демо</h1>
            <div>состояние: <strong>{this.state.ua_state}</strong></div>
            <div>описание: <strong>{this.state.ua_state_description}</strong></div>
            {
                this.state.registrationFailed &&
                <SimpleError error={this.state.registrationFailed}/>
            }
            <audio id="sounds" autoPlay ref={this._soundsControl}></audio>
            <audio id="localAudio" autoPlay ref={this._localAudio}></audio>
            <audio id="remoteAudio" autoPlay ref={this._remoteAudio}></audio>

            <Input value={this.state.client}
                   onChange={this.onChangeClient.bind(this)}
                   placeholder={'Клиент [sip:username | phone number (+3...)]'}/>

            <Input value={this.state.refer}
                   onChange={this.onChangeRefer.bind(this)}
                   placeholder={'Перевести на [sip:username | phone number (+3...)]'}/>

            <Button onClick={this.call.bind(this)}
                    disabled={!this.state.client ||
                    (![UA_STATE.ended, UA_STATE.failed, UA_STATE.registered].includes(this.state.ua_state as UA_STATE))}
                    className={style.btn}>Позвонить</Button>

            <Button onClick={this.cancel.bind(this)}
                    colorType={ButtonTypes.negative}
                    className={style.btn}>Завершить</Button>

            <Button onClick={this.hold.bind(this)}
                    colorType={ButtonTypes.warning}
                    className={style.btn}>Поставить на Холд</Button>

            <Button onClick={this.unhold.bind(this)} className={style.btn}>Снять с Холда</Button>

            <Button onClick={this.refer.bind(this)}
                    disabled={!this.state.refer}
                    colorType={ButtonTypes.positive}
                    className={style.btn}>Перевести</Button>
            <div>
                <div>Информация о звонке:</div>
                <div>direction: {this.state.direction}</div>
                <div>state: {this.state.incomingStatus}</div>
                <div>display name: {this.state.incomingLog?.request?.from?._display_name}</div>
                <div>user: {this.state.incomingLog?.request?.from?._uri?._user}</div>
            </div>

            <div>
                <Button onClick={this.answer.bind(this)}>Ответить на звонок</Button>
            </div>

        </div>;
    }
}

interface IMapStateToPropsWebPhone {
    yandex_login: string | undefined;
}

export const mapStateToProps = (store: IStore): IMapStateToPropsWebPhone => {
    return {
        yandex_login: store.AdminUser?.yandex_login,
    };
};

export default connect<IMapStateToPropsWebPhone, {}, {}>(mapStateToProps)(SupportWebphone);
