import React, { ReactElement } from 'react';

import { Dict } from '../../../types';
import { SessionHistoryInfoHandler } from '../../models/session';
import { Button, ButtonTypes, CancelButton } from '../../ui/Button';
import Checkbox from '../../ui/Checkbox';
import { FormatDateInString } from '../../ui/FormatDate';
import { Confirm, Window } from '../../ui/FullModal';
import * as coreStyle from '../../ui/index.css';
import Select from '../../ui/Select';
import { Request2 } from '../../utils/request';
import { deepCopy } from '../../utils/utils';
import { IStore } from '../App/store';
import { FormConstructor } from '../FormConstructor';
import { removeNullValues } from '../FormConstructor/utils/utils';
import { ProgressBar } from '../ProgressBar';
import { IProgressBarProps } from '../ProgressBar/types';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import { TagPhotoUploadModal } from '../TagPhotoUploader/component';
import * as style from './index.css';
import { REQUESTS, TAG_REQUESTS as requestConfigs } from './request';

export interface ITag {
    tag_id: string;
    author: string;
    comment: string;
    name: string;
    tag_name: string[];
    tag: string;
    ts: number;
    type: string;
    display_name: string;
    default_priority?: string;
    meta: string;
    entities?: string[];
    priority?: number;
    userInfo?: Dict<any>;
    performer?: Dict<any>;
    propositions?: Record<string, any>;
    proposition_id?: string;
}

export interface IConstantsFaceTag {
    constantsIFaceTag: (ids: string) => Promise<{ iface_tag_implementations: Dict<any> }>;
}

export enum OBJECT_TYPES {
    CAR = 'car',
    USER = 'user',
    AREA = 'area',
    ACCOUNT = 'account',
    TRACE = 'trace'
}

export enum Operations {
    CREATE = 'create',
    EVOLUTION = 'evolution',
    EDIT = 'edit',
}

const WINDOW_TITLES = {
    [Operations.CREATE]: 'Создание',
    [Operations.EDIT]: 'Редактирование',
    [Operations.EVOLUTION]: 'Эволюция',
};

const TAG_NAME = 'tagName';
const PRIORITY_KEY = 'priority';
const TAG_KEY = 'tag';
const EMAIL_FINE_SMOKE_TAG = 'email_fine_smoke';
const TECH_DISPATCH_TAGS = ['order_tech_dispatch_tag', 'assignment_tech_dispatch_tag'];

interface ITagModalProps extends IConstantsFaceTag, IStore {
    objectId?: { type: OBJECT_TYPES; id?: string | number };
    onClose: (tag?: any) => void;
    queueLength?: number;
    tagType?: string[];
    forceTags?: { text: string; value: string; disabled?: boolean }[];
    disabledChoice?: boolean;
    initialData?: Dict<any> | null;
    isEdit?: boolean;
    disableCloseModal?: boolean;
    operation?: Operations;
    evolutionTag?: ITag;
    withoutPropositions?: boolean;
    title?: string;
    callback?: Function;
    filterTags?: Function;
    attachmentHandler?: () => void;
    progressBarObj?: IProgressBarProps;
    session?: Record<string, any>;
}

interface ITagModalState {
    [key: string]: any;

    tagsIsLoading: boolean;

    iface_tag_implementations: Dict<any>;
    ifaceIsLoadingError: Error | null;
    ifaceIsLoading: boolean;

    allTags: ITag[];
    allTagsHash: Dict<ITag>;
    tags: { disabled?: boolean; text: string; value: string }[];
    tagName: string[];
    tagDescription: string;
    type: string;
    priority: string;
    tagIsAttaching: boolean;
    tagIsEditing: boolean;
    meta: Dict<any>;
    initialData: Record<string, any> | null;
    loadingError: Error | null;
    attachingError: Error | null;
    confirmIsOpen: boolean;
    afterConfirmAction: any;
    existsTag: Dict<any>;
    isAddingPhoto: boolean;
    ignore_telematics: boolean;
    addingPhotoTagId: string | null;
}

export default class TagModal extends React.Component<ITagModalProps, ITagModalState> {
    state: ITagModalState = {
        tagsIsLoading: false,
        iface_tag_implementations: {},
        ifaceIsLoadingError: null,
        ifaceIsLoading: false,
        allTags: [],
        allTagsHash: {},
        tags: [],
        tagName: [],
        tagDescription: '',
        type: '',
        priority: '',
        meta: {},
        initialData: null,
        tagIsAttaching: false,
        tagIsEditing: false,
        loadingError: null,
        attachingError: null,
        afterConfirmAction: null,
        confirmIsOpen: false,
        existsTag: {},
        isAddingPhoto: false,
        ignore_telematics: false,
        addingPhotoTagId: null,
    };
    request = new Request2({ requestConfigs });

    componentDidMount() {
        this.getTags();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getTags() {
        this.setState({ tagsIsLoading: true }, () => {

            this.request.exec(REQUESTS.GET_TAGS)
                .then(response => {
                    let allTags: ITag[] = response?.records ?? [];

                    if (this.props.filterTags) {
                        allTags = this.props?.filterTags(allTags);
                    }

                    if (this.props.objectId?.type) {
                        const type: string = this.props.objectId.type;
                        allTags = allTags
                            .filter(tag => tag.entities?.includes(type));
                    } else if (this.props.tagType?.length) {
                        const { tagType } = this.props;
                        allTags = allTags.filter(tag => tagType.includes(tag.type));
                    }

                    const allTagsHash: Dict<ITag> = allTags?.reduce((result: Dict<ITag>, cur: ITag) => {
                        return Object.assign({}, result, { [cur.name]: cur });
                    }, {});

                    const types = Array.from(new Set(allTags.map(tag => tag.type))).sort();

                    let tagList = types.reduce((result: { disabled: boolean; text: string; value: string }[], type) => {
                        return [
                            ...result,
                            ...allTags
                                .filter(tag => {
                                    return tag.type === type;
                                })
                                .sort(this.tagComparator()).map(this.tagMapper()),
                        ];
                    }, []);

                    const tagName = this.props.initialData && this.props.initialData.tag
                        ? [this.props.initialData.tag]
                        : [];

                    const filteredData = tagName?.[0]
                        && allTags?.filter((tag: ITag) => tag.name === tagName[0]);

                    const type = filteredData?.[0]?.type;

                    const priority = this.props.initialData?.priority ?? filteredData?.[0]?.default_priority ?? 0;

                    if (this.props.operation === Operations.EVOLUTION) {
                        if (this.props.evolutionTag) {
                            const evolutionTag: Dict<any> = this.props.evolutionTag || {};

                            const adminUser: Dict<any> = this.props.AdminUser || {};
                            const evolutions = adminUser?.permissions?.evolutions
                                ? adminUser.permissions.evolutions[evolutionTag.tag]
                                : [];

                            const evolutionsResults = evolutions.map(evolution => evolution.to);

                            tagList = tagList.filter(tag => evolutionsResults.includes(tag.value));
                        } else {
                            console.warn('No evolution tag for evolution operation!');
                        }
                    }

                    this.setState({
                        allTags,
                        allTagsHash,
                        tags: this.props.forceTags ? this.props.forceTags : tagList,
                        tagsIsLoading: false,
                        tagName,
                        tagDescription: this.props.initialData?.comment ?? '',
                        type,
                        priority,
                    }, () => {
                        if (Array.isArray(tagName) && tagName.length) {
                            this.getTagImplementations(); //если тег есть на старте сразу подтягиваем для него схему
                        }
                    });

                })
                .catch(loadingError => {
                    this.setState({ loadingError, tagsIsLoading: false });
                });
        });

    }

    getTagImplementations(callback?: () => void) {
        this.setState({
            ifaceIsLoading: true,
        }, () => {
            const { initialData: initialDataProps, constantsIFaceTag, isEdit } = this.props;
            const selectedTagName = this.state.tagName?.toString?.();
            const tagType = this.state.allTagsHash?.[selectedTagName]?.type;
            const ids = tagType !== selectedTagName ? `${tagType},${selectedTagName}` : selectedTagName;

            constantsIFaceTag(ids)
                .then((response) => {
                    let schema = response.iface_tag_implementations;
                    const key = schema?.hasOwnProperty(selectedTagName) ? selectedTagName : tagType;
                    const initialData = initialDataProps ? { ...initialDataProps } : null;

                    if (schema && TECH_DISPATCH_TAGS.includes(key)) {
                        const operations = { ...schema[key].operations };

                        if (isEdit) {
                            delete operations.default;
                        } else {
                            if (operations.default) {
                                const defaultOperations: any = [];
                                operations.default?.map((el) => {
                                    defaultOperations.push({ type: el });
                                });
                                operations.default = defaultOperations;
                            } else {
                                operations.default = [{ type: null }];
                            }
                        }

                        schema = { [key]: { ...schema[key], operations } };

                        if (initialData && initialData.operations) {
                            const initialOperations: any = [];
                            initialData.operations.map((el) => {
                                initialOperations.push({ type: el });
                            });

                            if (!initialData.operations.length) {
                                initialOperations.push({ type: null });
                            }

                            initialData.operations = initialOperations;
                        }
                    }

                    this.setState({
                        iface_tag_implementations: schema || {},
                        ifaceIsLoading: false,
                        initialData,
                    }, () => {
                        callback?.();
                    });
                })
                .catch((ifaceIsLoadingError) => {
                    this.setState({
                        ifaceIsLoadingError,
                        ifaceIsLoading: false,
                    });
                });
        });
    }

    tagComparator() {
        return (a: ITag, b: ITag) => {
            const displayNameA = a.display_name || '';
            const displayNameB = b.display_name || '';

            return displayNameA.localeCompare(displayNameB);
        };
    }

    tagMapper() {
        return (tag: ITag) => {
            return { text: tag.display_name || tag.name, value: tag.name, description: tag.comment };
        };
    }

    onSelect(type: string, value: string) {
        let meta = {};

        this.setState({ [type]: value ? [value] : [] }, () => {

            this.getTagImplementations(() => { // после выбора тега в выпадашке, подтягиваем его схему и дельше текущая магия
                if (value) {
                    const tag = this.state.allTags?.filter(item => item?.name === value)?.[0];
                    const priority = tag?.default_priority || '0';

                    if (type === TAG_NAME) {
                        meta = Object.assign({}, this.state.meta,
                            {
                                tag: value,
                                priority,
                            });
                        meta = removeNullValues(meta);
                    }

                    this.setState({ priority, meta, type: tag.type });
                } else {
                    this.setState({ priority: '', meta, type: '' });
                }
            });

        });
    }

    onChangeForm(data: Dict<any>) {
        this.setState({ meta: data });
    }

    formatTagData(data: Dict<any>) {
        data[PRIORITY_KEY] = +data[PRIORITY_KEY] || 0;
        data[TAG_KEY] = data?.[TAG_KEY]?.toString();

        return data;
    }

    checkBefore(isProposition: boolean) {
        const { objectIdData, type, data } = this.getInfo();
        if (type === OBJECT_TYPES.CAR || type === OBJECT_TYPES.USER || type === OBJECT_TYPES.TRACE) {
            let currentObjectTags: any;
            this.setState({
                tagIsAttaching: true,
            }, async () => {
                try {
                    currentObjectTags = await this.request.exec(REQUESTS?.[`GET_${type.toUpperCase()}_TAG_LIST`], {
                        queryParams: {
                            object_id: objectIdData.id,
                        },
                    });
                } catch (e) {
                    currentObjectTags = {};
                }

                const existsTag = currentObjectTags?.records?.find(item => item.tag === data.tag);
                if (existsTag) {
                    this.setState({
                        existsTag,
                        afterConfirmAction: this.attachTag.bind(this, isProposition),
                        tagIsAttaching: false,
                        confirmIsOpen: true,
                    });
                } else {
                    this.attachTag(isProposition);
                }
            });

        } else {
            this.attachTag(isProposition);
        }
    }

    getInfo() {
        const objectIdData = this.props.objectId || { type: '', id: '' };
        const { type } = objectIdData;
        const data = Object.assign({}, this.formatTagData(removeNullValues(this.state.meta)), {
            object_id: objectIdData.id,
        });

        return {
            objectIdData,
            type,
            data,
        };
    }

    attachTag(isProposition: boolean) {
        this.setState({ tagIsAttaching: true }, () => {
            const { objectIdData, type, data } = this.getInfo();

            let handler: null | REQUESTS = null;

            switch (type) {
            case OBJECT_TYPES.CAR:
                handler = isProposition ? REQUESTS.CAR_TAG_PROPOSE : REQUESTS.ATTACH_CAR_TAG;
                break;
            case OBJECT_TYPES.USER:
                handler = REQUESTS.ATTACH_USER_TAG;
                break;
            case OBJECT_TYPES.AREA:
                handler = REQUESTS.ATTACH_AREA_TAG;
                break;
            case OBJECT_TYPES.ACCOUNT:
                handler = REQUESTS.ATTACH_ACCOUNT_TAG;
                break;
            case OBJECT_TYPES.TRACE:
                handler = REQUESTS.ATTACH_TRACE_TAG;
                break;
            }

            handler && this.request.exec(handler, { queryParams: { object_id: objectIdData.id }, body: data })
                .then(response => {
                    this.props?.callback?.(data, response);

                    if (this.state.isAddingPhoto && type === OBJECT_TYPES.CAR) {
                        const addingPhotoTagId = response?.tagged_objects?.[0]?.tag_id?.[0]
                            || this.state.existsTag?.tag_id;
                        this.setState({ addingPhotoTagId });
                    } else {
                        this.props.onClose();
                    }

                })
                .catch(attachingError => {
                    this.setState({ attachingError, tagIsAttaching: false, disableCloseModal: false });
                });
        });
    }

    onCloseTagPhotoModal() {
        this.setState({ addingPhotoTagId: null }, () => {
            this.props.onClose();
        });
    }

    evolveTag() {
        this.setState({
            tagIsAttaching: true,
        }, () => {

            let tag: Dict<any> = Object.assign({}, this.state.meta, { priority: +this.state.meta.priority });

            tag = removeNullValues(tag);

            const tag_id = this.props.evolutionTag?.tag_id;
            const performer = this.props.evolutionTag?.performer || '';

            this.request.exec(REQUESTS.EVOLVE_TAG, {
                queryParams: { tag_id, evolution_mode: this.state.ignore_telematics ? 'ignore_telematic' : null },
                body: tag,
                headers: { UserIdDelegation: performer },
            })
                .then(() => {
                    this.setState({
                        tagIsAttaching: false,
                    });
                    this.props?.callback?.(tag);
                    this.props.onClose();
                })
                .catch(attachingError => {
                    this.setState({ attachingError, tagIsAttaching: false, disableCloseModal: false });
                });
        });
    }

    editTag() {
        const objectIdData = this.props.objectId || { type: '', id: '' };
        const { type } = objectIdData;

        const request = REQUESTS[`UPDATE_${type.toUpperCase()}_TAG`];

        request && this.setState({
            tagIsEditing: true,
        }, () => {

            const initData = Object.assign({}, deepCopy(this.state.initialData || {}), deepCopy(this.state.existsTag));

            let tag: Dict<any> = Object.assign({}, initData, this.state.meta,
                { priority: +this.state.meta.priority ?? 0 });

            tag = removeNullValues(tag);

            this.request.exec(request, { body: tag })
                .then(response => {
                    this.setState({
                        tagIsEditing: false,
                    });

                    this.props?.callback?.(tag);

                    if (this.state.isAddingPhoto && type === OBJECT_TYPES.CAR) {
                        const addingPhotoTagId = response?.tagged_objects?.[0]?.tag_id?.[0]
                            || this.state.existsTag?.tag_id
                            || tag.tag_id;
                        this.setState({ addingPhotoTagId });
                    } else {
                        this.props.onClose();
                    }

                })
                .catch(attachingError => {
                    this.setState({ attachingError, tagIsEditing: false, disableCloseModal: false });
                });
        });
    }

    onAddingPhotoChange(isAddingPhoto: boolean) {
        this.setState({ isAddingPhoto });
    }

    onIngoreTelematicsChange(ignore_telematics: boolean) {
        this.setState({ ignore_telematics });
    }

    showConfirm(confirmIsOpen) {
        this.setState({
            confirmIsOpen,
        });
    }

    render() {
        let {
            operation,
            onClose,
            queueLength,
            disabledChoice,
            disableCloseModal,
            withoutPropositions,
            session,
        } = this.props;
        const {
            iface_tag_implementations, tagName, type, priority, tagsIsLoading, tags, tagIsAttaching,
            tagIsEditing, initialData: initialDataState,
            attachingError, loadingError,
        } = this.state;

        operation = operation ? operation : Operations.CREATE;

        let schema = iface_tag_implementations?.hasOwnProperty(tagName[0])
            ? iface_tag_implementations[tagName[0]]
            : iface_tag_implementations[type] || {};

        if (schema.tag) {
            const tag = { ...schema.tag, read_only: true };
            schema = { ...schema, tag };
        }

        const checkIsFineTag = tagName?.[0]?.includes('fine_') && schema?.['links']; //#DRIVEFRONT-826
        let template_args = {};

        if (tagName?.[0] === EMAIL_FINE_SMOKE_TAG) {
            const carNumber = SessionHistoryInfoHandler.getCarNumber.call(session);
            const carModel = SessionHistoryInfoHandler.getModelName.call(session);
            const userInfo = SessionHistoryInfoHandler.getUserDetails.call(session);
            const rideStart = SessionHistoryInfoHandler.getStart.call(session);

            template_args = {
                user_name: userInfo?.first_name,
                start_time: FormatDateInString({ value: rideStart }),
                car_model: carModel,
                car_number: carNumber,
            };
        }

        const initialData = Object.assign({},
            { ...checkIsFineTag && { links: [{ uri: '', type: 'st' }] } },
            initialDataState ? deepCopy(initialDataState) : {},
            { template_args },
            {
                tag: tagName[0],
                priority: priority,
            });
        const { type: objectType } = this.getInfo();
        let duplicateQuestion: string | ReactElement;

        switch (objectType) {
        case OBJECT_TYPES.CAR:
            duplicateQuestion = 'Тег уже навешен! Продублировать тег?';
            break;
        case OBJECT_TYPES.USER:
            duplicateQuestion =
                    <div>Такой тег на пользователе уже есть.<br/> Проверь и прими решение. Навесить ещё один тег?"
                    </div>;
            break;
        case OBJECT_TYPES.TRACE:
            duplicateQuestion =
                    <div>Такой тег на сессии уже есть.<br/> Проверь и прими решение. Навесить ещё один?"
                    </div>;
            break;
        default:
            duplicateQuestion = 'Дубль!';
            break;
        }

        if (objectType === OBJECT_TYPES.USER && schema.priority) {
            schema = {
                ...schema,
                priority: {
                    ...schema.priority,
                    read_only: true,
                },
            };
        }

        return <Window onClose={onClose}
                       error={attachingError || loadingError}
                       title={
                           this.props.title ||
                           `${WINDOW_TITLES[operation]} тега ${queueLength ? `(${queueLength}) авто` : ''}`
                       }>
            <div>
                {tagsIsLoading
                    ? <Spin size={'l'}/>
                    : <>
                        <Select options={tags}
                                placeholder={'Тег'}
                                onSelect={this.onSelect.bind(this, TAG_NAME)}
                                disabled={disabledChoice || operation === Operations.EDIT}
                                initialValues={tagName}/>
                        {schema && Object.keys(schema)?.length
                            ? <>
                                <FormConstructor rawResult
                                                 schema={schema}
                                                 onChange={this.onChangeForm.bind(this)}
                                                 initialData={initialData}/>
                                {/*Only for cars without group attach. attachmentHandler is only for group attach*/}
                                {
                                    this.props.objectId?.type === OBJECT_TYPES.CAR && !this.props?.attachmentHandler
                                        ? <div className={style.add_photo_control_container}>
                                            <span>Добавить фото:</span>
                                            <Checkbox checked={this.state.isAddingPhoto}
                                                      onChange={this.onAddingPhotoChange.bind(this)}/>
                                        </div>
                                        : null
                                }
                            </>
                            : null}
                        {
                            this.state.ifaceIsLoading && <Spin size={'s'}/>
                        }
                        {
                            this.state.ifaceIsLoadingError && <SimpleError error={this.state.ifaceIsLoadingError}/>
                        }
                        {
                            operation === Operations.EVOLUTION && tagName?.[0]
                                ? <div className={style.add_photo_control_container}>
                                    <span>ingore_telematics:</span>
                                    <Checkbox checked={this.state.ingore_telematics}
                                              onChange={this.onIngoreTelematicsChange.bind(this)}/>
                                </div>
                                : null
                        }
                        {this.props.progressBarObj &&
                        <ProgressBar allLength={this.props.progressBarObj?.allLength}
                                     successLength={this.props.progressBarObj?.successLength}
                                     errors={this.props.progressBarObj?.errors}/>
                        }
                        <div className={coreStyle.button_container}>
                            {!disableCloseModal
                                ? <CancelButton isLoading={tagIsAttaching || tagIsEditing}
                                                onClick={onClose.bind(this)}/>
                                : null
                            }
                            {
                                operation === Operations.CREATE
                                    ? <>
                                        {!withoutPropositions
                                            ? <Button disabled={!tagName[0]}
                                                      isLoading={tagIsAttaching || tagIsEditing}
                                                      basic
                                                      onClick={this.props?.attachmentHandler
                                                          ?.bind(this, removeNullValues(this.state.meta), true)
                                                      ?? this.checkBefore.bind(this, true)}>
                                                Предложить
                                            </Button>
                                            : null}
                                        <Button disabled={!tagName[0]}
                                                isLoading={tagIsAttaching || tagIsEditing}
                                                onClick={this.props?.attachmentHandler
                                                    ?.bind(this, removeNullValues(this.state.meta), false)
                                                ?? this.checkBefore.bind(this, false)}>Создать</Button>
                                    </>
                                    : null
                            }
                            {
                                operation === Operations.EVOLUTION
                                    ? <Button disabled={!tagName[0]}
                                              isLoading={tagIsAttaching || tagIsEditing}
                                              onClick={this.evolveTag.bind(this)}>Эволюция</Button>
                                    : null
                            }
                            {
                                operation === Operations.EDIT
                                    ? <Button disabled={!tagName[0]}
                                              isLoading={tagIsAttaching || tagIsEditing}
                                              onClick={this.editTag.bind(this)}>Обновить</Button>
                                    : null
                            }
                        </div>
                    </>
                }
                {this.state.addingPhotoTagId !== null
                    ? <TagPhotoUploadModal onClose={this.onCloseTagPhotoModal.bind(this)}
                                           tagId={this.state.addingPhotoTagId}
                                           carId={this.props.objectId?.id
                                               ? this.props.objectId?.id?.toString()
                                               : null}/>
                    : null}
                {
                    this.state.confirmIsOpen
                        ? <Confirm accept={this.state.afterConfirmAction}
                                   error={null}
                                   question={duplicateQuestion}
                                   className={style.modal}
                                   isWorking={this.state.tagIsAttaching || tagIsEditing}
                                   additionalControls={
                                       <Button isLoading={this.state.tagIsAttaching || tagIsEditing}
                                               onClick={this.editTag.bind(this)}
                                               colorType={ButtonTypes.warning}>Обновить</Button>
                                   }
                                   onClose={this.showConfirm.bind(this, false)}/> : null
                }
            </div>
        </Window>;
    }
}
