import * as React from 'react';

import { Dict } from '../../../types';
import { Button, ButtonTypes } from '../../ui/Button';
import { FileListView } from '../../ui/FileListView';
import { Window } from '../../ui/FullModal';
import * as coreStyle from '../../ui/index.css';
import { Request2 } from '../../utils/request';
import { deepCopy, generateuuid4 } from '../../utils/utils';
import { FormConstructor } from '../FormConstructor';
import {
    CHAT_LINKS_FORM_SCHEMA,
    MARKER,
    PHOTOS_LINKS_FORM_SCHEMA,
    UPLOAD_METHOD_DISPLAY_NAME,
    UploadMethod,
    WINDOW_DESCRIPTION,
} from './constants';
import { REQUESTS, TAG_PHOTO_REQUESTS } from './request';
import * as style from './style.css';

interface ITagPhotoUploadModalProps {
    onClose: (success: boolean) => void;
    carId?: string | null;
    tagId?: string | null;
}

interface ITagPhotoUploadModalState {
    isLoading: boolean;
    loadingError: Error | null;
    files: {
        [UploadMethod.FROM_DISK]: File[];
        [UploadMethod.FROM_CHAT_LINK]: { link: string; file: File }[];
        [UploadMethod.FROM_PHOTO_LINK]: { link: string; file: File }[];
    };
    photosLinksFormData: Dict<any>;
    chatLinkFormData: Dict<any>;
    linkArrayFormInitialData: string[]; //if get links form chat, update form with this links
    uploadedPhotosLinks: Dict<boolean>;
    chatHistoryLoadingError: Error | null;
}

export class TagPhotoUploadModal extends React.Component<ITagPhotoUploadModalProps, ITagPhotoUploadModalState> {
    state: ITagPhotoUploadModalState = {
        isLoading: false,
        loadingError: null,
        files: { [UploadMethod.FROM_DISK]: [], [UploadMethod.FROM_CHAT_LINK]: [], [UploadMethod.FROM_PHOTO_LINK]: [] },
        photosLinksFormData: {},
        chatLinkFormData: {},
        linkArrayFormInitialData: [],
        uploadedPhotosLinks: {},
        chatHistoryLoadingError: null,
    };
    request = new Request2({ requestConfigs: TAG_PHOTO_REQUESTS });

    registerPhoto() {
        const { tagId, carId } = this.props;
        const { files: filesByUploadMethod } = this.state;

        const files: File[] = [...filesByUploadMethod[UploadMethod.FROM_DISK],
            ...filesByUploadMethod[UploadMethod.FROM_CHAT_LINK].map(chatFiles => chatFiles.file),
            ...filesByUploadMethod[UploadMethod.FROM_PHOTO_LINK].map(linkFiles => linkFiles.file)];

        this.setState({ isLoading: true, loadingError: null }, () => {

            const photoIds: string[] = files.map(() => generateuuid4());
            const photos = files.map((_, index) => {
                return {
                    marker: MARKER,
                    uuid: photoIds[index],
                    md5: '',
                };
            });

            const body = {
                tag_ids: [tagId],
                photos,
            };

            this.request.exec(REQUESTS.REGISTER_TAG_PHOTO, { body })
                .then(() => {
                    const filesUploadRequests = files.map((file, index) => {
                        return new Promise((resolve, reject) => {
                            const type = file.type;
                            const reader = new FileReader();
                            reader.readAsArrayBuffer(file);
                            reader.onload = () => {
                                resolve(this.request.exec(REQUESTS.UPLOAD_TAG_PHOTO, {
                                    queryParams: {
                                        object_id: carId,
                                        photo_id: photoIds[index],
                                    },
                                    file: reader.result,
                                    headers: {
                                        'Content-Type': type,
                                    },
                                }));
                            };

                            reader.onerror = reject;
                        });
                    });

                    Promise.all(filesUploadRequests)
                        .then(() => {
                            this.props.onClose(true);
                            this.setState({ isLoading: false });
                        })
                        .catch(loadingError => {
                            this.setState({
                                loadingError,
                                isLoading: false,
                            });
                        });
                })
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    componentWillUnmount() {
        this.request.abort();
    }

    openFiles() {
        const { files } = this.state;
        const file: HTMLInputElement = document.createElement('input');
        file.type = 'file';
        file.multiple = true;
        file.style.display = 'none';
        document.body.appendChild(file);
        file.click();
        file.onchange = (e: any) => {
            e.stopPropagation();
            e.preventDefault();
            const targetFiles = e.target.files;
            const renamedFiles = [...targetFiles].map(file => {
                const blob = file.slice(0, file.size, file.type);

                return new File([blob], generateuuid4(), { type: file.type });
            });

            files[UploadMethod.FROM_DISK].push(...renamedFiles);

            this.setState({ files });
        };
    }

    removeItem(method: UploadMethod, index: number) {
        const { files, uploadedPhotosLinks } = this.state;
        const filesByMethod = files[method];

        if (method === UploadMethod.FROM_PHOTO_LINK || method === UploadMethod.FROM_CHAT_LINK) {
            const file: any = filesByMethod[index];
            uploadedPhotosLinks[encodeURIComponent(file?.link)] = false;
        }

        filesByMethod.splice(index, 1);
        this.setState({ files });
    }

    onPhotosLinksFormChange(photosLinksFormData: Dict<any>) {
        this.setState({ photosLinksFormData });
    }

    onChatLinkFormChange(chatLinkFormData: Dict<any>) {
        this.setState({ chatLinkFormData });
    }

    getChatPhotosFromLink() {
        const CHAT_ID_PARAM_KEY = 'chat_id';
        const USER_ID_PARAM_KEY = 'user_id';
        const MEDIA_RESOURCE_KEY = 'media_resources';
        const HOST = 'https://carsharing-chat.s3.yandex.net/';

        const { chatLinkFormData } = this.state;

        const chatLink: string = chatLinkFormData?.chat_link ?? '';

        const params: Dict<string> = chatLink.split('?')[1]?.split('&')?.reduce((result, oneParamString: string) => {
            const [key, value] = oneParamString.split('=');
            result[key] = value;

            return result;
        }, {});

        const chatId = params[CHAT_ID_PARAM_KEY];
        const userId = params[USER_ID_PARAM_KEY];

        this.request.exec(REQUESTS.GET_CHAT_HISTORY, { queryParams: { user_id: userId, chat_id: chatId } })
            .then(chatHistory => {
                const { messages = [] } = chatHistory ?? {};
                const mediaResources = messages.filter(message => message.type === MEDIA_RESOURCE_KEY);

                const links = mediaResources.reduce((result: string[], mediaResource) => {
                    const { cached_images = [] } = mediaResource ?? {};
                    const linkArray = cached_images?.map(cachedImage => `${HOST}${cachedImage}`) ?? [];
                    result.push(...linkArray);

                    return result;
                }, []);

                const { photosLinksFormData } = this.state;
                const { link_chat_array } = photosLinksFormData;
                const linkArrayFormInitialData = [
                    ...link_chat_array,
                    ...links.filter(link => !link_chat_array.includes(link)),
                ];

                this.setState({ linkArrayFormInitialData });

            })
            .catch(chatHistoryLoadingError => {
                this.setState({ chatHistoryLoadingError });
            });
    }

    async uploadPhotosByLinks() {
        const fetchPhoto = async (link: string, method: UploadMethod) => {
            const image = await fetch(link);

            return { image, link, method };
        };

        const blobPhoto = async (photo: { image: any; link: string; method: UploadMethod }) => {
            const { link, image, method } = photo;
            const blob = await image.blob();

            return { image: blob, link, method };
        };

        const { photosLinksFormData, uploadedPhotosLinks, files } = this.state;
        const { link_core_array, link_chat_array } = photosLinksFormData;

        const uploadedPhotosLinksCopy: Dict<boolean> = deepCopy(uploadedPhotosLinks);
        const imageRequests = [
            ...link_core_array.map(link => fetchPhoto(link, UploadMethod.FROM_PHOTO_LINK)),
            ...link_chat_array.map(link => fetchPhoto(link, UploadMethod.FROM_CHAT_LINK)),
        ];
        const images: any = await Promise.allSettled(imageRequests);
        const uploadedImages = images.filter(image => image?.value).map(image => image.value);

        const blobRequests = uploadedImages.map(uploadedImage => blobPhoto(uploadedImage));
        const blobs: any = await Promise.allSettled(blobRequests);
        const blobImages = blobs.filter(image => image?.value).map(image => image.value);

        const filesWithLinks = blobImages.map(blobImage => {
            const { image, link, method } = blobImage;
            const file = new File([image], generateuuid4(), { type: image.type });

            return { file, link, method };
        });

        filesWithLinks.forEach(fileWithLink => {
            const { file, link, method } = fileWithLink;

            if (!uploadedPhotosLinksCopy[encodeURIComponent(link)]) {
                files[method].push({ file, link });
                uploadedPhotosLinksCopy[encodeURIComponent(link)] = true;
            }
        });

        this.setState({ files, uploadedPhotosLinks: uploadedPhotosLinksCopy });
    }

    render() {
        const { onClose } = this.props;
        const { loadingError, isLoading, files, linkArrayFormInitialData, photosLinksFormData } = this.state;
        const isAnyFiles = !!files?.[UploadMethod.FROM_DISK]?.length
            || !!files?.[UploadMethod.FROM_CHAT_LINK]?.length
            || !!files?.[UploadMethod.FROM_PHOTO_LINK]?.length;

        return <Window error={loadingError}
                       onClose={onClose.bind(this, false)}
                       title={'Загрузка фото'}
                       description={WINDOW_DESCRIPTION}>
            <h4 className={style.title}>Загрузить фото по ссылке:</h4>
            <FormConstructor schema={CHAT_LINKS_FORM_SCHEMA}
                             hideChanges={true}
                             onChange={this.onChatLinkFormChange.bind(this)}/>
            <div className={coreStyle.button_container}>
                <Button disabled={isLoading}
                        basic
                        onClick={this.getChatPhotosFromLink.bind(this)}>Добавить ссылки на фото из чата</Button>
            </div>
            <FormConstructor schema={PHOTOS_LINKS_FORM_SCHEMA}
                             hideChanges={true}
                             initialData={{ link_core_array: [], link_chat_array: linkArrayFormInitialData }}
                             onChange={this.onPhotosLinksFormChange.bind(this)}/>
            <div className={coreStyle.button_container}>
                <Button disabled={isLoading
                    || (photosLinksFormData.link_chat_array?.length
                        + photosLinksFormData.link_core_array?.length) === 0}
                        basic
                        colorType={ButtonTypes.positive}
                        onClick={this.uploadPhotosByLinks.bind(this)}>Загрузить по ссылке
                    ({photosLinksFormData.link_chat_array?.length + photosLinksFormData.link_core_array?.length})
                </Button>
            </div>
            <h4 className={style.title}>Загрузить фото с диска:</h4>
            <Button basic disabled={isLoading} colorType={ButtonTypes.positive} onClick={this.openFiles.bind(this)}>
                Выбрать файл на диске</Button>
            {files?.[UploadMethod.FROM_CHAT_LINK]?.length
                ? <>
                    <div className={style.uploaded_group}>{UPLOAD_METHOD_DISPLAY_NAME[UploadMethod.FROM_CHAT_LINK]}:
                    </div>
                    <FileListView files={files?.[UploadMethod.FROM_CHAT_LINK]?.map(chatFile => chatFile.file)}
                                  removeItem={this.removeItem.bind(this, UploadMethod.FROM_CHAT_LINK)}/>
                </>
                : null}
            {files?.[UploadMethod.FROM_PHOTO_LINK]?.length
                ? <>
                    <div className={style.uploaded_group}>{UPLOAD_METHOD_DISPLAY_NAME[UploadMethod.FROM_PHOTO_LINK]}:
                    </div>
                    <FileListView files={files?.[UploadMethod.FROM_PHOTO_LINK]?.map(linkFile => linkFile.file)}
                                  removeItem={this.removeItem.bind(this, UploadMethod.FROM_PHOTO_LINK)}/>
                </>
                : null}
            {files?.[UploadMethod.FROM_DISK]?.length
                ? <>
                    <div className={style.uploaded_group}>{UPLOAD_METHOD_DISPLAY_NAME[UploadMethod.FROM_DISK]}:</div>
                    <FileListView files={files?.[UploadMethod.FROM_DISK]}
                                  removeItem={this.removeItem.bind(this, UploadMethod.FROM_DISK)}/>
                </>
                : null}
            {!isAnyFiles
                ? <h4 className={style.no_photo_message}>Фото не выбраны</h4>
                : null}
            <div className={coreStyle.button_container}>
                <Button disabled={isLoading || !isAnyFiles}
                        onClick={this.registerPhoto.bind(this)}>Прикрепить фото к тегу</Button>
            </div>
        </Window>;
    }
}
