import React, { useEffect, useState } from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { Dict } from '../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../constants';
import FormatDate from '../../ui/FormatDate';
import { JsonModal } from '../../ui/FullModal/JsonModal';
import { Input } from '../../ui/Input';
import { Link } from '../../ui/Link';
import { LabelStatus, TLabel } from '../../ui/Table';
import { ITag, IUser } from './component';
import * as _style from './index.css';

interface IHistoryVirtualListProps {
    rows: ITag[];
    users: Dict<IUser>;
    usersRevert: Dict<IUser>;
}

const HEIGHT_PADDING = 230;

export const HistoryVirtualList = React.memo((props: IHistoryVirtualListProps) => {
    const [windowSize, setWindowSize] = useState(() => ({ width: window.innerWidth, height: window.innerHeight }));
    useEffect(() => {
        const onResize = () => setWindowSize({ width: window.innerWidth, height: window.innerHeight });
        window.addEventListener('resize', onResize);

        return () => window.removeEventListener('resize', onResize);
    });

    const [filter, setFilter] = useState('');
    const [filteredData, updateData] = useState([] as ITag[]);
    const [json, showJson] = useState(null);

    useEffect(() => {
        if (!filter) {
            updateData(props.rows);
        } else {
            updateData(props.rows.filter(el => {
                return el.Name.toUpperCase().includes(filter.toUpperCase().trim())
                    || el.Tag.toUpperCase().includes(filter.toUpperCase().trim())
                    || el.Comment.toUpperCase().includes(filter.toUpperCase().trim())
                    || el.Type.toUpperCase().includes(filter.toUpperCase().trim())
                    || el.Performer == props.usersRevert[filter.trim()]?.ID
                    || el.HistoryUserID == props.usersRevert[filter.trim()]?.ID;
            }));
        }
    }, [filter]);

    return <>
        <Input value={filter} onChange={setFilter} placeholder={'фильтр'}/>
        <div className={_style.table_row}>
            <span className={_style.th}>#</span>
            <span className={_style.th}>type</span>
            <span className={_style.th}>Date</span>
            <span className={_style.th}>Tag</span>
            <span className={_style.th}>HistoryUserID</span>
            <span className={_style.th}>Performer</span>
            <span className={_style.th}/>
        </div>
        <VirtualList width={'100%'}
                     height={windowSize.height - HEIGHT_PADDING}
                     itemCount={filteredData.length}
                     itemSize={40}
                     renderItem={({ index, style }) => {
                         const item = filteredData[index];

                         return <Item key={index}
                                      style={style}
                                      item={item}
                                      index={index}
                                      users={props.users}
                                      showInfo={showJson.bind(null, item)}/>;
                     }}/>
        {
            json && <JsonModal obj={json}
                               title={'Просмотр информации о теге'}
                               onClose={showJson.bind(null, null)}/>
        }
    </>;
});

interface IItemProps {
    style: any;
    users: Dict<IUser>;
    item: ITag;
    index: number;
    showInfo: () => void;
}

enum TAG_ACTIONS {
    set_performer = 'set_performer',
    drop_performer = 'drop_performer',
    evolve = 'evolve',
    update_data = 'update_data',
    add = 'add',
    remove = 'remove'
}

export const getStatusType = (text) => {
    switch (text) {
    case TAG_ACTIONS.add:
        return LabelStatus.POSITIVE;
    case TAG_ACTIONS.remove:
        return LabelStatus.NEGATIVE;
    case TAG_ACTIONS.set_performer:
        return LabelStatus.INFO;
    case TAG_ACTIONS.drop_performer:
        return LabelStatus.WARNING;
    default:
        return LabelStatus.DEFAULT;
    }
};

const Item = React.memo((props: IItemProps) => {
    const { item, index, style, showInfo, users } = props;

    return <div style={style} className={_style.table_row}>
        <span className={_style.td}>{index + 1}</span>
        <span className={_style.td}><TLabel className={_style.label}
                                            status={getStatusType(item.HistoryAction)}
                                            text={item.HistoryAction}/></span>
        <span className={_style.td}><FormatDate value={item.HistoryTimestamp * ONE_SECOND} withSecond/></span>
        <span className={_style.td}>
            <div>{item.Tag}</div>
            <div className={_style.tag_name}>{item.Name}</div>
        </span>
        <span className={_style.td}>
            {item.HistoryUserID
                ? <Link href={`#/clients/${item.HistoryUserID}/info`}>{users[item.HistoryUserID]?.Username}</Link>
                : EMPTY_DATA}
        </span>
        <span className={_style.td}>
            {item.Performer
                ? <Link href={`#/clients/${item.Performer}/info`}>{users[item.Performer]?.Username}</Link>
                : EMPTY_DATA}
        </span>
        <span className={_style.td}><Link onClick={showInfo}>info</Link></span>
    </div>;
});
