import moment from 'moment';
import React from 'react';

import { Dict } from '../../../types';
import { ONE_SECOND } from '../../constants';
import DatePicker from '../../ui/DatePicker';
import { Window } from '../../ui/FullModal';
import { Request2 } from '../../utils/request';
import Spin from '../Spin';
import { HistoryVirtualList } from './HistoryVirtualList';
import * as style from './index.css';
import { ANALYTICS_HISTORY as requestConfigs, REQUESTS } from './request';

export interface IUser {
    ID: string;
    Username: string;
}

export interface ITag {
    HistoryEventID: number;
    HistoryAction: string;
    HistoryTimestamp: number;
    HistoryUserID: string;
    HistoryOriginatorID: string;
    HistoryComment: string;
    ID: string;
    ObjectID: string;
    Tag: string;
    Performer: string;
    Priority: number;
    Snapshot: string;
    Data: any;
    Name: string;
    Comment: string;
    Type: string;
}

interface ITagsDeepHistory2State {
    [k: string]: any;

    rows: ITag[];
    users: Dict<IUser>;
    usersRevert: Dict<IUser>;
    isLoading: boolean;
    error: Error | null;
    Begin: any;
    End: any;
}

interface ITagsDeepHistory2Props {
    user_id?: string;
    car_id?: string;
    onClose: () => void;
}

export class TagsDeepHistory2 extends React.Component<ITagsDeepHistory2Props, ITagsDeepHistory2State> {
    request = new Request2({ requestConfigs });
    state: ITagsDeepHistory2State = {
        rows: [],
        users: {},
        usersRevert: {},
        isLoading: false,
        error: null,
        Begin: moment().subtract(1, 'month'),
        End: moment(),
    };

    componentDidMount(): void {
        this.getData();
    }

    getData() {
        this.setState({ isLoading: true, error: null }, async () => {
            let response;
            try {
                if (this.props.user_id) {
                    response = await this.request.exec(REQUESTS.GET_USER_TAGS, {
                        queryParams: {
                            user_id: this.props.user_id,
                            Begin: Math.ceil(this.state.Begin / ONE_SECOND),
                            End: Math.ceil(this.state.End / ONE_SECOND),
                        },
                    });
                }

                if (this.props.car_id) {
                    response = await this.request.exec(REQUESTS.GET_CARS_TAGS, {
                        queryParams: {
                            car_id: this.props.car_id,
                            Begin: Math.ceil(this.state.Begin / ONE_SECOND),
                            End: Math.ceil(this.state.End / ONE_SECOND),
                        },
                    });
                }

                this.setState({
                    isLoading: false,
                    rows: response.Rows?.reverse() || [],
                    users: (response.Users || []).reduce((_p, _c) => {
                        if (!_p.hasOwnProperty(_c.ID)) {
                            _p[_c.ID] = _c;
                        }

                        return _p;
                    }, {}),
                    usersRevert: (response.Users || []).reduce((_p, _c) => {
                        if (!_p.hasOwnProperty(_c.Username)) {
                            _p[_c.Username] = _c;
                        }

                        return _p;
                    }, {}),
                });
            } catch (error) {
                this.setState({
                    isLoading: false,
                    error,
                });
            }
        });
    }

    changeDate(field, value) {
        this.setState({
            [field]: value,
        }, () => {
            this.getData();
        });
    }

    render() {
        return <Window title={`Архив истории тегов пользователя ${this.state.rows && `(${this.state.rows.length})`}`}
                       className={style.archive}
                       error={this.state.error}
                       onClose={this.props.onClose}>
            <DatePicker value={this.state.Begin}
                        onChange={this.changeDate.bind(this, 'Begin')}
                        className={style.date}/>
            <DatePicker value={this.state.End}
                        onChange={this.changeDate.bind(this, 'End')}
                        className={style.date}/>
            {
                this.state.isLoading
                    ? <Spin/>
                    : <HistoryVirtualList users={this.state.users}
                                          usersRevert={this.state.usersRevert}
                                          rows={this.state.rows}/>
            }

        </Window>;
    }
}
