import * as React from 'react';

import { Dict } from '../../../../types';
import { Button, ButtonTypes } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { Window } from '../../../ui/FullModal';
import { Link } from '../../../ui/Link';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import { Request2 } from '../../../utils/request';
import { deepCopy } from '../../../utils/utils';
import Spin from '../../Spin';
import { REQUESTS, TRACKS_REQUESTS } from '../request';
import * as styles from './index.css';

interface ISensorsProps {
    imei: string;
    getInfo: (result: any) => void;
    className: string;
    carId: string;
    trace: any;
    removeTrace: () => void;
    removeSensors: () => void;
    drawTrace: () => void;
}

interface ISensorsState {
    error: Error | null;
    isLoading: boolean;
    isWorking: boolean;
    showModal: boolean;
    sensorsList: { name: string; id: number; subid: number }[];
    sensorsId: Dict<number>;
    showControls: boolean;
    resultArray: {
        id: string;
        subId?: string;
        name: string;
        value: number;
        coords: string[];
        timestamp: number;
    }[];
    showRemoveSensor: boolean;
    showRemoveTrace: boolean;
    errors: string[];
}

const MAX_SENSORS = 8;
const NO_MATCH_TIME = 10;
const FAVOURITE_SENSORS = ['fuel_level', 'engine_on', 'hand_brake_on', 'front_left_door_open'];

export class Sensors extends React.Component<ISensorsProps, ISensorsState> {
    state: ISensorsState = {
        error: null,
        isLoading: false,
        isWorking: false,
        showModal: false,
        sensorsList: [],
        sensorsId: {},
        showControls: false,
        resultArray: [],
        showRemoveSensor: true,
        showRemoveTrace: true,
        errors: [],
    };

    request = new Request2({ requestConfigs: TRACKS_REQUESTS });

    getData() {
        const { carId } = this.props;

        this.setState({
            error: null,
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_TELEMATICS_STATE, {
                queryParams: {
                    car_id: carId,
                },
            })
                .then((response) => {
                    this.setState({
                        sensorsList: response?.sensors
                            ?.sort((a, b) => a?.name.localeCompare(b?.name))
                            ?.sort((a, b) => {
                                return +FAVOURITE_SENSORS.includes(b?.name) - +FAVOURITE_SENSORS.includes(a?.name);
                            }) ?? [],
                        isLoading: false,
                    });
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    shouldComponentUpdate(nextProps: Readonly<ISensorsProps>, nextState: Readonly<ISensorsState>): boolean {
        return !isObjectEqual(this.state, nextState) || !isObjectEqual(this.props, nextProps);
    }

    toggleModal() {
        this.setState((prev) => ({
            showModal: !prev.showModal,
        }), () => {
            if (this.state.showModal) {
                this.getData();
            }
        });
    }

    toggleControls(showControls) {
        this.setState(() => ({
            showControls,
        }));
    }

    setSensorId({ id, name }) {
        const sensorsId = deepCopy(this.state.sensorsId);

        if (sensorsId[id]) {
            delete sensorsId[id];
        } else {
            if (Object.keys(sensorsId).length < MAX_SENSORS) {
                sensorsId[id] = name;
            }
        }

        this.setState({
            sensorsId,
        });
    }

    getSensorsInfoById() {
        const { imei, trace, removeSensors, getInfo } = this.props;
        const { sensorsId } = this.state;
        const sensorsIdKeys = Object.keys(sensorsId);
        const result: Dict<{ id; subId; name; value; coords? }[]> = {};
        const sensorsErrors: string[] = [];

        if (!sensorsIdKeys.length) {
            this.removeAllSensors();
            this.toggleModal();
        } else {
            this.setState({
                isWorking: true,
            }, async () => {

                const traceKeys = Object.keys(trace);
                const start_ts = traceKeys[0];
                const finish_ts = traceKeys[traceKeys.length - 1];

                try {
                    const response = await this.request.exec(REQUESTS.GET_TELEMATICS_HISTORY, {
                        queryParams: {
                            imei,
                        },
                    });

                    const responseKeys = Object.keys(response);

                    sensorsIdKeys.forEach((fullId) => {
                        const subId = fullId.split('_')[1] ? +fullId.split('_')[1] : undefined;
                        const id = +fullId.split('_')[0];

                        const sensorInfo = response[responseKeys[0]]
                            .find((sensor) => sensor.id === id && sensor?.subid === subId);
                        const sensorTimeline = [...sensorInfo?.timeline];

                        const shacked = sensorTimeline.reduce((_p, _c) => {
                            if (_p[_p.length - 1]?.value !== _c.value) {
                                _p.push(_c);
                            }

                            return _p;
                        }, []);

                        const shackedNext = shacked.reduce((_p, _c, index, array) => {
                            const nextAfterNext = 2;
                            if ((_c.ts < start_ts && array[index + nextAfterNext]?.ts >= start_ts)
                                || _c.ts >= start_ts && _c.ts <= finish_ts) {
                                _p.push(_c);
                            }

                            if ((_c.ts > finish_ts && (array[index - nextAfterNext]?.ts <= finish_ts))) {
                                _p.push(_c);
                            }

                            return _p;
                        }, []);

                        if (shackedNext.length) {
                            shackedNext.forEach((timestamp: { ts: number; value: number }) => {
                                if (!result[timestamp.ts]) {
                                    result[timestamp.ts] = [];
                                }

                                result[timestamp.ts].push({
                                    id: sensorInfo.id,
                                    subId: sensorInfo?.subid,
                                    name: sensorsId[fullId],
                                    value: timestamp.value,
                                });
                            });
                        } else {
                            sensorsErrors.push(`Нет данных по сенсору ${sensorsId[fullId]}`);
                        }
                    });

                    const resultKeys = Object.keys(result);
                    const resultArray: { name; id; subId; timestamp; value; coords; noMatch }[] = [];

                    resultKeys.forEach((timestamp) => {
                        const traceTimestamp: string = traceKeys
                            .reduce((prev, curr) => Math.abs(+curr - +timestamp) < Math.abs(+prev - +timestamp)
                                ? curr
                                : prev,
                            );
                        result[timestamp].forEach((el) => {
                            resultArray.push({
                                ...el,
                                coords: trace[traceTimestamp],
                                noMatch: Math.abs(+traceTimestamp - +timestamp) > NO_MATCH_TIME,
                                timestamp,
                            });
                        });
                    });

                    this.setState({
                        resultArray,
                        errors: sensorsErrors,
                        isWorking: false,
                    });

                    removeSensors();
                    getInfo({
                        result: resultArray,
                        errors: sensorsErrors,
                    });
                    this.toggleControls(true);
                    this.toggleModal();
                } catch (error) {
                    this.setState({
                        error,
                        isWorking: false,
                    });
                }
            });
        }
    }

    toggleTraceControl(showRemove: boolean) {
        const { drawTrace, getInfo, removeTrace } = this.props;
        const { resultArray, errors } = this.state;

        if (showRemove) {
            drawTrace();
            getInfo({
                result: resultArray,
                errors: errors,
            });
        } else {
            removeTrace();
        }

        this.setState({
            showRemoveTrace: showRemove,
        });
    }

    toggleSensorControl(showRemove) {
        const { removeSensors, getInfo } = this.props;
        const { resultArray, errors } = this.state;

        if (showRemove) {
            getInfo({
                result: resultArray,
                errors: errors,
            });
        } else {
            removeSensors();
        }

        this.setState({
            showRemoveSensor: showRemove,
        });
    }

    removeAllSensors() {
        const { removeSensors, getInfo } = this.props;

        removeSensors();
        getInfo(null);
        this.toggleControls(false);
        this.setState({
            sensorsId: {},
        });
    }

    componentWillUnmount() {
        this.request.abort();
    }

    render() {
        const {
            error,
            isLoading,
            showModal,
            sensorsList,
            sensorsId,
            showControls,
            showRemoveSensor,
            showRemoveTrace,
            isWorking,
        } = this.state;
        const { className } = this.props;
        const sensorsIdKeys = Object.keys(sensorsId);

        return (
            <>
                {showModal && (
                    <Window title={'Выбрать сенсоры для показа'}
                            error={error}
                            onClose={this.toggleModal.bind(this)}>
                        {isLoading ? <Spin/> : (
                            <div>
                                <h4 className={styles.warning}>
                                    Можно выбрать до {MAX_SENSORS} сенсоров:
                                </h4>
                                <div className={styles.sensorsList}>
                                    {sensorsList.map((item, index) => {
                                        const fullId = item?.subid ? `${item.id}_${item.subid}` : item.id;

                                        return (
                                            <React.Fragment key={`sensor-checkbox-${index}`}>
                                                <div className={styles.sensorCheckbox}
                                                     onClick={this.setSensorId.bind(this, {
                                                         id: fullId,
                                                         name: item.name,
                                                     })}>
                                                    <Checkbox checked={sensorsId.hasOwnProperty(fullId)}
                                                              disabled={sensorsIdKeys.length >= MAX_SENSORS
                                                        && !sensorsId.hasOwnProperty(fullId)}/>
                                                    <span>
                                                        {item.name != fullId &&
                                                        <span className={styles.full_id}>({fullId}) </span>}
                                                        {item.name}</span>
                                                </div>
                                                {index === FAVOURITE_SENSORS.length - 1 && <hr/>}
                                            </React.Fragment>
                                        );
                                    })}
                                </div>
                                <Button isLoading={isWorking}
                                        colorType={ButtonTypes.positive}
                                        className={styles.stickyButton}
                                        onClick={this.getSensorsInfoById.bind(this)}>
                                    {`Отобразить на карте ${sensorsIdKeys.length} сенсор(ов)`}
                                </Button>
                            </div>
                        )}
                    </Window>
                )}
                <div className={className}>
                    {showControls && <div className={styles.popup}>
                        {showRemoveSensor
                            ? <Link onClick={this.toggleSensorControl.bind(this, false)}
                                    className={styles.link}>
                                Убрать метки сенсоров
                            </Link>
                            : <Link onClick={this.toggleSensorControl.bind(this, true)}
                                    className={styles.link}>
                                Вернуть метки сенсоров
                            </Link>
                        }
                        {showRemoveTrace
                            ? <Link onClick={this.toggleTraceControl.bind(this, false)}
                                    className={styles.link}>
                                Убрать метки трека
                            </Link>
                            : <Link onClick={this.toggleTraceControl.bind(this, true)}
                                    className={styles.link}>
                                Вернуть метки трека
                            </Link>
                        }
                        {<Link onClick={this.removeAllSensors.bind(this)}
                               className={styles.link}>
                            Очистить выбор
                        </Link>}
                    </div>}
                    <Button className={styles.mapButton}
                            onClick={this.toggleModal.bind(this)}>Показать сенсоры</Button>
                </div>
            </>
        );
    }
}
