import * as React from 'react';
import { renderToStaticMarkup } from 'react-dom/server';
import VirtualList from 'react-tiny-virtual-list';
import ReactTooltip from 'react-tooltip';

import { Dict, ICar, IPoly } from '../../../types';
import { EMPTY_DATA, ONE_DAY, ONE_HOUR, ONE_SECOND } from '../../constants';
import { CarInfoHandler, IModel } from '../../models/car';
import SessionsHistoryItem, { SessionHistoryInfoHandler } from '../../models/session';
import { Button, ButtonTypes } from '../../ui/Button';
import DatePicker from '../../ui/DatePicker';
import { DragIndicator } from '../../ui/DragIndicator';
import FormatDate, { durationBetween, FormatDateInString } from '../../ui/FormatDate';
import { Input } from '../../ui/Input';
import { Link } from '../../ui/Link';
import { NoInformation } from '../../ui/NoInformation';
import Select from '../../ui/Select';
import { CSSettingsItems, CustomSettings } from '../../utils/customSettings';
import { Draggable } from '../../utils/Draggable';
import { hasWebphonePanel } from '../../utils/hasWebphonePanel';
import { isObjectEqual } from '../../utils/isObjectEqual';
import LS from '../../utils/localStorage/localStorage';
import { Request2, specialQueryParamsKeys } from '../../utils/request';
import { buttonLocationDetails, buttonNameDetails } from '../../utils/sendLogs/eventTypes/buttonDetails';
import { parseLocationQuery } from '../../utils/utils';
import { Copy } from '../Copy';
import DocForQueuePicker from '../Documents/DocForQueuePicker';
import { initMap } from '../MainMap/utils';
import { WIDTH_MAP_CONTROL } from '../Map2/helpers/widthMapControl';
import { drawFixPointElements, normalizeCoordinates } from '../MiniMap/utils';
import { BillingCalculatorButton } from '../SessionCard/modals/BillingCalculatorButton/component';
import { Address } from '../SessionCard/SessionCardHeader';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import { drawClientPolygonControl } from './drawClientPolygonControl';
import { drawFilterTraceControl } from './drawFilterTraceControl';
import { drawPolygonControl } from './drawPolygonControl';
import { drawShowAllStatusesControl } from './drawShowAllStatusesControl';
import * as styles from './index.css';
import { REQUESTS, TRACKS_REQUESTS } from './request';
import Sensors from './Sensors';
import { colorizeSensor } from './Sensors/colorizeSensor';

declare let ymaps: any;

const statuses = [
    { text: 'all', value: '', description: 'Все сегменты' },
    { text: 'ride', value: 'ride', description: 'Поездка' },
    { text: 'parking', value: 'parking', description: 'Парковка' },
    { text: 'reservation', value: 'reservation', description: 'Бронирование' },
    { text: 'reservation_paid', value: 'reserv_paid', description: 'Платное бронирование' },
    { text: 'post', value: 'post', description: 'После аренды' },
    { text: 'acceptance', value: 'acceptance', description: 'Осмотр' },
    { text: 'service', value: 'service', description: 'Сервис' },
];

const speedToKM = 3.6;
const parkingSpeed = 5;
const overSpeedBuffer = 10;

const tracelineProperties = {
    strokeColor: '#ca463ec8',
    strokeWidth: 4,
    strokeStyle: '2 3',
};

const trackProperties = {
    strokeColor: '#12ca4f',
    strokeWidth: 4,
};

const polylineProperties = {
    strokeColor: '#e0d600',
    strokeWidth: 4,
};

const SCORING_TRACE_TAG = 'scoring_trace_tag';

type ITrackMapState = {
    [x in string | number]: any;
} & {
    error: Error | null;
    isLoading: boolean;
    traceIsLoading: boolean;
    tracks: any[];
    session: SessionsHistoryItem | null;

    sessionId: string | null;
    carId: string | null;
    userId: string | null;
    status: string | null;
    since: string | number | null;
    until: string | number | null;
    trace: any;
    trace_id: string;
    sensorsCarId: string | null;

    expandConfiguration: boolean;
    expandTraceInfo: boolean;
    polygonControlsShows: boolean;
    calculatorSince: number;

    isDocQueueOpen: boolean;
    models: Dict<IModel>;
    sensorsInformation: Dict<any> | null;
    carInfo: ICar | null;

    showPoly: boolean;
    poly: IPoly[];
    traceTags: any;

    tableTsCoords: Dict<number[]>;
    tableCoord: any;
    tableTrace: any;
    tableVlimit: any;
    tableTimestamps: any;
    tableVdevice: any;
    selected: any;

    showLongTrackTooltip: boolean;
    preventLoadLongTrack: boolean;

    x: number;
    y: number;

    initRequest: boolean;
};

const POLYLINE_KEY = 'polyline';
const TABS = ['expandTraceInfo', 'expandConfiguration'];

const LEGEND_WIDTH = 280;
const MAP_PADDING = 10;
const FRACTION_DIGITS = 2;

const LONG_TRACK_DAYS = 5;
const LONG_TRACK_LIMIT = ONE_DAY * LONG_TRACK_DAYS;
const LONG_TRACK_TOOLTIP_ID = 'long-track-tooltip';
const LONG_TRACK_BUTTON_ID = 'force_tooltip_button';
const COPY_TOOLTIP = 'copy_tooltip';
const WEBPHONE_PADDING = 55;

const getMapBounds = (hasWebphonePanel: boolean) => {
    const webphonePadding = hasWebphonePanel ? WEBPHONE_PADDING : 0;

    return {
        x: window.innerWidth - LEGEND_WIDTH,
        y: window.innerHeight - webphonePadding,
    };
};

export class TrackMap extends React.Component<any, ITrackMapState> {
    state: ITrackMapState = {
        error: null,
        isLoading: true,
        traceIsLoading: false,
        tracks: [],
        session: null,

        sessionId: null,
        carId: null,
        userId: null,
        status: null,
        since: 0,
        until: 0,
        sensorsCarId: null,

        trace: null,
        trace_id: '',

        expandConfiguration: false,
        expandTraceInfo: false,
        polygonControlsShows: false,
        isDocQueueOpen: false,
        calculatorSince: 0,
        models: {},
        sensorsInformation: null,
        carInfo: null,

        showPoly: false,
        poly: [],

        tableTsCoords: {},
        tableCoord: [],
        tableTrace: [],
        tableVlimit: [],
        tableTimestamps: [],
        tableVdevice: [],
        selected: [],
        traceTags: null,

        x: getMapBounds(!!hasWebphonePanel(location.href, this.props.BlockRules)).x,
        y: getMapBounds(!!hasWebphonePanel(location.href, this.props.BlockRules)).y,

        preventLoadLongTrack: true,

        initRequest: false,
        showLongTrackTooltip: false,
    };

    map;
    searchPolygon;
    mySearchCollection;
    mySearchBoundsCollection;
    signsCollection;
    lineCollection;
    fixCollection;
    traceCollection;
    scoringTraceTagCollection;
    polygonsCollection;
    traceObjectManager;
    sensorsObjectManager;

    poly;

    listBoxControl;
    showAllControl;
    clientPolygonControl;
    polygonsControl;
    fixControl;

    tracksRef;
    configRef;

    constructor(props) {
        super(props);
        this.tracksRef = React.createRef<HTMLDivElement>();
        this.configRef = React.createRef<HTMLDivElement>();
    }

    request = new Request2({ requestConfigs: TRACKS_REQUESTS });
    ls = new LS();
    cs = new CustomSettings();

    legendMove(x, y) {
        this.setState({
            x, y,
        });
    }

    buildTrack() {
        const { userId, carId, sessionId, since, until, status } = this.state;
        const newQueries = new URLSearchParams();

        if (userId) {
            newQueries.append('user_id', userId);
        }

        if (carId) {
            newQueries.append('car_id', carId);
        }

        if (sessionId) {
            newQueries.append('session_id', sessionId);
        }

        if (since) {
            newQueries.append('since', String(since));
        }

        if (until) {
            newQueries.append('until', String(until));
        }

        if (status || status === '') {
            newQueries.append('status', status);
        }

        if (location && location.href) {
            location.href = `#/tracks?${newQueries.toString()}`;
        }

        this.state.showLongTrackTooltip && this.setState({
            //don't correct this expression in WebStorm!!!
            preventLoadLongTrack: !!(since && until && +since && +until && (+until - +since) > LONG_TRACK_LIMIT),
        });
    }

    initMap() {
        initMap('trackMap', (map) => {
            this.map = map;

            this.map.options.set('minZoom', 1);
            this.map.options.set('autoFitToViewport', 'always');
            this.map.options.set('suppressMapOpenBlock', true);
            this.map.options.set('copyrightLogoVisible', false);
            this.map.options.set('copyrightProvidersVisible', false);
            this.map.options.set('copyrightUaVisible', false);
            this.signsCollection = new ymaps.Clusterer({
                groupByCoordinates: true,
                clusterIconLayout: 'default#pieChart',
            });
            this.lineCollection = new ymaps.GeoObjectCollection();
            this.fixCollection = new ymaps.GeoObjectCollection();
            this.polygonsCollection = new ymaps.GeoObjectCollection();
            this.traceCollection = new ymaps.GeoObjectCollection();
            this.scoringTraceTagCollection = new ymaps.GeoObjectCollection();

            this.traceObjectManager = new ymaps.ObjectManager({
                clusterize: true,
                clusterIconLayout: 'default#pieChart',
                gridSize: 64,
            });

            this.map.geoObjects.add(this.signsCollection);
            this.map.geoObjects.add(this.fixCollection);
            this.map.geoObjects.add(this.polygonsCollection);
            this.map.geoObjects.add(this.traceCollection);
            this.map.geoObjects.add(this.scoringTraceTagCollection);
            this.map.geoObjects.add(this.traceObjectManager);
            this.map.geoObjects.add(this.lineCollection);

            this.mySearchCollection = new ymaps.GeoObjectCollection();
            this.map.geoObjects.add(this.mySearchCollection);

            this.mySearchBoundsCollection = new ymaps.GeoObjectCollection();
            this.map.geoObjects.add(this.mySearchBoundsCollection);

            this.sensorsObjectManager = new ymaps.ObjectManager({
                clusterize: true,
                gridSize: 64,
                clusterIconLayout: 'default#pieChart',
            });
            this.sensorsObjectManager.clusters.options.set({
                clusterIconPieChartRadius: 22,
            });

            this.map.geoObjects.add(this.sensorsObjectManager);

            const zoomControl = new ymaps.control.ZoomControl({
                options: {
                    size: 'small',
                },
            });

            const typeSelector = new ymaps.control.TypeSelector([]);
            typeSelector.addMapType('yandex#map');
            typeSelector.addMapType('yandex#hybrid');
            typeSelector.addMapType('yandex#satellite');

            const clusterBtn = new ymaps.control.Button({
                data: { content: 'Кластер' },
                options: {
                    maxWidth: WIDTH_MAP_CONTROL,
                },
            });
            clusterBtn?.events
                .add('select', () => {
                    this.traceObjectManager.options.set({ clusterize: false });
                })
                .add('deselect', () => {
                    this.traceObjectManager.options.set({ clusterize: true });
                });

            const togglePins = new ymaps.control.Button({
                data: { content: 'Скрыть "хвосты"' },
                options: {
                    maxWidth: WIDTH_MAP_CONTROL,
                },
            });

            togglePins?.events
                .add('select', () => {
                    const objects = this.signsCollection?.getGeoObjects();
                    objects.forEach(el => {
                        el.options.set('visible', false);
                    });
                    this.signsCollection?.options?.set('visible', false);

                })
                .add('deselect', () => {
                    this.signsCollection?.options?.set('visible', true);
                    const objects = this.signsCollection?.getGeoObjects();
                    objects.forEach(el => {
                        el.options.set('visible', true);
                    });
                });

            const SearchPolygon = new ymaps.control.Button({
                data: { content: 'Поиск в полигоне' },
                options: {
                    maxWidth: WIDTH_MAP_CONTROL,
                },
            });

            SearchPolygon?.events
                .add('select', () => {
                    this.setState({
                        polygonControlsShows: true,
                    }, () => {

                        this.createSearchPoly();
                        this.request.exec(REQUESTS.GET_CARS)
                            .then(response => {
                                const models: Dict<IModel> = response.models || {};
                                this.setState({ models });
                            });
                    });
                })
                .add('deselect', () => {
                    this.clearSearchPoly();
                    this.setState({
                        polygonControlsShows: false,
                    });
                });

            if (!localStorage.getItem('searchByPolygon')) {
                localStorage.setItem('searchByPolygon', '0');
            }

            if (this.props.BlockRules.searchPoly) {
                this.map.controls.add(SearchPolygon, { float: 'right' });
            }

            this.fixControl = new ymaps.control.Button({
                data: { content: 'Фикс' },
                options: {
                    maxWidth: WIDTH_MAP_CONTROL,
                },
            });

            this.fixControl?.events
                .add('select', () => {
                    drawFixPointElements({
                        offer: SessionHistoryInfoHandler.getCurrentOffer.call(this.state.session),
                        collection: this.fixCollection,
                    });
                })
                .add('deselect', () => {
                    this.fixCollection.removeAll();
                });

            const searchControl = new ymaps.control.SearchControl({
                options: {
                    provider: 'yandex#search',
                },
            });
            this.map.controls.add(togglePins, { float: 'right' });
            this.map.controls.add(zoomControl);
            this.map.controls.add(typeSelector);
            this.map.controls.add(clusterBtn, { float: 'right' });
            this.map.controls.add(this.fixControl, { float: 'right' });
            this.map.controls.add(searchControl, { float: 'right' });

            this.getData();
        });
    }

    async getData() {
        const queries = new URLSearchParams(this.props.location.search);
        const trackInfo: { [key: string]: any } = {};
        const specialQueryParams = queries.has('status') ? {
            specialQueryParams: {
                status: {
                    key: specialQueryParamsKeys.FORCE,
                    value: status,
                },
            },
        } : { specialQueryParams: {} };

        for (const queryEntry of queries.entries()) {
            switch (queryEntry[0]) {
            case 'session_id':
                trackInfo.sessionId = queryEntry[1];
                break;
            case 'user_id':
                trackInfo.userId = queryEntry[1];
                break;
            case 'car_id':
                trackInfo.carId = queryEntry[1];
                break;
            case 'since':
                trackInfo.since = Number(queryEntry[1]);
                break;
            case 'until':
                trackInfo.until = Number(queryEntry[1]);
                break;
            case 'acceptAllQueries':
                break;
            default:
                const queryValue = queryEntry[1];
                if (queries.get('acceptAllQueries') === '1') {
                    specialQueryParams.specialQueryParams[queryEntry[0]] = {
                        key: specialQueryParamsKeys.FORCE_ABSENT_KEY,
                        value: queryValue,
                    };
                }

                break;
            }
        }

        let session = this.state.session;
        let carInfo: ICar | null = null;
        let tracks;
        let sensorsCarId;
        let sessionStart;
        let sessionFinish;
        let traceTags;

        let showLongTrackTooltip = false;

        const currentSessionId = trackInfo.sessionId || this.props.sessionId;
        if (trackInfo.userId || trackInfo.carId || currentSessionId) {
            try {
                if (currentSessionId && currentSessionId !== this.state.sessionId) {
                    session = await this.request.exec(REQUESTS.GET_SESSION, {
                        queryParams: {
                            session_id: currentSessionId,
                        },
                    });
                }

                sessionStart = (trackInfo.since || SessionHistoryInfoHandler.getStart.call(session)) ?? 0;
                sessionFinish = (trackInfo.until || SessionHistoryInfoHandler.getFinish.call(session)) ?? 0;

                if (sessionStart && !sessionFinish) {
                    sessionFinish = Date.now();
                }

                if (!this.state.preventLoadLongTrack || sessionFinish - sessionStart < LONG_TRACK_LIMIT) {
                    tracks = await this.request.exec(REQUESTS.GET_TRACK, {
                        queryParams: {
                            session: currentSessionId,
                            car_id: trackInfo.carId,
                            user_id: trackInfo.userId,

                            since: trackInfo.since ? (trackInfo.since / ONE_SECOND).toFixed(0) : null,
                            until: trackInfo.until ? (trackInfo.until / ONE_SECOND).toFixed(0) : null,
                        },
                        ...specialQueryParams,
                    });
                    tracks = tracks.tracks?.sort((track1, track2) => track1.start_timestamp - track2.start_timestamp);
                } else {
                    showLongTrackTooltip = true;
                }

                sensorsCarId = trackInfo.carId || (currentSessionId ? tracks?.[0]?.car : null);
                if (sensorsCarId !== this.state.sensorsCarId && sensorsCarId) {
                    carInfo = await this.request.exec(REQUESTS.GET_CAR_INFO, {
                        queryParams: {
                            car_id: sensorsCarId,
                        },
                    }) as ICar;
                }

            } catch (error) {
                this.setState({
                    error,
                });
            }

            try {
                if (currentSessionId) { // #DRIVEFRONT-1280
                    traceTags = await this.request.exec(REQUESTS.GET_TRACE_TAG_LIST, {
                        queryParams: {
                            object_id: currentSessionId,
                        },
                    });
                }
            } catch (error) {

            }
        }

        this.setState((prev) => ({
            sessionId: currentSessionId,
            carId: trackInfo.carId,
            userId: trackInfo.cardId,
            status,
            session,
            since: showLongTrackTooltip ? sessionStart : trackInfo.since,
            until: showLongTrackTooltip ? sessionFinish : trackInfo.until,
            carInfo: carInfo ?? prev.carInfo,
            tracks,
            showLongTrackTooltip,
            sensorsCarId: sensorsCarId ?? prev.sensorsCarId,
            expandConfiguration: showLongTrackTooltip,
            isLoading: false,
            traceTags: traceTags?.records?.find(item => item.tag === SCORING_TRACE_TAG),
        }), () => {
            this.drawMap();
            this.setState({
                y: getMapBounds(!!hasWebphonePanel(location.href, this.props.BlockRules))
                    .y - this.tracksRef?.current?.offsetHeight - MAP_PADDING,
                preventLoadLongTrack: true,
            });

            this.map.events.add('sizechange', () => {
                this.setState({
                    x: getMapBounds(!!hasWebphonePanel(location.href, this.props.BlockRules)).x,
                    y: getMapBounds(!!hasWebphonePanel(location.href, this.props.BlockRules))
                        .y - this.tracksRef?.current?.offsetHeight - MAP_PADDING,
                });
            });
        });
    }

    drawTrack(tracks) {
        const fixSetting = this.cs.get(CSSettingsItems.showFixTrack);

        if (fixSetting) {
            this.fixControl?.select();
        }

        tracks && tracks
            .forEach((item: any, index: number) => {
                const line: number[][] = normalizeCoordinates(item.track);

                const balloon = {
                    balloonContentBody: renderToStaticMarkup(<TrackTooltip index={index} item={item}/>),
                };

                const polyline = new ymaps.Polyline(line, Object.assign({
                    hintContent: 'поездка ' + index + ' '
                        + new Date(item.start_timestamp * ONE_SECOND).toLocaleString() + '-'
                        + new Date(item.finish_timestamp * ONE_SECOND).toLocaleString(),
                    id: item.ride,
                }, balloon), trackProperties);

                this.lineCollection.add(polyline);

                if (line.length && item.status && item.status === 'ride') {
                    const start = new ymaps.Placemark(line[0],
                        Object.assign({
                            iconContent: 'Начало ' + (index + 1),
                            clusterCaption: 'Начало ' + (index + 1),
                        }, balloon),
                        {
                            preset: 'islands#darkBlueStretchyIcon',
                        },
                    );

                    const end = new ymaps.Placemark(line[line.length - 1],
                        Object.assign({
                            iconContent: 'Конец  ' + (index + 1),
                            clusterCaption: 'Конец  ' + (index + 1),
                        }, balloon),
                        {
                            preset: 'islands#darkOrangeStretchyIcon',
                        },
                    );
                    this.signsCollection.add(start);
                    this.signsCollection.add(end);
                }

                if (item.status && item.status !== 'ride') {
                    if (line && line.length === 1) {
                        const newStatusMark = new ymaps.Placemark(line[line.length - 1],
                            Object.assign({
                                iconContent: item.status,
                                clusterCaption: item.status,
                            }, balloon),
                            {
                                preset: 'islands#yellowStretchyIcon',
                            },
                        );

                        this.signsCollection.add(newStatusMark);
                    } else if (line && line.length > 1) {
                        const polylineNewStatus = new ymaps.Polyline(line, Object.assign({
                            hintContent: `${item.status} `
                                + new Date(item.start_timestamp * ONE_SECOND).toLocaleString() + '-'
                                + new Date(item.finish_timestamp * ONE_SECOND).toLocaleString(),
                            id: item.ride,
                        }, balloon), polylineProperties);

                        const newStatusMarkStart = new ymaps.Placemark(line[0],
                            Object.assign({
                                iconContent: `${item.status} - начало`,
                                clusterCaption: `${item.status} - начало`,
                            }, balloon),
                            {
                                preset: 'islands#yellowStretchyIcon',
                            },
                        );
                        const newStatusMarkEnd = new ymaps.Placemark(line[line.length - 1],
                            Object.assign({
                                iconContent: `${item.status} - конец`,
                                clusterCaption: `${item.status} - конец`,
                            }, balloon),
                            {
                                preset: 'islands#yellowStretchyIcon',
                            },
                        );

                        this.signsCollection.add(newStatusMarkStart);
                        this.signsCollection.add(newStatusMarkEnd);
                        this.lineCollection.add(polylineNewStatus);
                    }
                }

                if (item.violations) {
                    item.violations.forEach((violation: any) => {
                        const violationBody = [
                            `<div>Ограничение: <strong>${violation && violation.limit && violation.limit}</strong></div>`
                            + `<div>Скорость клиента: <strong>${violation && violation.peak && +violation.peak.toFixed(FRACTION_DIGITS)}</strong></div>`
                            + `<div><a href="https://n.maps.yandex.ru/#!/?z=18&ll=${violation && Array.isArray(violation.center) && violation.center[0]},`
                            + `${violation && Array.isArray(violation.center) && violation.center[1]}&l=nk#sat" target="_blank">Народная карта</a></div>`,
                        ].join('');
                        const sign = new ymaps.Placemark(violation && violation.center, {
                            hintContent: (violation && violation.limit && violation.limit) + '/' +
                                    (violation && violation.peak && violation.peak.toFixed(FRACTION_DIGITS)),
                            iconContent: violation && violation.limit && violation.limit,
                            balloonContent: !this.props.isScoringClassification ? violationBody : null,
                        },
                        {
                            preset: 'islands#redCircleIcon',
                        },
                        );

                        this.lineCollection.add(sign);

                        if (this.props.isScoringClassification) {
                            sign.events.add('click', () => {
                                this.props.setViolation(violation.center);
                            });
                        }
                    });
                }
            });
    }

    async drawMap() {
        const { sessionId, initRequest, tracks } = this.state;
        const polySetting = this.cs.get(CSSettingsItems.showPolyTrack);
        const arePolyShowing = this.clientPolygonControl?.isSelected() ?? polySetting;

        this.removePoints();

        if (initRequest) {
            this.setState({
                initRequest: false,
            });
        }

        this.clientPolygonControl && this.map.controls.remove(this.clientPolygonControl);
        this.polygonsControl && this.map.controls.remove(this.polygonsControl);

        if (!this.showAllControl) {
            this.showAllControl = drawShowAllStatusesControl(
                this.map,
                this.showAllControl,
                this.state.status,
                this.props.location.search,
                this.manageStatus.bind(this),
            );
        }

        if (sessionId && tracks?.length) {
            this.clientPolygonControl = drawClientPolygonControl(
                {
                    sessionId,
                    polyConfig: this.props.polyConfig,
                    polygonsCollection: this.polygonsCollection,
                    map: this.map,
                    polyControlPairData: {
                        polyFilters: this.props.polyFilters,
                        BlockRules: this.props.BlockRules,
                    },
                },
                this,
            );
            arePolyShowing && this.clientPolygonControl.select();
        } else {
            this.polygonsControl = drawPolygonControl({
                polyFilters: this.props.polyFilters,
                BlockRules: this.props.BlockRules,
                map: this.map,
                polygonsCollection: this.polygonsCollection,
            });
        }

        this.drawTrack(this.state.tracks);
        this.map.geoObjects.getBounds() && this.map.setBounds(this.map.geoObjects.getBounds());
        this.showAllControl.enable();
    }

    addClusterPoint(props: { index; coordinates; balloonContentBody; preset; hintContent }) {
        const { index, coordinates, balloonContentBody, preset, hintContent } = props;

        return {
            type: 'Feature',
            id: index,
            geometry: {
                type: 'Point',
                coordinates: coordinates[index],
            },
            properties: {
                iconContent: index,
                clusterCaption: index,
                hintContent,
                balloonContentBody,
            },
            options: {
                preset,
            },
        };
    }

    manageStatus(status) {
        this.setState({
            status,
        }, () => {
            this.buildTrack();
        });
    }

    getTrackInfo(ride: string) {
        this.map.geoObjects?.each((obj: any) => {
            if (obj?.properties?.get?.('id') === ride) {
                obj.balloon.open(this.map.getCenter());
            }
        });
    }

    trace(ride: string) {
        this.setState({
            traceIsLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_TRACE, {
                queryParams: {
                    ride,
                },
            })
                .then((response: any) => {
                    const traceTrack = response.tracks && response.tracks[0];

                    if (traceTrack) {
                        this.setState({
                            trace: traceTrack,
                            trace_id: traceTrack.trace_id,
                        }, () => {
                            this.drawTrace(this.state.trace);
                        });
                    }
                })
                .catch((error) => {
                    this.setState({
                        error,
                    });
                });
        });
    }

    drawTrace(traceTrack) {
        this.removePoints({ keepPoly: true });
        this.map.controls.remove(this.listBoxControl);

        let trace = traceTrack.coords && traceTrack.coords.trim().split(' ');
        let vLimit: any[] = traceTrack.v_limit && traceTrack.v_limit.split(' ');
        let timestamps: any[] = traceTrack.timestamps && traceTrack.timestamps.split(' ');
        const coord: any[] = [];
        let vDevice: any[] = traceTrack.v_device && traceTrack.v_device.split(' ');
        let tableTsCoords = {};
        const oddDiv = 2;

        let sinceIndex = 0;
        let untilIndex = vDevice.length;

        timestamps.forEach((ts, tsIndex) => {
            if (this.state.since && !sinceIndex && ts >= +this.state.since / ONE_SECOND) {
                sinceIndex = tsIndex;
            }

            if (this.state.until && ts <= +this.state.until / ONE_SECOND) {
                untilIndex = tsIndex;
            }
        });

        //trace contains pairs of coords in a flat array
        trace = trace.slice(sinceIndex * oddDiv, untilIndex * oddDiv);
        vDevice = vDevice.slice(sinceIndex, untilIndex);
        vLimit = vLimit.slice(sinceIndex, untilIndex);
        timestamps = timestamps.slice(sinceIndex, untilIndex);

        trace.forEach((item, index) => {
            index % oddDiv == 0 && coord.push(trace.slice(index, index + oddDiv));
        });

        timestamps.forEach((ts, index) => {
            tableTsCoords = {
                ...tableTsCoords,
                [ts]: coord[index],
            };
        });

        this.setState({
            tableTsCoords,
            tableCoord: coord,
            tableTrace: trace,
            tableVlimit: vLimit,
            tableTimestamps: timestamps,
            tableVdevice: vDevice,
            traceIsLoading: false,
        });

        const coordinates = trace && Array.isArray(trace) && trace.length
            ? normalizeCoordinates(trace)
            : undefined;

        if (coordinates) {
            const traceline = new ymaps.Polyline(coordinates, {}, tracelineProperties);

            const arr: any[] = [];
            vDevice.forEach((velocity: any, index: number) => {
                const balloonContentBody = renderToStaticMarkup(
                    <TraceTooltip vDevice={vDevice[index]}
                                  vLimit={vLimit[index]}
                                  timestamps={timestamps[index]}
                                  coordinates={coordinates && coordinates[index]}/>,
                );

                if (+velocity * speedToKM < parkingSpeed) {
                    arr.push(this.addClusterPoint({
                        index,
                        coordinates,
                        balloonContentBody,
                        hintContent: 'Стоянки',
                        preset: 'islands#violetIcon',
                    }));
                } else if ((+vLimit[index] > 0)
                    && (+velocity >= +vLimit[index] * speedToKM + overSpeedBuffer)) {
                    arr.push(this.addClusterPoint({
                        index,
                        coordinates,
                        balloonContentBody,
                        hintContent: 'Превышения',
                        preset: 'islands#redIcon',
                    }));
                } else {
                    arr.push(this.addClusterPoint({
                        index,
                        coordinates,
                        balloonContentBody,
                        hintContent: 'Остальное',
                        preset: 'islands#greenIcon',
                    }));
                }
            });

            const FeatureCollection = {
                'type': 'FeatureCollection',
                'features': arr,
            };

            this.traceObjectManager.add(FeatureCollection);
            this.traceCollection.add(traceline);
            this.listBoxControl = drawFilterTraceControl(this.map, this.listBoxControl, this.traceObjectManager);
        }
    }

    removeTrace(removeAll: boolean) {
        this.map.controls.remove(this.listBoxControl);
        this.traceObjectManager && this.traceObjectManager.removeAll();

        if (removeAll) {
            this.setState({
                trace: null,
                tableTsCoords: {},
                sensorsInformation: null,
            }, () => {
                this.traceCollection && this.traceCollection.removeAll();
                this.sensorsObjectManager && this.sensorsObjectManager.removeAll();
                this.drawTrack(this.state.tracks);
            });
        }
    }

    expandTab(key) {
        TABS.forEach((tab) => tab !== key ? this.closeTab(tab) : undefined);

        this.setState((prev) => ({
            [key]: !prev[key],
        }));
    }

    closeTab(key) {
        this.setState({
            [key]: false,
        });
    }

    onChange(key, value) {
        this.setState({
            [key]: value,
        });
    }

    removePoints(props: { keepPoly } = { keepPoly: false }) {
        this.traceObjectManager && this.traceObjectManager.removeAll();
        this.traceCollection && this.traceCollection.removeAll();
        this.signsCollection && this.signsCollection.removeAll();
        this.lineCollection && this.lineCollection.removeAll();
        this.sensorsObjectManager && this.sensorsObjectManager.removeAll();

        if (!props.keepPoly) {
            this.polygonsCollection && this.polygonsCollection.removeAll();
        }
    }

    shouldComponentUpdate(nextProps: Readonly<any>, nextState: Readonly<ITrackMapState>): boolean {
        return !isObjectEqual(this.state, nextState) || !isObjectEqual(this.props, nextProps);
    }

    componentDidUpdate(prevProps: Readonly<any>, prevState: Readonly<ITrackMapState>) {
        const hadHasStatus = prevProps.location.search.includes('status=')
            || this.props.location.search.includes('status=');
        const forceUpdateLongTrack = prevState.preventLoadLongTrack !== this.state.preventLoadLongTrack
            && !this.state.preventLoadLongTrack;

        if ((this.props.location.search !== prevProps.location.search
            && !(this.state.initRequest&& hadHasStatus))
            || forceUpdateLongTrack) {
            this.showAllControl?.disable();

            this.setState({
                isLoading: true,
            }, () => {
                this.getData();
            });
        }

        if (this.state.showLongTrackTooltip !== prevState.showLongTrackTooltip) {
            this.manageLongTrackTooltip();
        }

        if (!isObjectEqual(this.state.traceTags, prevState.traceTags)) {
            this.showScoringTagsOnMap(this.state.traceTags);
        }
    }

    translateTypeOfViolation(type) {
        const TYPES_TRANSLATE = {
            acceleration: 'Резкое ускорение',
            braking: 'Резкое торможение',
            straight_lateral_acceleration: 'Резкий поворот',
            turning_lateral_acceleration: 'Небезопасный манёвр',
        };

        return TYPES_TRANSLATE[type];
    }

    showScoringTagsOnMap(tag) {
        const { ShowKindOfScoringEvents } = this.props.BlockRules;

        const events = tag?.events || [];
        this.scoringTraceTagCollection?.removeAll();
        events.map((el, index) => {
            const iconDate = FormatDateInString({
                value: el.timestamp * ONE_SECOND,
                withSecond: true,
            });

            const iconCaption = ShowKindOfScoringEvents
                ? (this.translateTypeOfViolation(el.kind) ?? `Резкость`) + `, ${iconDate}`
                : `Резкость-${index + 1}, ${iconDate}`;

            const MARK = new ymaps.Placemark(el.location, {
                iconCaption,
            }, {
                preset: 'islands#redCircleDotIconWithCaption',
                iconCaptionMaxWidth: '350',
            });

            this.scoringTraceTagCollection.add(MARK);

            if (this.props.isScoringClassification) {
                MARK.events.add('click', () => {
                    this.props.setScoring(el.location);
                });
            }
        });
    }

    async manageLongTrackTooltip() {
        if (this.state.showLongTrackTooltip) {
            await ReactTooltip.show(this.configRef?.current);
            document.getElementById(LONG_TRACK_BUTTON_ID)?.addEventListener('click', this.showLongTrack.bind(this));
        } else {
            document.getElementById(LONG_TRACK_BUTTON_ID)?.removeEventListener('click', this.showLongTrack.bind(this));
            ReactTooltip.hide(this.configRef?.current);
        }
    }

    componentDidMount() {
        const showAllSetting = this.cs.get(CSSettingsItems.showAllTrack);
        const queries = new URLSearchParams(this.props.location.search);
        const status = queries.get('status');

        if (status === '' && !showAllSetting) {
            queries.delete('status');

            if (location && location.href) {
                location.href = `#/tracks?${queries.toString()}`;
            }
        }

        this.setState({
            initRequest: true,
        }, () => {
            this.initMap();
        });
    }

    componentWillUnmount() {
        this.map && this.map.destroy();
        this.request.abort();
        document.getElementById(LONG_TRACK_BUTTON_ID)?.removeEventListener('click', this.showLongTrack.bind(this));
    }

    createSearchPoly(e?: KeyboardEvent) {
        e && e.preventDefault();
        this.mySearchCollection && this.mySearchCollection.removeAll();
        this.mySearchBoundsCollection && this.mySearchBoundsCollection.removeAll();

        this.searchPolygon = new ymaps.Polygon([], {}, {
            // Курсор в режиме добавления новых вершин.
            editorDrawingCursor: 'crosshair',
            // Максимально допустимое количество вершин.
            editorMaxPoints: 5,
            minPoints: 4,
            // Цвет заливки.
            fillColor: '#00FF0000',
            // Цвет обводки.
            strokeColor: '#0000FF',
            // Ширина обводки.
            strokeWidth: 2,
            draggable: true,
            zIndex: 2,
        });
        // Добавляем многоугольник на карту.
        this.map && this.mySearchCollection && this.mySearchCollection.add(this.searchPolygon);

        // В режиме добавления новых вершин меняем цвет обводки многоугольника.
        const stateMonitor = new ymaps.Monitor(this.searchPolygon.editor.state);
        stateMonitor.add('drawing', (newValue: any) => {
            this.searchPolygon.options.set('strokeColor', newValue ? '#FF0000' : '#0000FF');
        });

        this.searchPolygon.editor.startDrawing();
    }

    clearSearchPoly(e?: KeyboardEvent) {
        e && e.preventDefault();
        this.mySearchCollection && this.mySearchCollection.removeAll();
        this.mySearchBoundsCollection && this.mySearchBoundsCollection.removeAll();
    }

    openDocQueue() {
        this.setState({ isDocQueueOpen: true });
    }

    closeDocQueue() {
        this.setState({ isDocQueueOpen: false });
    }

    getDocInitialData() {
        const searchValues = parseLocationQuery(this.props.location.search);

        const coords = this.mySearchCollection.getBounds() || [];

        return Object.assign({}, {
            use_location_filter: true,
            session_start_timestamp: (+new Date() - ONE_HOUR) / ONE_SECOND,
            session_finish_timestamp: +new Date() / ONE_SECOND,
            polyline: coords,
        }, searchValues);
    }

    selectRow(index: number) {
        const isSelected = this.state.selected[index];
        this.setState({
            selected: {
                ...this.state.selected,
                [index]: !isSelected,
            },
        }, () => {
            if (!isSelected) {
                const point = new ymaps.Placemark(
                    this.state.tableCoord[index],
                    { traceIndex: index, iconContent: index },
                );
                this.traceCollection.add(point);
            } else {
                this.traceCollection.each((item: any) => {
                    if (item.properties.get('traceIndex') == index) {
                        this.traceCollection.remove(item);
                    }
                });
            }
        });
    }

    manageCalculator(value) {
        this.setState({
            calculatorSince: value,
        });
    }

    getSensorsInformation(sensorsInformation) {
        this.setState({
            sensorsInformation,
        });

        if (sensorsInformation?.result?.length) {
            const colors: Dict<string> = {};
            const noMatchColor = 'islands#grayStretchyIcon';
            const maxLetters = 12;
            const sensorPoints = sensorsInformation.result.map((el, index) => {
                const balloon = {
                    clusterCaption: `${el.name} ${el.value}`,
                    balloonContentBody: renderToStaticMarkup(
                        <SensorTooltip value={el.value}
                                       name={el.name}
                                       id={el.id}
                                       subid={el?.subId}
                                       timestamp={el.timestamp}/>,
                    ),
                };

                if (!colors[el.name]) {
                    colors[el.name] = colorizeSensor(Object.keys(colors).length);
                }

                return (
                    {
                        type: 'Feature',
                        id: `sensor-${index}`,
                        geometry: {
                            type: 'Point',
                            coordinates: el.coords,
                        },
                        properties: {
                            iconContent:
                                `${el.name.length > maxLetters ? `${el.name.substr(0, maxLetters)}...` : el.name}` +
                                ` ${el.value}`,
                            hintContent: el.name,
                            ...balloon,
                        },
                        options: {
                            preset: el.noMatch ? noMatchColor : colors[el.name],
                        },
                    }
                );
            });

            const FeatureCollection = {
                'type': 'FeatureCollection',
                'features': sensorPoints,
            };

            this.sensorsObjectManager.add(FeatureCollection);
        }
    }

    removeSensors() {
        this.sensorsObjectManager && this.sensorsObjectManager.removeAll();
    }

    showLongTrack() {
        this.setState({
            preventLoadLongTrack: false,
        });
    }

    changePosition(location) {
        const prettyZoomLevel = 15;
        this.map?.setCenter(location);
        this.map.setZoom(prettyZoomLevel);
    }

    render() {
        //DRIVEFRONT-1672
        const { isScoringClassification } = this.props;

        return <>
            {this.state.error ? <SimpleError error={this.state.error}/> : null}

            {!this.state.tracks?.length && !this.state.isLoading && !this.state.showLongTrackTooltip
                ? <div className={`${styles.mapNotification} ${styles.mapData}`}>
                    Нет данных
                </div>
                : this.state.isLoading
                    ? <div className={`${styles.mapNotification} ${styles.mapData}`}>
                        <span>Загрузка информации</span>
                        <Spin size={'s'}/>
                    </div>
                    : null
            }
            {!isScoringClassification
                ? <ScoringTraceLabel traceTags={this.state.traceTags} onClick={this.changePosition.bind(this)}/>
                : null
            }
            <div className={styles.sensorError}>
                {this.state.sensorsInformation?.errors?.map((error, index) => (
                    <div key={`error-${index}`}
                         className={styles.mapNotification}>
                        {error}
                    </div>
                ))}
            </div>
            <div id={'trackMap'}
                 className={styles.map}/>
            {!this.state.isLoading && this.state.polygonControlsShows && !isScoringClassification &&
                <div className={styles.polygon_search}>
                    <div className={styles.item_control}>
                        <Link onClick={this.createSearchPoly.bind(this)}>Отметить полигон</Link>&nbsp;&nbsp;
                        <Link className={styles.clear} onClick={this.clearSearchPoly.bind(this)}>Отмена</Link>
                    </div>
                    <div>
                        <Button basic
                                onClick={this.openDocQueue.bind(this)}>Добавить документ в очередь</Button>
                    </div>
                </div>}
            {!isScoringClassification
                ? <Draggable x={this.state.x} y={this.state.y} onMove={this.legendMove.bind(this)}>
                    <div className={styles.tracks_legend}
                         ref={this.tracksRef}>
                        <div className={styles.header}>
                            Треки:
                            <DragIndicator/>
                        </div>
                        <ReactTooltip id={COPY_TOOLTIP}
                                      effect="solid"
                                      html>
                        </ReactTooltip>
                        {this.state.traceIsLoading
                            ? <Spin/>
                            : Array.isArray(this.state.tracks)
                            && this.state.tracks?.map((track: any, index: number) => {
                                return (
                                    <div key={index}
                                         className={styles.track_item}
                                         onClick={this.getTrackInfo.bind(this, track.ride)}>
                                        <span>
                                            {index + 1}.&nbsp;
                                        </span>
                                        <span className={styles.status}>
                                            {track.status || EMPTY_DATA}
                                        </span>
                                        <Copy text={FormatDateInString({
                                            value: track.start_timestamp * ONE_SECOND,
                                            withSecond: true,
                                        })}
                                              externalTooltipId={COPY_TOOLTIP}>
                                            <FormatDate withSecond
                                                        value={track.start_timestamp * ONE_SECOND}/>
                                        </Copy>
                                        <span className={styles.length}>
                                            {track.summary && track.summary.length
                                                    && <Copy externalTooltipId={COPY_TOOLTIP}>
                                                        {track.summary.length.toFixed(FRACTION_DIGITS) + 'м'}
                                                    </Copy> || EMPTY_DATA}
                                        </span>
                                        <Copy text={FormatDateInString({
                                            value: track.finish_timestamp * ONE_SECOND,
                                            withSecond: true,
                                        })}
                                              externalTooltipId={COPY_TOOLTIP}>
                                            <FormatDate withSecond
                                                        value={track.finish_timestamp * ONE_SECOND}/>
                                        </Copy>
                                        <span className={styles.length}>
                                            {track.start_timestamp && track.finish_timestamp
                                                    &&
                                                    <Copy externalTooltipId={COPY_TOOLTIP}>
                                                        {durationBetween([
                                                            track.finish_timestamp,
                                                            track.start_timestamp,
                                                        ])}
                                                    </Copy>
                                                    || EMPTY_DATA}
                                        </span>
                                        <span className={styles.info}>
                                                Ссылки: {track.car && <Link href={`#/cars/${track.car}`}>авто</Link>}
                                            {track.user && <Link href={`#/clients/${track.user}`}>клиент</Link>}
                                            {track.session && <Link href={`#/session/${track.session}`}>поездка</Link>}
                                        </span>
                                        <div className={styles.controls}>
                                            <Link onClick={this.trace.bind(this, track.ride)}>Подробней</Link>&nbsp;
                                            {this.state.trace && this.state.trace_id === track.ride &&
                                                <Link onClick={this.removeTrace.bind(this, true)}>Очистить</Link>
                                            }
                                        </div>
                                    </div>
                                );
                            })}
                    </div>
                </Draggable>
                : null
            }
            {this.state.showLongTrackTooltip && this.state.preventLoadLongTrack && (
                <ReactTooltip className={styles.long_track_tooltip_outer}
                              id={LONG_TRACK_TOOLTIP_ID}
                              event={'no-event'}
                              place={'top'}
                              html
                              clickable>
                    {`<div class="${styles.long_track_tooltip_inner}">
                            <p>
                                Треки в этом запросе длятся дольше 5 дней, советуем уменьшить интервал просмотра "с/по" для комфортного использования
                            </p>
                            <div>
                                <span id="${LONG_TRACK_BUTTON_ID}">
                                    ${renderToStaticMarkup(
                    <Button className={styles.tooltip_button}
                            colorType={ButtonTypes.negative}>Все равно показать всё</Button>,
                )}
                                </span>
                            </div>
                        </div>`}
                </ReactTooltip>
            )}

            {!isScoringClassification
                ? <div className={`${styles.tabs}`}>
                    {this.state.expandConfiguration && (
                        <div className={`${styles.configuration} ${styles.content}`}
                             data-tip
                             ref={this.configRef}
                             data-for={LONG_TRACK_TOOLTIP_ID}>
                            <DatePicker timeFormat={'HH:mm:ss'}
                                        className={styles.input}
                                        value={this.state.since || ''}
                                        placeholder={'C'}
                                        onChange={this.onChange.bind(this, 'since')}/>
                            <DatePicker timeFormat={'HH:mm:ss'}
                                        className={styles.input}
                                        value={this.state.until || ''}
                                        placeholder={'По'}
                                        onChange={this.onChange.bind(this, 'until')}/>
                            <Input className={styles.input}
                                   placeholder={'ID машины (car_id)'}
                                   value={this.state.carId || ''}
                                   onChange={this.onChange.bind(this, 'carId')}/>
                            <Input className={styles.input}
                                   placeholder={'ID клиента (user_id)'}
                                   value={this.state.userId || ''}
                                   onChange={this.onChange.bind(this, 'userId')}/>
                            <Input className={styles.input}
                                   placeholder={'ID сессии (session_id)'}
                                   value={this.state.sessionId || ''}
                                   onChange={this.onChange.bind(this, 'sessionId')}/>
                            <Select className={styles.input}
                                    placeholder={'Тип сессии'}
                                    options={statuses}
                                    initialValues={this.state.status ? [this.state.status] : []}
                                    onSelect={this.onChange.bind(this, 'status')}/>
                            <Button className={styles.build_button}
                                    onClick={this.buildTrack.bind(this)}
                                    ytLog={{ button_name: buttonNameDetails.BUILD_TRACK }}>Построить трек</Button>
                        </div>
                    )}
                    {this.state.expandTraceInfo && (
                        <div className={styles.content}>
                            {this.state?.tableTimestamps.length
                                ? <>
                                    <div className={`${styles.tableHeader} ${styles.tableRow}`}>
                                        <span>#</span>
                                        <span>дата</span>
                                        <span>лимит (км/ч)</span>
                                        <span>скорость (км/ч)</span>
                                        <span>координаты</span>
                                        {this.state.session && <span/>}
                                    </div>
                                    <VirtualList width={'100%'}
                                                 height={300}
                                                 itemCount={this.state?.tableTimestamps?.length}
                                                 itemSize={35}
                                                 renderItem={({ index, style }) => {
                                                     const statusClassname = +this.state
                                                         .tableVdevice[index] < parkingSpeed
                                                         ? styles.parking
                                                         : (+this.state.tableVlimit[index] > -1
                                                             && +this.state.tableVdevice[index]
                                                    >= +this.state.tableVlimit[index] * speedToKM + overSpeedBuffer)
                                                             ? styles.limit
                                                             : '';
                                                     const selectedClassname = this.state.selected[index]
                                                         ? styles.selected
                                                         : '';
                                                     const item = this.state?.tableTimestamps?.[index];

                                                     return (
                                                         <div key={index}
                                                              style={style}
                                                              className={`${styles.tableRow} ${selectedClassname} ${statusClassname}`}
                                                              onClick={this.selectRow.bind(this, index)}>
                                                             <span>{index}</span>
                                                             <span>
                                                                 <FormatDate value={item * ONE_SECOND} withSecond/>
                                                             </span>
                                                             <span>{
                                                                 +this.state.tableVlimit[index] !== -1
                                                        && (+this.state.tableVlimit[index] * speedToKM).toFixed(0)
                                                        || EMPTY_DATA
                                                             }</span>
                                                             <span>{this.state.tableVdevice[index].toString()}</span>
                                                             <span>{this.state.tableCoord[index].join(', ')}</span>
                                                             {this.state.session && <span>
                                                                 <BillingCalculatorButton onClick={this.manageCalculator
                                                                     .bind(this, item * ONE_SECOND)}
                                                                                          since={this.state
                                                                                              .calculatorSince}
                                                                                          location={
                                                                                              buttonLocationDetails
                                                                                                  .TRACK_MAP}
                                                                                          BlockRules={this.props
                                                                                              .BlockRules}
                                                                                          session={this.state.session}
                                                                                          isLink>
                                                            возврат
                                                                 </BillingCalculatorButton>
                                                             </span>}
                                                         </div>
                                                     );
                                                 }}/>
                                </>
                                : <NoInformation/>
                            }
                        </div>
                    )}
                    <div className={`${styles.tab} ${styles.configuration} ${this.state.expandConfiguration
                        ? styles.tab_expanded
                        : ''}`}
                         onClick={this.expandTab.bind(this, 'expandConfiguration')}>
                        Настроить
                    </div>
                    <div className={`${styles.tab} ${styles.traceInfo} ${this.state.expandTraceInfo
                        ? styles.tab_expanded
                        : ''}`}
                         onClick={this.expandTab.bind(this, 'expandTraceInfo')}>
                        Trace info
                    </div>
                    {(this.state.carInfo && Object.keys(this.state.tableTsCoords).length)
                        ? <Sensors getInfo={this.getSensorsInformation.bind(this)}
                                   className={`${styles.sensors}`}
                                   carId={CarInfoHandler.getId.call(this.state.carInfo)}
                                   imei={CarInfoHandler.getImei.call(this.state.carInfo)}
                                   trace={this.state.tableTsCoords}
                                   removeTrace={this.removeTrace.bind(this)}
                                   removeSensors={this.removeSensors.bind(this)}
                                   drawTrace={this.drawTrace.bind(this, this.state.trace)}/>
                        : null
                    }
                </div>
                : null
            }

            {this.state.isDocQueueOpen
                ? <DocForQueuePicker filterFields={[POLYLINE_KEY]}
                                     initValues={this.getDocInitialData()}
                                     onClose={this.closeDocQueue.bind(this)}/>
                : null}
        </>;
    }
}

const TrackTooltip = (props: { index; item }) => {
    const { index, item } = props;
    const trackFirst = 2;

    return (
        <>
            <div>Трек: <strong>{index + 1}</strong></div>
            <div>Тип трека: <strong>{item.status || EMPTY_DATA}</strong></div>
            <div>Расстояние: <strong>
                {item.summary
                && item.summary.length
                && item.summary.length.toFixed(FRACTION_DIGITS)}м
            </strong>
            </div>
            <div>Время трека: <strong>{FormatDateInString({
                value: item.start_timestamp * ONE_SECOND,
                withSecond: true,
            })} - {
                FormatDateInString({ value: item.finish_timestamp * ONE_SECOND, withSecond: true })}</strong>
            </div>
            <div>Начало трека: <strong>{item.track.slice(0, trackFirst).reverse().toString()}</strong></div>
            <div>Конец трека: <strong>{item.track.slice(-trackFirst).reverse().toString()}</strong></div>
            <div><Link href={`#/cars/${item.car}`} target={'_blank'}>Автомобиль</Link></div>
            <div><Link href={`#/clients/${item.user}`} target={'_blank'}>Пользователь</Link></div>
        </>
    );
};

const TraceTooltip = (props: { vDevice; vLimit; timestamps; coordinates }) => {
    const { vDevice, vLimit, timestamps, coordinates } = props;

    return (
        <>
            <div>Скорость: <strong>{+vDevice}км/ч</strong></div>
            <div>Ограничение: <strong>
                {+vLimit > -1 && (+vLimit * speedToKM).toFixed(FRACTION_DIGITS) + 'км/ч' || EMPTY_DATA}
            </strong>
            </div>
            <div>Время: <strong>{new Date(+timestamps * ONE_SECOND).toLocaleString()}</strong></div>
            <div>Сырое время: <strong>{+timestamps}</strong></div>
            <div>
                <Link href={`https://n.maps.yandex.ru/#!/?z=18&ll=${coordinates || ''}&l=nk#sat`} target="_blank">
                    Народная карта
                </Link>
            </div>
        </>
    );
};

const SensorTooltip = (props: { value; name; id; subid; timestamp }) => {
    const { value, name, id, subid, timestamp } = props;

    return (
        <>
            <div>Название: <strong>{name}</strong></div>
            <div>Значение: <strong>{value}</strong></div>
            <div>Id/subId (если есть): <strong>{id}{subid ? `/${subid}` : ''}</strong>
            </div>
            <div>Время: <strong>{new Date(+timestamp * ONE_SECOND).toLocaleString()}</strong></div>
            <div>Сырое время: <strong>{+timestamp}</strong></div>
        </>
    );
};

const ScoringTraceLabel = (props) => {
    const events = props.traceTags?.events || [];

    return events.length && <div className={styles.scoring_trace_label}>
        <h4>Резкостей: {events.length}</h4>
        {
            events.map((el, index) => {
                const { location = [] } = el || {};

                return <div key={index}
                            className={styles.scoring_trace_label_item}
                            onClick={props.onClick.bind(null, location)}>
                    <div>{index + 1} ) <FormatDate value={el.timestamp * ONE_SECOND} withSecond/></div>
                    <Address latitude={location[1]} longitude={location[0]}/>
                </div>;
            })
        }
    </div> || null;
};
