import { Dict, IPoly } from '../../../../types';
import { Request2 } from '../../../utils/request';
import { drawPoly } from '../../MiniMap/utils';
import { REQUESTS, TRACKS_REQUESTS } from '../request';

declare let ymaps;

const filterPolygons = (polyFilters: IPolyFilter[], polygons: IPoly[], polyTags: string[]): IPoly[] => {
    const resultPoly = new Set<IPoly>();
    polyTags
        .forEach((polyTag) => {
            const { tags = [], filter } = polyFilters.find((polyFilter) => polyFilter.key === polyTag) ?? {};

            polygons.forEach((poly: IPoly) => {
                if (poly.area_tags) {
                    //empty tag array -> match every polygon
                    if (!tags.length) {
                        resultPoly.add(poly);
                    }

                    poly.area_tags?.split(',').forEach(tag => {
                        const tag_trimmed = tag.trim();
                        if (tags.includes(tag_trimmed)) {
                            resultPoly.add(poly);
                        }

                        filter && filter.forEach(str => {
                            if (tag_trimmed.includes(str)) {
                                resultPoly.add(poly);
                            }
                        });
                    });
                }
            });
        });

    return Array.from(resultPoly);
};

const clickHandler = (e) => {
    const map = e.get('map');
    const target = e.get('target');
    const position = e.get('coords');

    const polygons = target.getParent();
    const placemark = new ymaps.Placemark(position);
    const geoQuery = ymaps.geoQuery(polygons);

    const result = geoQuery.searchContaining(placemark);
    const polyListHtml: string[] = [];
    result.each(el => {
        polyListHtml.push(`<div><span>${polyListHtml.length + 1} </span><strong>${el.options.get('area_title') || el.options.get('area_id')}</strong></div>`);
    });
    map.hint.open(position, `<div>${polyListHtml.join('')}<div>`);
};

const selectPolygons = (props: { polygonsCollection; errorHandler; polyControlPair; polygonsLayer; polyFilters }) => {
    const { polygonsCollection, errorHandler, polyControlPair, polygonsLayer, polyFilters } = props;
    const request = new Request2({ requestConfigs: TRACKS_REQUESTS });

    polyControlPair && polyControlPair.deselect();
    const allCurrentPolyTags: string[] = [];
    const iterator = polygonsLayer.getIterator();
    let obj;
    while ((obj = iterator.getNext()) !== iterator.STOP_ITERATION) {
        obj.disable();
        if (obj.state.get('selected')) {
            allCurrentPolyTags.push(obj.state.get('keyPoly'));
        }
    }

    polygonsCollection.events.remove('click', clickHandler);
    if (allCurrentPolyTags.length) {
        request.exec(REQUESTS.GET_POLYGONS)
            .then((response) => {
                const filteredPolygons = filterPolygons(polyFilters, response?.areas, allCurrentPolyTags);
                polygonsCollection.removeAll();

                drawPoly({
                    polygons: filteredPolygons,
                    polygonsCollection: polygonsCollection,
                });
                polygonsCollection.events.add('click', clickHandler);

                const iterator = polygonsLayer.getIterator();
                while ((obj = iterator.getNext()) != iterator.STOP_ITERATION) {
                    obj.enable();
                }

                if (errorHandler) {
                    errorHandler(null);
                }
            })
            .catch((error) => {
                if (errorHandler) {
                    errorHandler(error);
                }

                const iterator = polygonsLayer.getIterator();
                while ((obj = iterator.getNext()) != iterator.STOP_ITERATION) {
                    obj.enable();
                    obj.deselect();
                }
            });
    } else {
        polygonsCollection.events.remove('click', clickHandler);
        polygonsCollection.removeAll();

        const iterator = polygonsLayer.getIterator();
        while ((obj = iterator.getNext()) != iterator.STOP_ITERATION) {
            obj.enable();
        }
    }
};

export enum ControlFloat {
    LEFT = 'left',
    RIGHT = 'right'
}

interface IPolyFilter {
    key: string;
    name: string;
    filter?: string[];
    tags?: string[];
    blockrules?: string[];
    roles?: string[];
}

export const drawPolygonControl = (props: {
    polyFilters: IPolyFilter[];
    BlockRules?: Dict<any>;
    map: any;
    polygonsCollection: any;
    errorHandler?: (error) => void;
    polyControlPair?: any;
    float?: ControlFloat;
    polyFiltersRoles?: string[];
}) => {
    const {
        BlockRules,
        map,
        polygonsCollection,
        errorHandler,
        polyControlPair,
        float = ControlFloat.RIGHT,
        polyFiltersRoles,
        polyFilters,
    } = props;

    if (polyFilters && BlockRules) {
        const polyList = Array.isArray(polyFilters)
            && polyFilters
                .filter((polyFilter) => {
                    if (polyFilter?.blockrules?.length) {
                        return polyFilter.blockrules.some((rule) => BlockRules?.[rule] ?? true);
                    }

                    return true;

                })
                .map(polyFilter => {
                    const selected = polyFiltersRoles?.some((role) => polyFilter?.roles?.includes(role)) ?? false;

                    return new ymaps.control.ListBoxItem({
                        data: {
                            content: polyFilter.name,
                        },
                        state: {
                            selected,
                            keyPoly: polyFilter.key,
                        },
                    });
                });

        const polygonsLayer = new ymaps.control.ListBox({
            data: {
                content: 'Полигоны',
            },
            items: polyList,
        });

        polygonsLayer.events
            .add(['select', 'deselect'], () => {
                selectPolygons({
                    polygonsCollection,
                    errorHandler,
                    polyControlPair,
                    polygonsLayer,
                    polyFilters,
                });
            });

        map.controls.add(polygonsLayer, { float });

        selectPolygons({
            polygonsCollection,
            errorHandler,
            polyControlPair,
            polygonsLayer,
            polyFilters,
        });

        return polygonsLayer;
    }
};
