import * as React from 'react';

import { Button } from '../../ui/Button';
import Checkbox from '../../ui/Checkbox';
import { Confirm, Window } from '../../ui/FullModal';
import { Input } from '../../ui/Input';
import { Link } from '../../ui/Link';
import { LabelStatus, TLabel } from '../../ui/Table';
import * as tblStyle from '../../ui/Table/index.css';
import TextArea from '../../ui/TextArea';
import { Request2 } from '../../utils/request';
import { SETTINGS_KEY_UI } from '../InterfaceAdminConfig/adminConfigKeys';
import { GLOBAL_VARS_REQUESTS as requestConfigs, REQUESTS } from '../Settings/GlobalVarsView/request';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import { IAdminUser } from './index';
import * as style from './index.css';

type IProps = IAdminUser

interface ISettingsState {
    showModal: boolean;
    showConfirm: boolean;
    selectedId: any;
    uiSettings: IRule[];
    error: any;
    removingError: any;
    dataIsLoading: boolean;
    isWorking: boolean;
}

class UISettings extends React.Component<IProps, ISettingsState> {
    state: ISettingsState = {
        showModal: false,
        showConfirm: false,
        selectedId: null,
        uiSettings: [],
        error: null,
        removingError: null,
        dataIsLoading: false,
        isWorking: false,
    };

    request = new Request2({
        requestConfigs,
    });

    showModal(state: boolean) {
        this.setState({
            showModal: state,
            selectedId: !state ? null : this.state.selectedId,
        });
    }

    getData() {
        this.request.exec(REQUESTS.GET_GLOBAL_SETTINGS)
            .then(response => {
                let error = null;
                let uiSettings = response.settings
                    && Array.isArray(response.settings)
                    && response.settings.find(el => el.setting_key === SETTINGS_KEY_UI);
                try {
                    uiSettings = uiSettings && uiSettings.setting_value
                        && JSON.parse(uiSettings.setting_value);
                    if (!uiSettings) {
                        throw new Error(`Проблемы с настройками UI или они отсутствуют, ${JSON.stringify(uiSettings)}`);
                    }
                } catch (e) {
                    error = e;
                    uiSettings = [];
                }

                this.setState({
                    uiSettings: uiSettings.sort((a: IRule, b: IRule) => b.id && a.id && (+b.id - +a.id)),
                    error,
                    dataIsLoading: false,
                });
            })
            .catch((e) => {
                this.setState({
                    error: e,
                    dataIsLoading: false,
                });
            });
    }

    componentDidMount(): void {
        this.setState({
            dataIsLoading: true,
        }, () => {
            this.getData();
        });
    }

    confirmDeleteRule(selectedId: any) {
        this.setState({
            selectedId,
            showConfirm: true,
        });
    }

    deleteRule() {
        this.setState({
            isWorking: true,
        }, () => {
            let _data: IRule[] = [] as IRule[];

            if (this.state.uiSettings && Array.isArray(this.state.uiSettings)) {
                _data = this.state.uiSettings.filter((el: IRule) => el.id !== this.state.selectedId);
            }

            const postData: IPostSettingItem = {
                settings: [{
                    setting_key: SETTINGS_KEY_UI,
                    setting_value: JSON.stringify(_data),
                }],
            };

            this.request.exec<IPostSettingItem>(REQUESTS.UPSERT_SETTINGS, { body: postData })
                .then(() => {
                    this.setState({
                        selectedId: null,
                        isWorking: false,
                        removingError: null,
                        showConfirm: false,
                    }, () => {
                        this.getData();
                    });
                })
                .catch(removingError => {
                    this.setState({
                        isWorking: false,
                        removingError,
                    });
                });
        });
    }

    editRule(id: any) {
        this.setState({
            selectedId: id,
            showModal: true,
        });
    }

    showConfirm(status: boolean) {
        this.setState({
            showConfirm: status,
        });
    }

    render() {
        const currentRule = this.state.selectedId && this.state.uiSettings
            .find(el => el.id === this.state.selectedId);

        return <div className={style.component}>
            <h4>Настройки интерфейса</h4>
            <Button onClick={this.showModal.bind(this, true)}>Добавить правило</Button>
            <div className={style.table__before_margin}>
                {
                    this.state.dataIsLoading
                        ? <Spin/>
                        : this.state.error
                            ? <SimpleError error={this.state.error} data={{ label: 'UI' }}/>
                            : <div>{this.state.uiSettings
                        && Array.isArray(this.state.uiSettings)
                        && <table className={`${tblStyle.table} ${style.tbl}`}>
                            <thead>
                                <tr className={'header'}>
                                    <th>#</th>
                                    <th>статус</th>
                                    <th>доступно мне</th>
                                    <th>название</th>
                                    <th>блок</th>
                                    <th className={style.roles}>роли</th>
                                    <th/>
                                </tr>
                            </thead>
                            <tbody>
                                {
                                    this.state.uiSettings.map((el, index) => {
                                        const myRole = this.props.AdminUser
                                        && this.props.AdminUser.rules
                                        && this.props.AdminUser.rules[el.ui];

                                        return <tr key={index}>
                                            <td>{++index}</td>
                                            <td className={style.cellLabel}>
                                                <TLabel status={el.active ? LabelStatus.POSITIVE : LabelStatus.NEGATIVE}
                                                        text={el.active ? 'active' : 'not active'}/></td>
                                            <td>{
                                                myRole !== undefined
                                            && <TLabel status={LabelStatus.INFO} text={'available'}/>
                                            }</td>
                                            <td><Link onClick={this.editRule.bind(this, el.id)}>
                                                {el.ruleName}
                                            </Link></td>
                                            <td>{el.ui}</td>
                                            <td className={style.roles}>{el.roles}</td>
                                            <td><Link onClick={this.confirmDeleteRule.bind(this, el.id)}>
                                                удалить
                                            </Link></td>
                                        </tr>;
                                    })}
                            </tbody>
                        </table>
                            }</div>
                }
            </div>
            {
                this.state.showModal
                && <AddUIRule id={this.state.selectedId}
                              request={this.request}
                              updateData={this.getData.bind(this)}
                              data={this.state.uiSettings}
                              onClose={this.showModal.bind(this, false)}/>
            }
            {
                this.state.showConfirm
                && <Confirm question={`Удалить "${currentRule && currentRule.ruleName}"?`}
                            accept={this.deleteRule.bind(this)}
                            isWorking={this.state.isWorking}
                            error={this.state.removingError}
                            onClose={this.showConfirm.bind(this, false)}/>
            }
        </div>;
    }
}

interface IAddUIRule {
    id: number | null;
    request: Request2;
    data: any;
    onClose: () => void;
    updateData: () => void;
}

interface IRule {
    ruleName: string;
    ui: string;
    roles: string;
    active: boolean;
    id: number | null;

    [key: string]: any;
}

interface IState {
    item: IRule;
    isLoading: boolean;
    err: any;
}

interface ISettingItem {
    setting_key: string;
    setting_value: string;
}

interface IPostSettingItem {
    settings: ISettingItem[];
}

class AddUIRule extends React.Component<IAddUIRule, IState> {
    state: IState = {
        item: {
            ruleName: '',
            ui: '',
            roles: '',
            active: false,
            id: this.props.id || null,
        },
        isLoading: false,
        err: null,
    };

    onChange(param, val) {
        const { item } = this.state;
        item[param] = val;
        this.setState({ item });
    }

    componentDidMount(): void {
        const selectedData: IRule = this.props.data
            && this.props.id
            && Array.isArray(this.props.data)
            && this.props.data.find(el => el.id === this.props.id);
        selectedData && this.setState({ item: selectedData });
    }

    addRole() {
        let _data: IRule[] = [] as IRule[];
        if (this.props.data
            && this.props.id
            && this.state.item.id
            && Array.isArray(this.props.data)) {
            _data = this.props.data
                .map((el: IRule) => el.id === this.state.item.id ? ({ ...this.state.item }) : el);
        } else if (!this.state.item.id && this.props.data && Array.isArray(this.props.data)) {
            _data = [...this.props.data,
                Object.assign({}, this.state.item, { id: new Date().getTime() })];
        }

        const postData: IPostSettingItem = {
            settings: [{
                setting_key: SETTINGS_KEY_UI,
                setting_value: JSON.stringify(_data),
            }],
        };

        this.setState({
            isLoading: true,
        }, () => {
            this.props.request.exec<IPostSettingItem>(REQUESTS.UPSERT_SETTINGS, { body: postData })
                .then(() => {
                    this.setState({
                        isLoading: false,
                        err: null,
                    });
                    this.props.updateData();
                    this.props.onClose();
                })
                .catch(err => {
                    this.setState({
                        isLoading: false,
                        err,
                    });
                });
        });

    }

    render() {
        return <Window title={'Правило UI'}
                       onClose={this.props.onClose.bind(this)}
                       className={style['add-rule-window']}
                       error={this.state.err}>
            <div className={style['add-rule']}>
                <Input value={this.state.item.ruleName}
                       disabled={this.state.isLoading}
                       placeholder={'название правила'}
                       onChange={this.onChange.bind(this, 'ruleName')}/>

                <Input value={this.state.item.ui}
                       disabled={this.state.isLoading}
                       placeholder={'блок UI / роут'}
                       onChange={this.onChange.bind(this, 'ui')}/>

                <TextArea placeholder={'Список ролей'}
                          value={this.state.item.roles}
                          disabled={this.state.isLoading}
                          onChange={this.onChange.bind(this, 'roles')}/>
                <div className={style.controls}>
                    <div className={style.controls__checkbox}>
                        <div>
                            <Checkbox checked={this.state.item.active} onChange={this.onChange.bind(this, 'active')}/>
                        </div>
                        <div>{this.state.item.active ? 'ВКЛ' : 'ВЫКЛ'}</div>
                    </div>
                    <Button disabled={this.state.isLoading}
                            className={style.button___right}
                            onClick={this.addRole.bind(this)}>
                        {!this.state.item.id ? 'Добавить' : 'Обновить'}
                    </Button>
                </div>
            </div>
        </Window>;
    }
}

export {
    UISettings,

    IRule,
    ISettingItem,
    IPostSettingItem,
};
