import React, { useState } from 'react';

import { ENV_GROUPS, ENVIRONMENT, EXTRA_ENV, LANGUAGES, LSSettingItems, USER_ACTION } from '../../../types';
import { DUTY_STATE } from '../../constants';
import { LocalizationReducerState } from '../../reducers/localizationReducer';
import { Button, ButtonTypes } from '../../ui/Button';
import { Window } from '../../ui/FullModal';
import { Link } from '../../ui/Link';
import Select, { IOptionInfo } from '../../ui/Select';
import { getCookie } from '../../utils/cookie';
import { isObjectEqual } from '../../utils/isObjectEqual';
import LS from '../../utils/localStorage/localStorage';
import { Translate } from '../../utils/translate';
import { IStore } from '../App/store';
import { isFeedState } from '../ChatNotify/component';
import { Copy } from '../Copy';
import { ToBreakConfirm } from '../Duty/ToBreakConfirm';
import { AbstractMenuSideBar, IPropsUserMenu } from '../Menu';
import * as style from '../Menu/index.css';
import { UNKNOWN_USER } from '../User/unknownUser';
import { IDispatchToProps } from './index';
import RolesForQueue from './RolesForQueue';

interface IProps extends IPropsUserMenu, IDispatchToProps, IStore {
    status_tags: any[];
}

export const LOGIN_URL = `https://passport.yandex-team.ru/auth?origin=maps&retpath=${location.href}`;

export class UserMenu extends React.Component<IProps> {
    state = {
        userInfoModalOpen: false,
    };

    shouldComponentUpdate(nextProps: Readonly<IProps>): boolean {
        return !isObjectEqual(this.props, nextProps);
    }

    openUserInfoModal() {
        this.props.toggleSideBar(false);
        this.props.openUserInfoModal?.();
    }

    render() {
        const { Lang = {} as LocalizationReducerState } = this.props;
        const t = new Translate(Lang);

        return <AbstractMenuSideBar {...this.props} menuId={'menu-user'}>
            <UserInfo userId={this.props.AdminUser?.userId}
                      yandex_login={this.props.AdminUser?.yandex_login}
                      t={t}
                      toggleSideBar={this.props.toggleSideBar}/>

            <Button basic={true}
                    className={style.user_info_btn}
                    onClick={this.openUserInfoModal.bind(this)}>Пользовательские настройки</Button>

            <ChangeEnv t={t}/>
            {
                isFeedState.call(this, DUTY_STATE.COMMON)
                    ? <DutyBreak status_tags={this.props?.status_tags}/>
                    : null
            }
            <ChatsQueue/>
            <ChangeLang setCurrentLang={this.props.setCurrentLang}
                        t={t}
                        toggleSideBar={this.props.toggleSideBar}/>
        </AbstractMenuSideBar>;
    }
}

const DutyBreak = React.memo((props: { status_tags: any }) => {
    const [popupIsOpen, switchPopup] = useState(false);

    return <div className={style.chats_queue}>
        <Button onClick={switchPopup.bind(null, !popupIsOpen)}
                colorType={ButtonTypes.warning}>Перерыв ☕ / Закончить смену</Button>
        {
            popupIsOpen && <ToBreakConfirm onClose={switchPopup.bind(null, !popupIsOpen)} tags={props?.status_tags}/>
        }
    </div>;
});

const ChatsQueue = React.memo(() => {
    const [popupIsOpen, switchPopup] = useState(false);
    const [error, setError] = useState(null);

    const onError = (err) => {
        setError(err);
    };

    return <div className={style.chats_queue}>
        <Button onClick={switchPopup.bind(null, !popupIsOpen)}>Очереди чатов</Button>
        {
            popupIsOpen && <Window onClose={switchPopup.bind(null, false)}
                                   className={style.chat_popup}
                                   error={error}
                                   title={'Очереди чата'}>
                <RolesForQueue onError={onError}/>
            </Window>
        }
    </div>;
});

interface IUserInfoProps {
    t: any;
    yandex_login: string | undefined;
    userId: string | undefined;
    toggleSideBar: (state?: boolean) => void;
}

class UserInfo extends React.Component<IUserInfoProps, any> {
    ls = new LS();

    userAction(type: USER_ACTION, e: KeyboardEvent) {
        e.preventDefault();
        e.stopPropagation();
        this[type]();
    }

    feedback(e: KeyboardEvent) {
        e.preventDefault();
        e.stopPropagation();

        const env = this.ls.get(LSSettingItems.env);
        const loc = encodeURIComponent(location.toString());
        const version = process.env.VERSION;
        const url = `https://forms.yandex-team.ru/surveys/27289/?adm_link=${loc}&adm_build=${version}&adm_type=${env}`;
        window.open(url);
    }

    exit() {
        const exit = confirm(this.props.t.getSimpleText('admin_settings_confirm_exit_text'));
        const logoutUrl = `https://passport.yandex-team.ru/?mode=logout&yu=${getCookie('yandexuid')}&retpath=${location.href}`;
        exit && (location.href = logoutUrl);
    }

    enter() {
        location.href = LOGIN_URL;
    }

    render() {
        const { t } = this.props;
        const versionLink = `https://st.yandex-team.ru/issues/`
            + `?_q=Text:+${process.env.VERSION}+AND+Queue:+DRIVEFRONT+AND+Status:+Закрыт`;

        return <div className={style.short_user_info}>
            <div>
                <span>{this.props.yandex_login || UNKNOWN_USER}, </span>
                <span>
                    <Link href={versionLink} target={'_blank'}>{process.env.VERSION}</Link>
                </span>
            </div>
            <div className={style.user_id}>
                <Copy>{this.props.userId}</Copy>
            </div>
            {this.props.userId
                ? <div>
                    <Link href={`#/clients/${this.props.userId}/user-access`}
                          onClick={this.props.toggleSideBar}
                          disablePreventDefault>
                        Профиль
                    </Link>
                        /
                    <Link href={`#/clients/${this.props.userId}/request_aggregator`}
                          onClick={this.props.toggleSideBar}
                          disablePreventDefault>
                    Мои обращения
                    </Link>
                </div>
                : null}
            <div className={style.user_action}>
                {this.props.userId
                    ? <div onClick={this.userAction.bind(this, USER_ACTION.EXIT)}>
                        {t.getItem('admin_settings_exit')}
                    </div>
                    : <div onClick={this.userAction.bind(this, USER_ACTION.ENTER)}>
                        {t.getItem('admin_settings_enter')}
                    </div>
                }
            </div>
            <br/>
            <br/>
            <div className={style.user_action}>
                <div onClick={this.feedback.bind(this)}>
                    🐞️ {t.getItem('admin_settings_feedback')}
                </div>
            </div>
        </div>;
    }
}

interface IChangeLangState {
    currentLang: LANGUAGES;
}

interface IChangeLangProps extends IDispatchToProps {
    t: Translate;
    toggleSideBar: (state?: boolean) => void;
}

class ChangeLang extends React.Component<IChangeLangProps, IChangeLangState> {
    ls = new LS();
    state: IChangeLangState = {
        currentLang: this.ls.get(LSSettingItems.lang),
    };

    langs: IOptionInfo[];

    constructor(props: IChangeLangProps) {
        super(props);

        this.langs = [
            ...Object.entries(LANGUAGES).map(ln => ({ text: ln[0], value: ln[1] })),
        ];
    }

    changeLang(currentLang) {
        //don't change lang if it empty
        if (!currentLang || !(currentLang && currentLang.toString())) {
            return;
        }

        this.setState({
            currentLang,
        }, () => {
            const curLang = currentLang.toString();
            curLang && this.ls.set(LSSettingItems.lang, curLang);
            //dispatch to store new cur lang
            this.props.setCurrentLang(curLang);
        });
        this.props.toggleSideBar();
    }

    render() {
        return <div className={style.lang_control}>
            <h4>{this.props.t.getItem('admin_settings_language')}:</h4>
            <Select multiSelect={false}
                    withoutAutoFocus
                    initialValues={[this.state.currentLang]}
                    options={this.langs}
                    placeholder={this.props.t.getSimpleText('admin_settings_language_gui')}
                    onSelect={this.changeLang.bind(this)}/>
        </div>;
    }
}

export const syncEnv2 = (ls) => {
    const env = ls.get(LSSettingItems.env);
    const extraEnv = EXTRA_ENV[env.toUpperCase()];
    localStorage.setItem('_da_service', extraEnv?.env || env);
    localStorage.setItem('_da_custom_service', env === ENVIRONMENT.CUSTOM ? '1' : '0');
    localStorage.setItem('saas_proxy', ls.get(LSSettingItems.custom_env));
};

export class ChangeEnv extends React.Component<any, any> {
    state = {
        env: ENVIRONMENT.PRESTABLE,
        collapsed: {},
    };
    ls = new LS();

    componentDidMount(): void {
        this.setState({
            env: this.ls.get(LSSettingItems.env),
        });
    }

    change(env: ENVIRONMENT) {
        this.ls.set(LSSettingItems.env, env);
        this.setState({
            env,
        }, () => {
            syncEnv2(this.ls);
            location.reload();
        });
    }

    collapse(value) {
        this.setState((prev) => {
            return {
                collapsed: { ...prev.collapsed, [value]: !prev.collapsed[value] },
            };
        });
    }

    render() {
        const label = this.props?.t?.getItem('admin_settings_environment');

        return <div className={`${style.wrapper} ${this.props.className || ''}`}>
            {
                label && <h4>{label}:</h4>
            }

            {
                ENV_GROUPS.map(el => {
                    const selected = <span>{el.envs?.includes(this.state.env)
                        ? <strong>: {this.state.env}</strong>
                        : ''}</span>;

                    return <React.Fragment key={el.name}>
                        <div className={`${style.env} ${style.group}`}
                             onClick={this.collapse.bind(this, el.name)}>{el.name}{selected}</div>
                        {
                            el.envs.map((_el, index) => {
                                const isActive = _el === this.state.env ? style.active : '';
                                const collapsed = !this.state.collapsed[el.name] ? style.collapsed : '';

                                return <div key={index}
                                            className={`${style.env} ${isActive} ${collapsed}`}
                                            onClick={this.change.bind(this, _el)}>{_el}</div>;
                            })
                        }
                    </React.Fragment>;
                })
            }
        </div>;
    }
}
