import * as React from 'react';

import { Dict } from '../../../../types';
import { EMPTY_DATA } from '../../../constants';
import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { Window } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { WARNING_LABEL } from '../../../ui/Select';
import { Switcher } from '../../../ui/Switcher';
import * as styleTable from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import { IJSONData } from '../../ExcelUploader';
import { ProgressBar, RetryProgressButtons } from '../../ProgressBar';
import QueryScheduler, { IProgressData } from '../../QueryScheduler/QueryScheduler';
import * as style from '../index.css';
import { BATCH_UPLOAD_CARS_REQUESTS as requestConfigs, REQUESTS } from '../request';
import { INSURANCE_TYPES } from './types';

const PACKAGE_SIZE = 50;

interface IUploadingInsuranceModalProps {
    fileData: IJSONData[];
    onClose: () => void;
    fileError: Error | null;
}

interface IUploadingInsuranceModalSate {
    isBatchUploading: boolean;
    uploadingProgress: {
        queue: (() => {})[];
        success: string[];
        errors: { data: number; error: Error }[];
    };
    currentInsuranceType: string;
    baseCost: number;
    activeLists: { [key: string]: boolean };
}

export class UploadingInsuranceModal extends
    React.Component<IUploadingInsuranceModalProps, IUploadingInsuranceModalSate> {
    state: IUploadingInsuranceModalSate = {
        isBatchUploading: false,
        uploadingProgress: {
            queue: [],
            success: [],
            errors: [],
        },
        currentInsuranceType: INSURANCE_TYPES.RENINS.value,
        baseCost: 35,
        activeLists: {},
    };

    request = new Request2({ requestConfigs });

    componentDidMount(): void {
        const fileData = this.props.fileData ?? [];
        const sheetNames = fileData.map(sheet => sheet.sheetName);
        const activeLists = sheetNames.reduce((result: { [key: string]: boolean }, sheetName: string) => {
            result[sheetName] = true;

            return result;
        }, {});

        this.setState({ activeLists });
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    onProgress(data: IProgressData) {
        const { failed, queue, success } = data;

        const uploadingProgress = {
            queue,
            success,
            errors: failed,
        };

        this.setState({ uploadingProgress });
    }

    onSuccess(data: IProgressData) {
        const { failed, queue, success } = data;

        const uploadingProgress = {
            queue,
            success,
            errors: failed,
        };

        this.setState({ uploadingProgress, isBatchUploading: false });
    }

    changeCurrentInsuranceType(currentInsuranceType: string) {
        this.setState({ currentInsuranceType });
    }

    changeActiveList(sheetName: string) {
        const activeLists = { ...this.state.activeLists };
        activeLists[sheetName] = !activeLists[sheetName];
        this.setState({ activeLists });
    }

    onBaseCostChange(baseCost: number) {
        this.setState({ baseCost });
    }

    buildFileConfirmQuestion() {
        const fileData = this.props.fileData ?? [];
        const { currentInsuranceType, baseCost, activeLists } = this.state;

        return <>
            <h4 className={style.insurance_type_title}>Тип полисов:</h4>
            <Switcher items={Object.values(INSURANCE_TYPES)}
                      active={currentInsuranceType}
                      onChange={this.changeCurrentInsuranceType.bind(this)}/>
            <Input value={baseCost}
                   onChange={this.onBaseCostChange.bind(this)}
                   placeholder={'Размер выплаты по страхованию жизни (копейки)'}
                   description={'Размер выплаты по страхованию жизни добавится ко всем полисам'}/>
            {fileData?.length
                ? <table className={`${styleTable.table} ${style.uploaded_file_preview}`}>
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Название листа</th>
                            <th>Количество записей</th>
                            <th/>
                        </tr>
                    </thead>
                    <tbody>
                        {fileData.map((sheet, index) => {
                            return <tr key={sheet.sheetName}>
                                <td>{index + 1}</td>
                                <td>{sheet?.sheetName ?? EMPTY_DATA}</td>
                                <td>{sheet?.data?.length ?? EMPTY_DATA}</td>
                                <td>
                                    <Checkbox checked={activeLists?.[sheet.sheetName]}
                                              onChange={this.changeActiveList.bind(this, sheet.sheetName)}/>
                                </td>
                            </tr>;
                        })}
                    </tbody>
                </table>
                : <h4>{WARNING_LABEL} В файле не найдено записей</h4>}
        </>;
    }

    uploadFileData() {
        this.setState({ isBatchUploading: true }, () => {
            const fileData = this.props.fileData ?? [];
            const { currentInsuranceType, baseCost, activeLists } = this.state;
            const cars = fileData?.filter(sheet => {
                return activeLists[sheet.sheetName];
            }).reduce((res: Dict<string>[], curr) => {
                res.push(...curr.data);

                return res;
            }, []);

            const batchedQueue: Dict<string>[][] = [];
            for (let index = 0; index < cars.length;) {
                const carsCount = cars.length - 1 - index < PACKAGE_SIZE ? cars.length - index : PACKAGE_SIZE;

                batchedQueue.push(cars.slice(index, index + carsCount));
                index += carsCount;
            }

            const queue = batchedQueue.map(carsArr => this.request.exec.bind(this.request,
                REQUESTS.BATCH_UPLOAD_CARS_INSURANCE, {
                    body: {
                        cars: carsArr,
                        type: currentInsuranceType,
                        base_cost: baseCost,
                    },
                }));

            const qs = new QueryScheduler({
                queue,
                limit: PACKAGE_SIZE,
                onProgress: this.onProgress.bind(this),
                onSuccess: this.onSuccess.bind(this),
            });
            qs.run();
        });
    }

    getFileData() {
        const fileData = this.props.fileData ?? [];
        const cars = fileData?.reduce((res: Dict<string>[], curr) => {
            res.push(...curr.data);

            return res;
        }, []);

        const batchedQueue: Dict<string>[][] = [];
        for (let index = 0; index < cars.length;) {
            const carsCount = cars.length - 1 - index < PACKAGE_SIZE ? cars.length - index : PACKAGE_SIZE;

            batchedQueue.push(cars.slice(index, index + carsCount));
            index += carsCount;
        }

        return batchedQueue;
    }

    render() {
        const { onClose, fileData = [], fileError } = this.props;
        const { uploadingProgress, isBatchUploading } = this.state;

        const isFileDataEmpty = !fileData.length;

        return <Window onClose={onClose}
                       title={'Загрузить данные?'}
                       error={fileError}
                       closeWithConfirm={isBatchUploading}>
            <div className={style.insurance_upload_modal}>
                {this.buildFileConfirmQuestion()}
                {!isFileDataEmpty
                    ? <div className={style.progress_bar_container}>
                        <ProgressBar allLength={uploadingProgress?.queue?.length}
                                     successLength={uploadingProgress?.success?.length}
                                     errors={uploadingProgress?.errors ?? []}/>
                    </div>
                    : null}
                <Button disabled={isBatchUploading || isFileDataEmpty}
                        onClick={this.uploadFileData.bind(this)}>Загрузить</Button>
                {uploadingProgress?.errors?.length
                    ? <RetryProgressButtons uploadingProgress={uploadingProgress}
                                            fileData={this.getFileData()}
                                            multiplyBatches={true}
                                            isWorking={isBatchUploading}
                                            onProgress={this.onProgress.bind(this)}
                                            onSuccess={this.onSuccess.bind(this)}/>
                    : null}
            </div>
        </Window>;
    }
}
