import * as React from 'react';

import { Dict } from '../../../../types';
import { EMPTY_DATA } from '../../../constants';
import { Button } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import { WARNING_LABEL } from '../../../ui/Select';
import * as styleTable from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import { IJSONData } from '../../ExcelUploader';
import { ProgressBar, RetryProgressButtons } from '../../ProgressBar';
import QueryScheduler, { COUNT_FIRST, IProgressData } from '../../QueryScheduler/QueryScheduler';
import * as style from '../index.css';
import { BATCH_UPLOAD_CARS_REQUESTS as requestConfigs, REQUESTS } from '../request';

interface IUploadingVinModalProps {
    onClose: () => void;
    fileData: IJSONData[];
}

interface IUploadingVinModalState {
    isBatchUploading: boolean;
    uploadingProgress: {
        queue: Request2[];
        success: any[];
        errors: { data: number; error: Error }[];
    };
}

export class UploadingVinModal extends React.Component<IUploadingVinModalProps, IUploadingVinModalState> {
    state: IUploadingVinModalState = {
        isBatchUploading: false,
        uploadingProgress: {
            queue: [],
            success: [],
            errors: [],
        },
    };
    request = new Request2({ requestConfigs });

    componentWillUnmount(): void {
        this.request.abort();
    }

    buildFileConfirmQuestion() {
        const fileData = this.props.fileData ?? [];

        return fileData?.length
            ? <>
                <table className={styleTable.table}>
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Название листа</th>
                            <th>Количество записей</th>
                        </tr>
                    </thead>
                    <tbody>
                        {fileData.map((sheet, index) => {
                            const { sheetName = '', data } = sheet;

                            return <tr key={sheetName}>
                                <td>{index + 1}</td>
                                <td>{sheetName ?? EMPTY_DATA}</td>
                                <td>{data?.length ?? EMPTY_DATA}</td>
                            </tr>;
                        })}
                    </tbody>
                </table>
            </>
            : <h4>{WARNING_LABEL} В файле не найдено записей</h4>;
    }

    onProgress(data: IProgressData) {
        const { failed, queue, success } = data;

        const uploadingProgress = {
            queue,
            success,
            errors: failed,
        };

        this.setState({ uploadingProgress });
    }

    onSuccess(data: IProgressData) {
        const { failed, queue, success } = data;

        const uploadingProgress = {
            queue,
            success,
            errors: failed,
        };

        this.setState({ uploadingProgress, isBatchUploading: false });
    }

    uploadFileData() {
        const fileData = this.props.fileData ?? [];

        this.setState({ isBatchUploading: true }, () => {
            const cars = fileData?.reduce((res: Dict<string>[], curr) => {
                res.push(...curr.data);

                return res;
            }, []);

            const queue = cars.map(car => this.request.exec.bind(this.request,
                REQUESTS.BATCH_UPLOAD_CARS, { body: { cars: [car] } }));

            const qs = new QueryScheduler({
                queue,
                limit: COUNT_FIRST,
                onProgress: this.onProgress.bind(this),
                onSuccess: this.onSuccess.bind(this),
            });

            qs.run();
        });
    }

    getFileData() {
        const fileData = this.props.fileData ?? [];

        return fileData?.reduce((res: Dict<string>[], curr) => {
            res.push(...curr.data);

            return res;
        }, []);
    }

    render() {
        const fileData = this.props.fileData ?? [];
        const { onClose } = this.props;
        const { uploadingProgress, isBatchUploading } = this.state;

        const isFileDataEmpty = !fileData.length;

        return <Window onClose={onClose}
                       title={'Загрузить данные по VIN?'}
                       closeWithConfirm={isBatchUploading}>
            <div className={style.vin_upload_modal}>
                {this.buildFileConfirmQuestion()}
                <div className={style.progress_bar_container}>
                    <ProgressBar allLength={uploadingProgress?.queue?.length}
                                 successLength={uploadingProgress?.success?.length}
                                 errors={uploadingProgress.errors}/>
                </div>
                <Button disabled={isBatchUploading || isFileDataEmpty}
                        onClick={this.uploadFileData.bind(this)}>Загрузить</Button>
                {uploadingProgress?.errors?.length
                    ? <RetryProgressButtons uploadingProgress={uploadingProgress}
                                            fileData={this.getFileData()}
                                            isWorking={isBatchUploading}
                                            onProgress={this.onProgress.bind(this)}
                                            onSuccess={this.onSuccess.bind(this)}/>
                    : null}
            </div>
        </Window>;
    }
}
