import React, { useEffect, useState } from 'react';
import { RouteComponentProps } from 'react-router';
import { withRouter } from 'react-router-dom';

import { Button } from '../../../ui/Button';
import { Collapse2 } from '../../../ui/Collapse2';
import { JsonModal } from '../../../ui/FullModal/JsonModal';
import { SimpleError } from '../../SimpleError';
import { VOYAGE_PATH } from '../constants';
import style from '../style.css';
import { IVoyageItem, IVoyageResponse } from '../types';
import { CarVariantsModal } from './CarVariantsModal';

interface IVoyageInfoProps extends RouteComponentProps {
    data: IVoyageResponse | null;
    error: Error | null;
}

const VoyageInfo = React.memo((props: IVoyageInfoProps) => {
    const { data, error } = props;

    const [item, setItem] = useState<IVoyageItem | null>(null);
    const [isJsonOpen, openJson] = useState<boolean>(false);
    const [isCarsModalOpen, openCarsModal] = useState<boolean>(false);

    useEffect(() => {
        const voyageId = props.location.pathname.substr(VOYAGE_PATH.length);
        const item = data?.items?.find(el => el.id === voyageId) ?? null;

        setItem(item);
    });

    return <div className={style.voyage_info}>
        {item
            ? <>
                <h3>{item.name}</h3>

                <div className={style.tags}>
                    {item.tags?.map((tag, ind) => {
                        return <div key={tag.name + ind} className={style.tag}>
                            <img src={tag.icon} alt={tag} className={style.tag_img}/>
                            {tag.name}
                        </div>;
                    })}
                </div>

                <img alt={item.name} src={item.photo} className={style.info_img}/>
                <div className={style.info_dec}>{item.description_full}</div>

                <div className={style.features}>
                    {item.features?.map(feat => {
                        return <div className={style.feat_item} key={feat.name}>
                            <div className={style.feat_name}>{feat?.name}</div>
                            <div className={style.feat_value}>{feat?.value}</div>
                        </div>;
                    })}
                </div>

                <Button onClick={openCarsModal.bind(null, true)} className={style.info_button}>
                    На чём поедем?
                </Button>

                <Button onClick={openJson.bind(null, true)}
                        basic={true}
                        className={style.info_button}>
                    Вся информация о точке
                </Button>
            </>
            : error
                ? <Collapse2 title={'Упс, произошла ошибка :('} children={<SimpleError error={error}/>}/>
                : <h3>Куда поедем?</h3>
        }

        {isJsonOpen
            ? <JsonModal obj={item} onClose={openJson.bind(null)}/>
            : null
        }

        {isCarsModalOpen
            ? <CarVariantsModal data={data?.map_filters}
                                allowedVariants={item?.filters_refs ?? []}
                                onClose={openCarsModal.bind(null)}/>
            : null
        }
    </div>;
});

export default withRouter<any,any>((VoyageInfo));
