import React from 'react';
import { RouteComponentProps } from 'react-router';
import { Route } from 'react-router-dom';

import { isValidJSONString } from '../../utils/isValidJSONString';
import { Request2 } from '../../utils/request';
import { initMap } from '../MainMap/utils';
import { drawCityControl } from '../Map2/controls/drawCityControl';
import VoyageInfo from './ChildComponents/VoyageInfo';
import { ActionType, VOYAGE_MAP_ID, VOYAGE_PATH, VoyageGVarsKey } from './constants';
import { REQUESTS, VOYAGE_REQUESTS } from './requests';
import style from './style.css';
import { IVoyageResponse } from './types';

type IVoyageProps = RouteComponentProps

interface IVoyageState {
    data: IVoyageResponse | null;
    error: Error | null;
    selectedPointId: string | null;
    invalidId: boolean;
}

declare let ymaps;

export class Voyage extends React.Component<IVoyageProps, IVoyageState> {
    state: IVoyageState = {
        data: null,
        error: null,
        selectedPointId: null,
        invalidId: false,
    };

    map: any;
    request = new Request2({ requestConfigs: VOYAGE_REQUESTS });

    componentDidMount() {
        initMap(VOYAGE_MAP_ID, (map) => {
            this.map = map;
            this.getData();

            const hrefPointId = getVoyageIdFromHref(this.props.location.pathname);
            if (hrefPointId) {
                this.setState({ selectedPointId: hrefPointId, invalidId: false });
            }
        });
    }

    componentDidUpdate(prevProps: Readonly<IVoyageProps>, prevState: Readonly<IVoyageState>, snapshot?: any) {
        const hrefPointId = getVoyageIdFromHref(this.props.location.pathname);

        if (hrefPointId !== this.state.selectedPointId) {
            this.setState({ selectedPointId: hrefPointId, invalidId: false }, () => {
                this.setMapCenterToPoint();
            });
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
        this.map && this.map.destroy();
    }

    getData() {
        this.request.exec(REQUESTS.GET_ACTION)
            .then(response => {
                const action = response?.report?.find(el => {
                    return el.action_type === ActionType;
                });
                const setting = action?.action_meta?.settings?.find(el => el.key === VoyageGVarsKey) || {};
                const data = isValidJSONString(setting.value) ? JSON.parse(setting.value) : {};

                this.setState({
                    data,
                }, () => {
                    this.drawMapControls();
                    this.drawPoints();

                    if (this.state.selectedPointId) {
                        this.setMapCenterToPoint();
                    }
                });
            })
            .catch(error => {
                this.setState({ error });
            });
    }

    drawMapControls() {
        drawCityControl(this.map, true);
        const zoomControl = new ymaps.control.ZoomControl();
        this.map.controls.add(zoomControl);
    }

    drawPoints() {
        const points = this.state.data?.items ?? [];

        points.forEach(point => {
            const { lat, lon } = point.position;

            const placemark = new ymaps.Placemark(
                [lon, lat],
                { iconCaption: point.name, id: point.id },
                { preset: 'islands#blueCircleDotIconWithCaption' },
            );

            placemark.events.add('click', this.onPlacemarkClick.bind(this, point.id));

            this.map?.geoObjects.add(placemark);
        });
    }

    setMapCenterToPoint() {
        const point = this.state.data?.items?.find(el => el.id === this.state.selectedPointId);
        const coord = point?.position;

        point ? this.map.geoObjects.each((point) => this.markPoint(point)) : null;
        coord ? this.map.panTo([coord.lon, coord.lat]) : null;

        if (!point && this.state.selectedPointId) {
            this.setState({ invalidId: true });
        }
    }

    markPoint(point) {
        if (point.properties?._data?.id === this.state.selectedPointId) {
            point.options.set('preset', 'islands#redCircleDotIconWithCaption');
        } else {
            point.options.set('preset', 'islands#blueCircleDotIconWithCaption');
        }
    }

    onPlacemarkClick(selectedPointId) {
        location.hash = `${VOYAGE_PATH}${selectedPointId}`;
    }

    render() {
        return <div className={style.wrapper}>
            <div id={VOYAGE_MAP_ID} className={style.map_container}/>

            {this.state.invalidId
                ? <h3>Нет путешествия с таким id: {this.state.selectedPointId}</h3>
                : <Route path={`${VOYAGE_PATH}:voyage_id?`}
                         render={() => <VoyageInfo error={this.state.error} data={this.state.data}/>}/>
            }
        </div>;
    }
}

export function getVoyageIdFromHref(pathname: string): string {
    return pathname.substr(VOYAGE_PATH.length);
}
