import * as React from 'react';

import { ClassificationType } from '../../../types';
import {
    endAction,
    isWebphoneNotRegistered,
    IWebphoneReducerState,
    WEBPHONE_ACTION_KEYS,
    WebphoneReducerType,
} from '../../reducers/webphoneReducer';
import { ErrorsModal } from '../../ui/ErrorBar';
import { CSSettingsItems, CustomSettings } from '../../utils/customSettings';
import { isObjectEqual } from '../../utils/isObjectEqual';
import ClassificationDialog from '../Chats/ClassificationDialog';
import { WebphoneMain } from './helpers/webphoneMain';
import { CALL_STATUS, CallSourceOptions, IGetCallIdProps, WEBPHONE_STATUS } from './helpers/webphoneTypes';
import { EnableMicrophoneModal } from './ui/EnableMicrophoneModal';
import WebphonePanel from './ui/WebphonePanel';
import { WelcomePopup } from './ui/WelcomePopup/component';
import { WelcomePopupWebphone } from './ui/WelcomePopupWebphone';

interface IWebphoneProps {
    dispatch: any;
    webphoneStore: IWebphoneReducerState;
    yandexLogin: string;
    callSourceKeys: {"phone_number": number}[];
    additionalWebphoneNumber: {"phone_number": number}[];
}

interface IWebphoneState {
    errors: Error[];
    showWelcomePopup: boolean;
    shouldNeverShowPopup: boolean;
    showClassificationModal: boolean;
    callId: string;
    callTagId: string;
    classifiedCall: boolean;
}

const webphoneOutgoingKeyword = 'webphone_outgoing';

export class Webphone extends React.Component<IWebphoneProps, IWebphoneState> {
    webphoneMain;
    cs;
    state: IWebphoneState;

    constructor(props) {
        super(props);
        const { dispatch, yandexLogin, webphoneStore } = props;
        this.webphoneMain = new WebphoneMain({
            dispatch,
            yandexLogin,
            webphoneStore,
            setError: this.setError.bind(this),
        });
        this.cs = new CustomSettings();
        this.state = {
            errors: [],
            showWelcomePopup: true,
            shouldNeverShowPopup: this.cs.get(CSSettingsItems.shouldNeverShowWebphoneWelcomePopup),
            showClassificationModal: false,
            callId: '',
            callTagId: '',
            classifiedCall: false,
        };
    }

    getMobIdHeaderForCallSource(callInfo: IGetCallIdProps): string | undefined {
        if (callInfo.callSource === CallSourceOptions.CUSTOM) {
            const customPhoneNumber = this.props
                .callSourceKeys[Math.floor(Math.random() * this.props.callSourceKeys.length)]
                .phone_number;

            return `X-MOBID: ${customPhoneNumber}`;
        }

        const additionalPhoneNumber = this.props.additionalWebphoneNumber?.[0]?.phone_number;

        if (additionalPhoneNumber) {
            return `X-MOBID: ${additionalPhoneNumber}`;
        }

        return undefined;
    }

    async componentDidUpdate(prevProps, prevState) {
        const { dispatch, webphoneStore } = this.props;
        if (!isObjectEqual(prevProps?.webphoneStore, this.props?.webphoneStore)) {
            if (this.props.webphoneStore.action === WEBPHONE_ACTION_KEYS[WebphoneReducerType.REGISTER]) {
                this.webphoneMain.register.call(this.webphoneMain);
                dispatch(endAction());
            } else if (webphoneStore.call !== null) {
                const { callId, callTagId } = await this.webphoneMain.getCallId
                    .call(this.webphoneMain, webphoneStore.call.callInfo);

                this.setState({
                    callId, callTagId,
                }, () => {
                    if (webphoneStore.call?.callInfo.chatTagId) {
                        const mobIdHeader = this.getMobIdHeaderForCallSource(webphoneStore.call.callInfo);
                        this.webphoneMain.call
                            .call(this.webphoneMain, webphoneStore.call, callId, callTagId, mobIdHeader);
                        dispatch(endAction());
                    } else {
                        this.setState({
                            showClassificationModal: true,
                            classifiedCall: false,
                        });
                    }
                });

            } else if (webphoneStore.action === WEBPHONE_ACTION_KEYS[WebphoneReducerType.END_CALL]) {
                this.webphoneMain.endCall.call(this.webphoneMain);
                dispatch(endAction());
            } else if (webphoneStore.hold !== null) {
                this.webphoneMain.holdCall.call(this.webphoneMain, webphoneStore.hold);
                dispatch(endAction());
            } else if (webphoneStore.mute !== null) {
                this.webphoneMain.muteCall.call(this.webphoneMain, webphoneStore.mute);
                dispatch(endAction());
            }
        } else if (this.state.showClassificationModal !== prevState.showClassificationModal
            && !this.state.showClassificationModal) {
            const { callId, callTagId, classifiedCall } = this.state;

            if (webphoneStore.call && callId && callTagId && classifiedCall) {
                const mobIdHeader = this.getMobIdHeaderForCallSource(webphoneStore.call.callInfo);
                this.webphoneMain.call.call(this.webphoneMain, webphoneStore.call, callId, callTagId, mobIdHeader);
            } else {
                this.setError(new Error('Отсутствует callId или callTagId или звонок не классифицирован'));
            }

            dispatch(endAction());
        }
    }

    /* confirm reload and close tab */
    beforeUnloadListener(e) {
        const callStatus = this.props.webphoneStore?.callStatus;

        if (callStatus !== null && callStatus !== CALL_STATUS.ended && callStatus !== CALL_STATUS.failed) {
            e.preventDefault();
            e.returnValue = '';
        }
    }

    closeWebphone() {
        window.removeEventListener('beforeunload', this.beforeUnloadListener);
        window.removeEventListener('unload', this.closeWebphone);
        const status = this.props.webphoneStore?.status;

        if (status !== WEBPHONE_STATUS.notRegistered) {
            this.webphoneMain.closeWebphone.call(this.webphoneMain);
        }
    }

    componentDidMount() {
        window.addEventListener('beforeunload', this.beforeUnloadListener.bind(this));
        window.addEventListener('unload', this.closeWebphone.bind(this));
    }

    componentWillUnmount() {
        if (!this.props.dispatch(isWebphoneNotRegistered)) {
            this.closeWebphone();
        }
    }

    setError(error: Error) {
        this.setState((prev) => ({
            errors: [...prev.errors, error],
        }));
    }

    clearErrors() {
        this.setState({
            errors: [],
        });
    }

    toggleWelcomePopup(shouldBeOpen: boolean) {
        this.setState({
            showWelcomePopup: shouldBeOpen,
        });
    }

    setNeverShowPopup(value) {
        this.cs.set(CSSettingsItems.shouldNeverShowWebphoneWelcomePopup, value);
    }

    render() {
        return <>
            {this.state.showClassificationModal && this.state.callTagId && (
                <ClassificationDialog tag_id={this.state.callTagId}
                                      onClose={() => {
                                          this.setState({
                                              showClassificationModal: false,
                                          });
                                      }}
                                      onClassified={() => {
                                          this.setState({
                                              showClassificationModal: false,
                                              classifiedCall: true,
                                          });
                                      }}
                                      type={ClassificationType.call_outgoing}
                                      user_id={this.props.webphoneStore?.callMetaInfo?.id ?? null}
                                      chat_id={null}
                                      keyword={webphoneOutgoingKeyword}
                                      showCoreComponentsOnly
                                      suggest={false}/>
            )}
            {this.state.showWelcomePopup && !this.state.shouldNeverShowPopup
                ? <WelcomePopup onClose={this.toggleWelcomePopup.bind(this, false)}
                                neverShowOption={this.setNeverShowPopup.bind(this)}
                                title={'Бета-версия телефонии'}>
                    <WelcomePopupWebphone/>
                </WelcomePopup>
                : null
            }
            {this.state.errors?.length
                ? this.state.errors.every((error) => error.message === 'Permission denied')
                    ? <EnableMicrophoneModal onClose={this.clearErrors.bind(this)}/>
                    : <ErrorsModal onClose={this.clearErrors.bind(this)}
                                   errors={this.state.errors}
                                   title={'Ошибка телефонии'}/>
                : null
            }
            <WebphonePanel/>
        </>;
    }
}
